subroutine noema_tsys(line,error)
  use gbl_message
  use gkernel_interfaces
  use astro_interfaces, except_this=>noema_tsys
  !---------------------------------------------------------------------
  ! @ private
  ! Support routine for command
  !   NOEMA\TSYS TableName
  ! Compute the Tsys + Opacity table
  !---------------------------------------------------------------------
  character(len=*), intent(in)    :: line
  logical,          intent(inout) :: error
  ! Local
  character(len=*), parameter :: rname='TSYS'
  character(len=filename_length) :: tsysfile
  integer(kind=4) :: nc
  !
  call sic_ch(line,0,1,tsysfile,nc,.true.,error)
  if (error)  return
  !
  if (gag_inquire(tsysfile,nc).eq.0) then
    call astro_message(seve%e,rname,  &
      'File '//tsysfile(1:nc)//' already exists. Remove it first.')
    error = .true.
    if (error)  return
  endif
  !
  call noema_tsys_table(tsysfile,error)
  if (error)  return
  !
end subroutine noema_tsys
!
subroutine noema_tsys_table(file,error)
  use gbl_message
  use gkernel_interfaces
  use gkernel_types
  use atm_interfaces_public
  use astro_interfaces, except_this=>noema_tsys_table
  use astro_types
  !---------------------------------------------------------------------
  ! @ private
  ! Compute the Tsys + Opacity table
  !---------------------------------------------------------------------
  character(len=*), intent(in)    :: file
  logical,          intent(inout) :: error
  ! Local
  character(len=*), parameter :: rname='TSYS'
  character(len=message_length) :: mess
  integer(kind=4) :: nf,na,nw,if,ia,iw,it,ier,ib
  real(kind=4) :: p1,temis,temii,tatm,tauox,tauw,dummy,taut
  real(kind=4), allocatable :: f(:),a(:),w(:),vtrec(:),vfeff(:)
  real(kind=4), allocatable :: tsys(:,:,:,:)
  type(time_t) :: time
  type(receiver_t) :: rec
  !
  ! Parameters
  real(kind=4), parameter :: h0=2.560    ! [km] NOEMA altitude
  real(kind=4), parameter :: p0=1013.0   ! [HPa] At sea level
  real(kind=4), parameter :: gim=0.05    ! [   ] Image gain ratio
  real(kind=4), parameter :: intfreq=0.  ! [GHz] Intermediate frequency
  ! IF (used to compute the image frequency) is set to 0 because:
  ! 1) typical values show a Tsys difference near 0% at the center of
  !    the bands (where Tsys is flat), and less than 2% at the band
  !    sides (where Tsys varies faster in one sideband than in the
  !    other). => NB: probably not true anymore now that gim is 0.05
  ! 2) it is quite difficult/impossible to produce a generic Tsys table
  !    which would know in advance what is the image frequency distance
  !    and side (lower or upper). In other words we can not know what is
  !    the image frequency associated to a given signal frequency.
  !
  ! Frequencies
  integer(kind=4), parameter :: nb=3     ! Number of bands
  real(kind=4),    parameter :: f_step=1.  ! [GHz]
  real(kind=4)               :: f_min(nb)  ! [GHz] Read from 'rec_define_noema'
  real(kind=4)               :: f_max(nb)  ! [GHz] Read from 'rec_define_noema'
  ! Airmass
  real(kind=4),    parameter :: a_step=0.5  ! [Neper]
  real(kind=4),    parameter :: a_min=1.    ! [Neper] Elevation 90 deg
  real(kind=4),    parameter :: a_max=6.    ! [Neper] Elevation ~9.5 deg
  ! PWV
  real(kind=4),    parameter :: w_step=0.5  ! [mm]
  real(kind=4),    parameter :: w_min=2.    ! [mm]
  real(kind=4),    parameter :: w_max=7.    ! [mm]
  ! Temperatures
  integer(kind=4), parameter :: nt=2     ! Number of atmosphere temperatures
  real(kind=4),    parameter :: t(nt) = (/ 273.,283. /)  ! [K]
  ! Trec, Feff
  real(kind=4),    parameter :: trec(nb) = (/ 35.,45.,55. /)  ! [K] Trec value
  real(kind=4),    parameter :: feff(nb) = (/ 0.95, 0.95, 0.90 /)  ! Forward efficiency
  !
  p1 = p0*2.0**(-h0/5.5)  ! [HPa] Pressure at altitude h0
  !
  ! Get receiver parameters
  call rec_define_noema(rec,error)
  if (error)  return
  !
  if (rec%desc%n_rbands.ne.nb) then
    ! If error, adapt Trec and Feff arrays
    call astro_message(seve%e,rname,'Number of bands not supported')
    error = .true.
    return
  endif
  !
  ! Frequency ranges
  do ib=1,rec%desc%n_rbands
    ! Compute our frequency range. Be conservative (floor/ceiling) + align
    ! to the nearest GHz (purely cosmetic)
    f_min(ib) = floor(rec%desc%rflim(1,ib)/1000.)
    f_max(ib) = ceiling(rec%desc%rflim(2,ib)/1000.)
  enddo
  ! Number of frequencies: the whole range is sampled with no gap between
  ! bands. Beware this feature is used by PMS.
  nf = (f_max(rec%desc%n_rbands)-f_min(1))/f_step+1  ! Assume that bands are ordered increasingly
  ! Number of airmasses: easy
  na = (a_max-a_min)/a_step+1
  ! Number of pwv: easy
  nw = (w_max-w_min)/w_step+1
  !
  ! Define the vectors
  allocate(f(nf),a(na),w(nw),vtrec(nf),vfeff(nf),stat=ier)
  if (failed_allocate(rname,'dimension buffers',ier,error))  return
  ! Frequencies: regular sampling. Beware this feature is used by PMS.
  do if=1,nf
    f(if) = f_min(1)+(if-1)*f_step
  enddo
  ! Airmasses
  do ia=1,na
    a(ia) = a_min+(ia-1)*a_step
  enddo
  ! PWV
  do iw=1,nw
    w(iw) = w_min+(iw-1)*w_step
  enddo
  ! Vectorized Trec and Feff:
  vtrec(:) = 1e4   ! Blank value: very bad Trec (but valid in computations)
  vfeff(:) = 1e-4  ! Blank value: very bad Feff (but valid in computations)
  do ib=1,rec%desc%n_rbands
    where (f.ge.f_min(ib) .and. f.le.f_max(ib))
      vtrec = trec(ib)
      vfeff = feff(ib)
    end where
  enddo
  !
  call astro_message(seve%i,rname,'Computing Tsys table with')
  write (mess,10)  nf,' frequencies  from ',f(1),' to ',f(nf),' x '
  call astro_message(seve%i,rname,mess)
  write (mess,10)  na,' airmasses    from ',a(1),' to ',a(na),' x '
  call astro_message(seve%i,rname,mess)
  write (mess,10)  nw,' pwv          from ',w(1),' to ',w(nw),' x '
  call astro_message(seve%i,rname,mess)
  write (mess,10)  nt,' temperatures from ',t(1),' to ',t(nt)
  call astro_message(seve%i,rname,mess)
10 format(I5,A,F6.2,A,F6.2,A)
  !
  ! Now compute Tsys
  allocate(tsys(nf,na,nw,nt),stat=ier)
  if (failed_allocate(rname,'tsys buffers',ier,error))  return
  !
  call gtime_init(time,nt*nw*na*nf,error)
  if (error)  return
  !
  do it=1,nt
    call atm_atmosp(t(it),p1,h0)
    do iw=1,nw
      do ia=1,na
        do if=1,nf
          ! Signal
          call atm_transm(w(iw),a(ia),f(if),temis,tatm,tauox,tauw,taut,ier)
          ! Image
          if (intfreq.eq.0.) then
            temii = temis
          else
            call atm_transm(w(iw),a(ia),f(if)+2.*intfreq,temii,tatm,tauox,tauw,dummy,ier)
          endif
          ! Store Tsys at current airmass
          tsys(if,ia,iw,it) = ( gim * (vfeff(if)*temii+(1.0-vfeff(if))*t(it) + vtrec(if)) +  &
                                      (vfeff(if)*temis+(1.0-vfeff(if))*t(it) + vtrec(if))    &
                              ) / vfeff(if) * exp(taut*a(ia))
          !
          call gtime_current(time)
          if (sic_ctrlc()) then
            call astro_message(seve%e,rname,'Aborted')
            error = .true.
            return
          endif
          !
        enddo
      enddo
    enddo
  enddo
  !
  call write_bintable(file,error)
  if (error)  return
  !
contains
  subroutine write_bintable(file,error)
    use gildas_def
    use gbl_format
    !-------------------------------------------------------------------
    !
    !-------------------------------------------------------------------
    character(len=*), intent(in)    :: file
    logical,          intent(inout) :: error
    ! Local
    integer(kind=4) :: ier,nfile,lun,currec,recpos
    character(len=message_length) :: mess
    character(len=4) :: tab_code
    integer(kind=4), parameter :: reclen=128  ! Words
    integer(kind=4) :: buffer(reclen)
    !
    ier = sic_getlun(lun)
    nfile = len_trim(file)
    open(unit=lun,file=file(1:nfile),status='NEW',access='DIRECT',   &
        form='UNFORMATTED',iostat=ier,recl=reclen*facunf)
    if (ier.ne.0) then
      call astro_message(seve%e,rname,'Filename: '//file)
      call putios('E-ATM, Open error: ',ier)
      error = .true.
      goto 99
    endif
    !
    call gdf_getcod(tab_code)
    call chtoby(tab_code,buffer(1),4)
    call r4tor4(p1,buffer(2),1)
    call r4tor4(gim,buffer(3),1)
    call r4tor4(intfreq,buffer(4),1)
    call i4toi4(nf,buffer(5),1)
    call i4toi4(na,buffer(6),1)
    call i4toi4(nw,buffer(7),1)
    call i4toi4(nt,buffer(8),1)
    currec = 1
    recpos = 9
    !
    call write_my_array(lun,f,nf,buffer,currec,recpos,error)
    if (error)  goto 98
    call write_my_array(lun,a,na,buffer,currec,recpos,error)
    if (error)  goto 98
    call write_my_array(lun,w,nw,buffer,currec,recpos,error)
    if (error)  goto 98
    call write_my_array(lun,t,nt,buffer,currec,recpos,error)
    if (error)  goto 98
    call write_my_array(lun,vtrec,nf,buffer,currec,recpos,error)
    if (error)  goto 98
    call write_my_array(lun,vfeff,nf,buffer,currec,recpos,error)
    if (error)  goto 98
    !
    call write_my_array(lun,tsys,nf*na*nw*nt,buffer,currec,recpos,error)
    if (error)  goto 98
    !
    if (recpos.ne.1) then
      ! Flush last record
      write(lun,rec=currec,iostat=ier)  buffer
      if (ier.ne.0) then
        write(mess,'(A,I0)') 'Error writing record ',currec
        call astro_message(seve%e,rname,mess)
        error = .true.
        goto 98
      endif
    endif
    !
    write(mess,'(I0,A)') currec,' records written'
    call astro_message(seve%i,rname,mess)
    !
98  close(lun)
99  call sic_frelun(lun)
  end subroutine write_bintable
  !
  subroutine write_my_array(lun,array,n,buffer,currec,recpos,error)
    integer(kind=4), intent(in)    :: lun
    integer(kind=4), intent(in)    :: n
    real(kind=4),    intent(in)    :: array(n)
    integer(kind=4), intent(inout) :: buffer(:)
    integer(kind=4), intent(inout) :: currec
    integer(kind=4), intent(inout) :: recpos
    logical,         intent(inout) :: error
    ! Local
    integer(kind=4), parameter :: reclen=128
    integer(kind=4) :: nwritetot,nwritecur
    !
    nwritetot = 0
    do while (nwritetot.lt.n)
      !
      nwritecur = min(n-nwritetot,reclen-recpos+1)
      call r4tor4(array(nwritetot+1),buffer(recpos),nwritecur)
      nwritetot = nwritetot+nwritecur
      recpos = recpos+nwritecur
      !
      if (recpos.gt.reclen) then
        ! print *,"Writing record ",currec
        write(lun,rec=currec,iostat=ier)  buffer
        if (ier.ne.0) then
          write(mess,'(A,I0)') 'Error writing record ',currec
          call astro_message(seve%e,rname,mess)
          error = .true.
          return
        endif
        currec = currec+1
        recpos = 1
      endif
      !
    enddo
  end subroutine write_my_array
  !
end subroutine noema_tsys_table
