subroutine rec_define_noema(rec,mode,error)
  use gbl_message
  use astro_interfaces, except_this=>rec_define_noema
  use astro_types
  !-----------------------------------------------------------------------
  ! @ private
  ! fill in the receiver type according to input name
  !-----------------------------------------------------------------------
  type(receiver_t), intent(inout) :: rec ! receiver parameters
  character(len=*), intent(in) :: mode ! OFFLINE or ONLINE
  logical, intent(inout)        :: error
  ! local
  integer(kind=4)       :: i
  !
  rec%desc%name = 'NOEMA'
  if (mode.eq.'OFFLINE') then
    rec%desc%n_rbands = 3
  else if (mode.eq.'ONLINE') then
    rec%desc%n_rbands = 4
  endif
  rec%desc%bandname(1) = 'Band_1'
  rec%desc%bandname(2) = 'Band_2'
  rec%desc%bandname(3) = 'Band_3'
  rec%desc%bandname(4) = 'Band_4'
  rec%desc%rejection(1) = 30
  rec%desc%rejection(2) = 30
  rec%desc%rejection(3) = 30
  rec%desc%rejection(4) = 30
  rec%desc%n_sbands = 2
  rec%desc%n_bbands = 2
  rec%desc%n_polar = 2
  rec%desc%n_backends = 1
  rec%desc%iflim(1) = 3872d0
  rec%desc%iflim(2) = 11616d0
  rec%desc%ifband = rec%desc%iflim(2)-rec%desc%iflim(1)
  !basebands
  rec%desc%flo2 = 7744d0
  ! LO2 separation zone: (from commissioning output)
  ! Mark only the strong parasite
  !Attenuation 1.5chunk in INNER and OUTER defined in IF1 - not yet plotted
  rec%desc%if1conf(1) = rec%desc%flo2-10d0
  rec%desc%if1conf(2) = rec%desc%flo2+10d0
  rec%desc%bbname(1) = 'OUTER'
  rec%desc%bblim(1,1) = 7744d0
  rec%desc%bblim(2,1) = 11616d0
  rec%desc%bbref(1) = 10000d0
  rec%desc%bbname(2) = 'INNER'
  rec%desc%bblim(1,2) = 3872d0
  rec%desc%bblim(2,2) = 7744d0
  rec%desc%bbref(2) = 6000d0
  ! Polars
  rec%desc%polname(1) = 'HORIZONTAL'
  rec%desc%polname(2) = 'VERTICAL'
  !
  ! Limits as stated in the tuning tables  (Jan2017)
  rec%desc%lohard(1,1) = 82.0d3
  rec%desc%lohard(2,1) = 110d3
  rec%desc%lohard(1,2) = 136d3
  rec%desc%lohard(2,2) = 172d3
  rec%desc%lohard(1,3) = 207.744d3
  rec%desc%lohard(2,3) = 268d3
  rec%desc%lohard(1,4) = 280d3
  rec%desc%lohard(2,4) = 370d3
  !
  ! Limits from ACC = Specs from SVN noema (may2018)
  rec%desc%locall(1,1) = 82.0d3 ! to match the specs 20dec2017 (was 82.5)
  rec%desc%locall(2,1) = 108.256d3
  rec%desc%locall(1,2) = 138.616d3
  rec%desc%locall(2,2) = 171.256d3
  rec%desc%locall(1,3) = 207.744d3
  rec%desc%locall(2,3) = 264.384d3
  rec%desc%locall(1,4) = 280d3
  rec%desc%locall(2,4) = 370d3
  !
  rec%desc%lotol = 100d0
  !
  ! Wider limits to let some margin to explore the edge of the bands
  do i=1,rec%desc%n_rbands
      rec%desc%lolim(1,i) = min(rec%desc%locall(1,i)-rec%desc%lotol,rec%desc%lohard(1,i))
      rec%desc%lolim(2,i) = max(rec%desc%locall(2,i)+rec%desc%lotol,rec%desc%lohard(2,i))
  enddo
  !
  ! RF limits from LOLIM +- IFmax
  do i=1,rec%desc%n_rbands
      rec%desc%rflim(1,i) = rec%desc%lolim(1,i)-rec%desc%iflim(2)
      rec%desc%rflim(2,i) = rec%desc%lolim(2,i)+rec%desc%iflim(2)
  enddo
  ! RF limits from LOCALL +- IFmax
  do i=1,rec%desc%n_rbands
      rec%desc%rfcall(1,i) = rec%desc%locall(1,i)-rec%desc%iflim(2)
      rec%desc%rfcall(2,i) = rec%desc%locall(2,i)+rec%desc%iflim(2)
  enddo
  !
  rec%desc%tuninggrid = .true.
  rec%desc%gridbin = 500d0 ! MHz
  rec%desc%gridtol = 1d0     ! MHz
  !
end subroutine rec_define_noema
!
subroutine rec_define_noema_ifproc(noemaif,error)
  use gbl_message
  use astro_interfaces, except_this=>rec_define_noema_ifproc
  use astro_types
  !-----------------------------------------------------------------------
  ! @ private
  ! if proc characteristics
  !-----------------------------------------------------------------------
  type(correlator_input_t), intent(inout) :: noemaif !
  logical, intent(inout)        :: error
  ! local
  integer(kind=4) :: i
  ! Ifproc
  noemaif%n_ifcables = 8
  noemaif%m_usercables = 0
  noemaif%mode = ''
  noemaif%defined = .false.
  do i=1,m_ifcables
    noemaif%ifc(i)%label = ''
    noemaif%ifc(i)%iband = 0
    noemaif%ifc(i)%sb_code = 0
    noemaif%ifc(i)%bb_code = 0
    noemaif%ifc(i)%pol_code = 0
  enddo
  !
end subroutine rec_define_noema_ifproc
!
subroutine noema_reset_setup(noema,error)
  use astro_interfaces, except_this=>noema_reset_setup
  use astro_types
  !-----------------------------------------------------------------------
  ! @ private
  ! reset noema specific parameters
  !-----------------------------------------------------------------------
  type(noema_t), intent(inout) :: noema
  logical, intent(inout)        :: error
  ! Local
  integer(kind=4)       :: i
  !
  !
  !Command parameters
  noema%comm%n_tunings = 0
  do i=1,m_tunings
    noema%comm%frest(i) = 0
    noema%comm%fcent(i) = 0
  enddo
  noema%comm%zoommode = ''
  noema%comm%fz1 = 0
  noema%comm%fz2 = 0
  noema%comm%fixed_scale = .false. 
  !
  !IF PROCESSOR
  noema%ifproc%mode = ''
  noema%ifproc%defined = .false.
  do i=1,m_ifcables
    noema%ifproc%ifc(i)%label = ''
    noema%ifproc%ifc(i)%iband = 0
    noema%ifproc%ifc(i)%sb_code = 0
    noema%ifproc%ifc(i)%bb_code = 0
    noema%ifproc%ifc(i)%pol_code = 0
  enddo
  !
  !Baseband selection
  noema%selunit%n_ifsel=-1
  noema%selunit%polmode='NULL'
  do i=1,m_backunits
    noema%selunit%usel(i)=-1
  enddo
  !
end subroutine noema_reset_setup
!
subroutine rec_noema(line,error)
  use gbl_message
  use gkernel_interfaces
  use astro_interfaces, except_this=>rec_noema
  use astro_types
  use ast_astro
  use my_receiver_globals
  use plot_molecules_globals
  use frequency_axis_globals
  !-----------------------------------------------------------------------
  ! @ private
  ! Command to define and visualize NOEMA receiver coverage
  !-----------------------------------------------------------------------
  character(len=*), intent(inout) :: line        ! command line
  logical, intent(inout) :: error
  ! local
  character(len=*), parameter :: rname='TUNING'
  integer(kind=4),parameter :: optzoom=1 ! option /ZOOM
  integer(kind=4),parameter :: optpagewidth=2 ! option /PAGEWIDTH
  integer(kind=4),parameter :: optfixfreq=3 ! option /FIXED_FREQ
  integer(kind=4),parameter :: optinfo=4 ! option /INFO
  integer(kind=4) :: lpar,izoom,nzmode
  integer(kind=4), parameter :: m_zmodes=2
  real(kind=8) :: fcomm
  character(len=20) ::  zmodes(m_zmodes),zmode
  logical :: noarg,doinfo
  data zmodes/'ID','BOTH'/
  !
  error = .false.
  !
  if (obsname.ne.'NOEMA') then
    call astro_message(seve%e,rname,'Inconsistency between Receiver Name and Observatory')
    error = .true.
    return
  endif
  !
  !Define receiver parameters
  call astro_def_receiver(rname,'NOEMA',rec,error)
  if (error) return
  !Define ifproc parameters
  call rec_define_noema_ifproc(noema%ifproc,error)
  if (error) return
  ! Put in memory the current ASTRO%SOURCE parameters
  rec%desc%redshift = .true. ! tuning takes into accound source redshift
  call astro_def_recsource(rname,rec,error)
  if (error) return
  !
  !Print Receiver info in the terminal
  doinfo=sic_present(optinfo,0)
  !
  !Decoding Line Command
  ! First tuning
  zmode='ID'
  izoom=0
  noema%comm%n_tunings = 0
  if (error) return
  if (sic_present(0,1)) then
    noarg = .false.
    call astro_message(seve%i,rname,'Resetting tuning')
    call noema_reset_setup(noema,error)
    noema%comm%n_tunings = 1
    noema%comm%name(1) = ''
    ! Get the user value
    call sic_r8(line,0,1,fcomm,.true.,error)
    if (error) return
    ! Convert input freq to rest
    call rec_inputtorest(rname,fcomm,freq_axis%main,rec%source,noema%comm%frest(1),error)
    if (error) return
    if (sic_present(0,2)) then ! parameters are all given
      call sic_ke (line,0,2,noema%comm%sideband(1),lpar,.true.,error)
      if (error) return
      call sic_r8(line,0,3,noema%comm%fcent(1),.true.,error)
      if (error) return
    else
      call noema_default_tuning(rname,rec,noema%comm%n_tunings,noema%comm,error)
      if (error) return
    endif
    ! Second tuning
    noema%comm%name(2) = ''
    noema%comm%frest(2) = 0
    noema%comm%fcent(2) = 0
    noema%comm%sideband(2) = ''
    if (sic_present(0,4)) then
      noema%comm%n_tunings = 2
      call sic_r8(line,0,4,fcomm,.true.,error)
      if (error) return
      ! convert input freq to rest
      call rec_inputtorest(rname,fcomm,freq_axis%main,rec%source,noema%comm%frest(2),error)
      if (error) return
      call sic_ke (line,0,5,noema%comm%sideband(2),lpar,.true.,error)
      if (error) return
      call sic_r8(line,0,6,noema%comm%fcent(2),.true.,error)
      if (error) return
    endif
  else ! case without argument
    noarg=.true.
    call astro_message(seve%i,rname,'Showing the coverage of NOEMA receiver bands')
  endif
  ! Options
  ! Zoom parameter
  if (sic_present(optzoom,0)) then
    if (sic_present(optzoom,1)) then
      noema%comm%zoommode = 'NULL'
      call sic_ke (line,optzoom,1,noema%comm%zoommode,lpar,.false.,error)
      if (error) return
      call sic_ambigs_sub(rname,trim(noema%comm%zoommode),zmode,nzmode,zmodes,m_zmodes,error)
      if (error) return
      if (nzmode.eq.0) then !means that keyword was not recognized
        if (sic_present(optzoom,2)) then
          zmode='FREQ'
          call sic_r8(line,optzoom,1,noema%comm%fz1,.true.,error)
          if (error) return
          call sic_r8(line,optzoom,2,noema%comm%fz2,.true.,error)
          if (error) return
        else
          if (noema%comm%n_tunings.eq.0) then
            call astro_message(seve%e,rname,'Try to zoom on a not tune band')
            error = .true.
            return
          endif
          zmode='SINGLE'
          call sic_i0(line,optzoom,1,izoom,.true.,error)
          if (error) return
        endif
      endif
    else
      if (noema%comm%n_tunings.eq.1) then
        izoom = 1
        zmode='SINGLE'
      else if (noema%comm%n_tunings.eq.2) then
        izoom = 0
        zmode='BOTH'
      else
        call astro_message(seve%e,rname,'/ZOOM option without argument is ambiguous in this case')
        error = .true.
        return
      endif
    endif
  endif
  ! /pagewidth option
  if (sic_present(optpagewidth,0)) then
    noema%comm%fixed_scale = .false.
  else
    noema%comm%fixed_scale = .true.
  endif
  !
  ! /fixed_freq option
  if (sic_present(optfixfreq,0)) then
    noema%comm%fixedfreq = .true.
  else
    noema%comm%fixedfreq = .false.
  endif
  !
  !Prepare for the plot and tune
  call noema_setup(rname,noema,rec,error)
  if (error) return
  !
  if (zmode.eq.'ID') then
    call rec_plot_mbands(rec,molecules,noema%comm%fixed_scale,cplot,freq_axis,error)
    if (error) return
  else if (zmode.eq.'BOTH') then
    if (rec%n_tunings.ne.2) then
      call astro_message(seve%e,rname,'Tried to zoom on a not tuned band')
      error = .true.
      return
    endif
    call rec_plot_nbands(rec,molecules,noema%comm%fixed_scale,cplot,freq_axis,error)
    if (error) return
  else if (zmode.eq.'SINGLE') then
    if (izoom.le.0.or.izoom.gt.noema%comm%n_tunings.or.noema%comm%n_tunings.eq.0) then
      call astro_message(seve%e,rname,'Tried to zoom on a not tuned band')
      error = .true.
      return
    endif
    call rec_plot_sidebands(rec,izoom,cplot,molecules,freq_axis,error)
    if (error) return
  else if (zmode.eq.'FREQ') then
    call rec_zoom(molecules,noema%comm%fz1,noema%comm%fz2,rec,cplot,freq_axis,error)
    if (error) return
  endif
  !
  if (noarg) then
    ! if no tuning was entered, reset the rec%n_tuning to 0
    ! like this ifproc/backend command will not work (need at least 1 tuning)
    rec%n_tunings = 0
  else
    ! define if ranges transported
    call noema_ifproc(rname,rec,noema,error)
    if (error) return
    ! Update the plot
    call noema_ifproc_plot(rname,noema,molecules,rec,cplot,freq_axis,error)
    if (error) return
    ! define polyfix and assign if "cables" to correlator units [only input, no mode yet]
    call noema_define_pfx(pfx,error)
    if (error) return
    call noema_reset_backend(pfx,spw%out,error)
    spw%comm%itype=-1
    if (error) return
    call noema_assign_units(rec,noema,pfx,error)
    if (error) return
  endif
  !
  if (doinfo) then
    call astro_receiver_info(rec,rname,error)
    if (error) return
  endif
  !
  ! Let the user with the limits in main axis frequency frame
  call rec_set_limits(rname,cplot,freq_axis%main,error)
  if (error) return
  !
end subroutine rec_noema
!
subroutine noema_default_tuning(rname,rec,it,ncomm,error)
  use gbl_message
  use astro_interfaces, except_this=>noema_default_tuning
  use astro_types
  !-----------------------------------------------------------------------
  ! @ private
  ! reset noema specific parameters
  !-----------------------------------------------------------------------
  character(len=*), intent(in) :: rname
  type(receiver_t), intent(in) :: rec   ! all receiver info
  integer(kind=4), intent(in)  :: it    ! index of the tuning
  type(noema_comm_t), intent(inout) :: ncomm ! command-related stuff
  logical, intent(inout)        :: error
  ! Local
  integer(kind=4)       :: i,iband,sbcode
  real(kind=8)          :: frf,fumin
  !
  ! Find Band
  frf = ncomm%frest(it)*rec%source%dopshift*mhzperghz
  iband=0
  do i = 1,rec%desc%n_rbands
    if (frf.lt.rec%desc%rflim(1,i).or.frf.gt.rec%desc%rflim(2,i)) cycle
    iband = i
  enddo
  if (iband.eq.0) then
    call astro_message(seve%e,rname,'Frequency out of NOEMA range')
    error=.true.
    return
  endif
  !
  !Find SideBand
  fumin=rec%desc%lolim(1,iband)+rec%desc%iflim(1)+rec%desc%ifband/4d0
  !Choose to switch from LSB to USB at the center of the band
  sbcode=usb_code
  if (frf.lt.fumin) then
    sbcode=lsb_code
  endif
  ncomm%sideband(it)=sideband(sbcode)
  !
  !Find FIF
  ncomm%fcent(it)=rec%desc%bbref(2) ! Inner baseband by default
  ! Check band edge
  ! LSB case
  if (sbcode.eq.lsb_code) then
    if ((frf+ncomm%fcent(it)).lt.rec%desc%lolim(1,iband)) then
      ncomm%fcent(it)=rec%desc%lolim(1,iband)-frf
    else if ((frf+ncomm%fcent(it)).gt.rec%desc%lolim(2,iband)) then
      ncomm%fcent(it)=rec%desc%lolim(2,iband)-frf
    endif
  ! USB case
  else if (sbcode.eq.usb_code) then
    if ((frf-ncomm%fcent(it)).lt.rec%desc%lolim(1,iband)) then
      ncomm%fcent(it)=frf-rec%desc%lolim(1,iband)
    else if ((frf-ncomm%fcent(it)).gt.rec%desc%lolim(2,iband)) then
      ncomm%fcent(it)=frf-rec%desc%lolim(2,iband)
    endif
  endif
  !
end subroutine noema_default_tuning
!
subroutine rec_noema_online(line,error)
  use gbl_message
  use gkernel_interfaces
  use astro_interfaces, except_this=>rec_noema_online
  use astro_types
  use ast_astro
  use my_receiver_globals
  use plot_molecules_globals
  use frequency_axis_globals
  !-----------------------------------------------------------------------
  ! @ private
  ! ASTRO implementation of the tuning command in OBS\LINE
  ! Decode the line and fill noema%comm structure used as input for true tuning routines
  !-----------------------------------------------------------------------
  character(len=*), intent(inout) :: line        ! command line
  logical, intent(inout) :: error
  ! local
  character(len=*),parameter :: rname='LINE'
  integer(kind=4), parameter :: optrec=1        ! /RECEIVER option
  integer(kind=4) :: inputrec,izoom,itune,lname
  character(len=128) :: mess
  ! taken form obs:
  logical :: sideband_entered,fcent_entered
  integer :: ipar,lpar,harm
  real(kind=8) :: fcent,tmpnum
  character(len=12) :: par
  character(len=4) :: band
  !
  error = .false.
  !
  if (obsname.ne.'NOEMA') then
    call astro_message(seve%e,rname,'Inconsistency between Receiver Name and Observatory')
    error = .true.
    return
  endif
  !
  !Define receiver parameters
  call astro_def_receiver(rname,'NOEMA',rec,error)
  if (error) return
  !Define ifproc parameters
  call rec_define_noema_ifproc(noema%ifproc,error)
  if (error) return
  ! Put in memory the current ASTRO%SOURCE parameters
  rec%desc%redshift=.false.           ! source redshift not taken into account
  call astro_def_recsource(rname,rec,error)
  if (error) return
  ! Warning in redshift because line does not handle this
  if (rec%source%sourcetype.ne.soukind_none.and.rec%source%z.ne.0) then
    call astro_message(seve%e,rname,'LINE command does not take into account source redshift')
    call astro_message(seve%e,rname,'Change your source and try again')
    error=.true.
    return
  endif
  !
  ! Defaults for some parameters in the NOEMA_OBS case:
  call noema_reset_setup(noema,error)
  if (error) return
  noema%comm%n_tunings=1                 ! start with only 1 tuning in memory
  itune=1
  noema%comm%fixedfreq=.true.           ! no concept of tuning grid in OBS
  noema%comm%zoommode='SINGLE'
  izoom=1
  !
  ! Now parse the command line
  ! Line name
  call sic_ch(line,0,1,noema%comm%name(itune),lname,.true.,error)
  if (error) return
  ! Frequency
  call sic_r8(line,0,2,noema%comm%frest(itune),.true.,error)
  if (error) return
  !Option /RECEIVER
  inputrec=0
  call sic_i4(line,optrec,1,inputrec,.false.,error)
  if (error) return
  !
  noema%comm%sideband(itune)='USB '
  noema%comm%fcent(itune)=rec%desc%bbref(2) !center of inner baseband
  ! OBS variables - some will not be used but must be parsed
  harm = 0
  sideband_entered = .false.
  fcent_entered =  .false.
  do ipar=3,sic_narg(0)
    call sic_ke(line,0,ipar,par,lpar,.true.,error)
    if (error) return
    !
    if (par(1:1).eq.'H') then
      !
      ! HIGH
      call astro_message(seve%w,rname,  &
            'LOCK LOW|HIGH is not an input parameter anymore')
      !
    elseif (par(1:1).eq.'L') then
      !
      ! LOW
      if (lpar.ge.2.and.par(1:2).eq.'LO') then
          call astro_message(seve%w,rname, &
              'LOCK LOW|HIGH is not an input parameter anymore')
          cycle
      endif
      !
      ! LSB
      if (sideband_entered) then
        call astro_message(seve%e,rname,'Ambiguous entries: LSB or USB?')
        error = .true.
        return
      endif
! TO BE CHECKED AFTER TUNING (LIKE /RECE and found rec)
!         if (band(1:1).eq.'U') then
!            call astro_message(seve%w,rname,'Default tuning (USB) forced to LSB')
!         endif
      band = 'LSB '
      sideband_entered = .true.
      !
    elseif (par(1:1).eq.'U') then
       !
       ! USB
      if (sideband_entered) then
        call astro_message(seve%e,rname,'Ambiguous entries: LSB or USB?')
        error = .true.
        return
      endif
!         if (band(1:1).eq.'L') then
!            call astro_message(seve%w,rname,'Default tuning (LSB) forced to USB')
!         endif
      band = 'USB '
      sideband_entered = .true.
        !
    else
      ! A number...
      call sic_math_dble (par,lpar,tmpnum,error)
      if (error) then
        write (mess,'(a,1x,a)') 'Error decoding input:',par(1:lpar)
        call astro_message(seve%e,rname,mess)
        error=.true.
        return
      endif
      if (tmpnum.gt.rec%desc%iflim(1).and.tmpnum.lt.rec%desc%iflim(2)) then
        if (fcent_entered) then
          call astro_message(seve%e,rname,'IF center frequency already entered')
          error = .true.
          return
        else
          ! Fcent in MHz
          fcent = tmpnum
          fcent_entered = .true.
        endif
      elseif (tmpnum.ge.32 .and. tmpnum.le.70) then
        ! Harmonic number - old system  (<2010)
        call astro_message(seve%e,rname,'Invalid harmonic number 1')
        error = .true.
        return
      elseif (tmpnum.ge.5 .and. tmpnum.le.9) then
        ! Harmonic number - new system  (>2010)
        harm = tmpnum
      else
        write (mess,'(a,1x,a)') 'Error decoding input:',par(1:lpar)
        call astro_message(seve%e,rname,mess)
        error=.true.
        return
      endif
    endif
  enddo
  if (sideband_entered) then
    noema%comm%sideband(itune)=band
  endif
  if (fcent_entered) then
    noema%comm%fcent(itune)=fcent
  endif
  !
  !Do the tuning
  call noema_setup(rname,noema,rec,error)
  if (error) return
  !
  !Check that receiver band is OK with /RECEIVER option
  if (inputrec.ne.0.and.inputrec.ne.rec%tune(itune)%iband) then
    call astro_message(seve%e,rname,'Frequency/ReceiverBand mismatch:')
    write (mess,'(f0.3,1x,a,1x,i0)') noema%comm%frest(itune),'is not in Band',inputrec
    call astro_message(seve%e,rname,mess)
    error=.true.
    return
  endif
  !
  ! Now the plotting
  !
  if (noema%comm%zoommode.eq.'SINGLE') then
    if (izoom.le.0.or.izoom.gt.noema%comm%n_tunings.or.noema%comm%n_tunings.eq.0) then
      call astro_message(seve%e,rname,'Tried to zoom on a not tuned band')
      error = .true.
      return
    endif
    call rec_plot_sidebands(rec,izoom,cplot,molecules,freq_axis,error)
    if (error) return
  else 
    call astro_message(seve%e,rname,'Plot mode not understood')
    error = .true.
    return
  endif
  !
  ! IF transport
  ! define if ranges transported
  call noema_ifproc(rname,rec,noema,error)
  if (error) return
  ! Update the plot
  call noema_ifproc_plot(rname,noema,molecules,rec,cplot,freq_axis,error)
  if (error) return
  ! define polyfix and assign if "cables" to correlator units [only input, no mode yet]
  call noema_define_pfx(pfx,error)
  if (error) return
  call noema_reset_backend(pfx,spw%out,error)
  if (error) return
  spw%comm%itype=-1
  call noema_assign_units(rec,noema,pfx,error)
  if (error) return
  !
  ! Let the user with the limits in main axis frequency frame
  call rec_set_limits(rname,cplot,freq_axis%main,error)
  if (error) return
  !
end subroutine rec_noema_online
!
subroutine noema_setup(rname,noema,rec,error)
  use gbl_message
  use gkernel_interfaces
  use astro_interfaces, except_this=>noema_setup
  use astro_types
  use frequency_axis_globals
  !-----------------------------------------------------------------------
  ! @ private
  ! Check feasability and tune noema requested bands
  !-----------------------------------------------------------------------
  character(len=*), intent(in) :: rname
  type(noema_t), intent(inout) :: noema !command line parameters
  type(receiver_t), intent(inout) :: rec ! receiver description and tuning
  logical, intent(inout)        :: error
  ! Local
  integer(kind=4) :: it
  logical       :: ongrid
  type(receiver_comm_t) :: gridcomm
  real(kind=8)  :: newfcent,ftune
  character(len=200) :: mess
  !
  rec%n_tunings=noema%comm%n_tunings
  if (rec%n_tunings.gt.1) then
    call astro_message(seve%e,rname,'Only 1 band can be tuned for the time being')
    error = .true.
    return
  endif
  do it=1,rec%n_tunings
    rec%comm(it)%rec_name = 'NOEMA'
    rec%comm(it)%frest = noema%comm%frest(it)
    call sic_upper(noema%comm%sideband(it))
    if (noema%comm%sideband(it)(1:1).eq.'U') then
      rec%comm(it)%sideband = 'USB'
    else if (noema%comm%sideband(it)(1:1).eq.'L') then
      rec%comm(it)%sideband = 'LSB'
    else
      call astro_message(seve%e,rname,'Problem decoding tuning sideband code (LSB or USB)')
      error = .true.
      return
    endif
    rec%comm(it)%fcent = noema%comm%fcent(it)
    rec%comm(it)%name = noema%comm%name(it)
    ! tune the bands
    call rec_resttooutput(rname,noema%comm%frest(it),freq_axis%main,rec%source,ftune,error)
    if (error) return
    if (noema%comm%fixedfreq) then
      ! ignore the grid
      call astro_tune_receiver(rname,rec%desc,rec%source,rec%comm(it),rec%tune(it),error)
      if (error) return
    else ! we shift the tuning to the grid
      ongrid = .false.
      gridcomm = rec%comm(it)
      do while (.not.ongrid)
        call astro_tune_receiver(rname,rec%desc,rec%source,gridcomm,rec%tune(it),error)
        if (error) return
        write(mess,'(a,f10.5,a)') ' FLOTUNE = ',rec%tune(it)%flotune*ghzpermhz,' GHz'
        call astro_message(seve%i,rname,mess)
        call astro_tunegrid(rname,rec%desc,rec%tune(it),rec%source,ongrid,newfcent,error)
        if (error) return
        gridcomm%fcent = newfcent
      enddo
      rec%tune(it)%ongrid = .true.
    endif
  enddo
  !
end subroutine noema_setup
!
subroutine astro_tunegrid(rname,rdesc,rtune,rsou,ongrid,newfcent,error)
  use gbl_message
  use gkernel_interfaces
  use astro_interfaces, except_this=>astro_tunegrid
  use astro_types
  use frequency_axis_globals
  !-----------------------------------------------------------------------
  ! @ private
  ! Check that LO is on the tuning grid, propose the right if freq
  !-----------------------------------------------------------------------
  character(len=*), intent(in) :: rname
  type(receiver_desc_t), intent(in) :: rdesc ! receiver description 
  type(receiver_tune_t), intent(in) :: rtune ! receiver tuning
  type(receiver_source_t), intent(in) :: rsou ! receiver tuning
  logical, intent(inout)            :: ongrid
  real(kind=8), intent(inout)       :: newfcent
  logical, intent(inout)        :: error
  character(len=32) :: cname
  ! Local
  real(kind=8)  :: fgrid,foutput
  character(len=200) :: mess
  !
  fgrid = nint(rtune%flotune/rdesc%gridbin)*rdesc%gridbin
  if (abs(fgrid-rtune%flotune).lt.rdesc%gridtol) then
    newfcent = rtune%fcent
    ongrid = .true.
    return
  endif
  call astro_message(seve%i,rname,'Original tuning does not match the grid')
  newfcent = (rtune%flsr-fgrid)/sb_sign(rtune%sb_code)
  write(mess,'(a,f0.3,a)') 'Tuning automatically shifted to the IF Frequency = ',newfcent,' MHz'
  call astro_message(seve%i,rname,mess)
  write(mess,'(a,f0.3,a)') 'This corresponds to a shift of ',abs(rtune%fcent-newfcent),' MHz'
  call astro_message(seve%i,rname,mess)
  call astro_message(seve%i,rname,'Actual command:')
  if (rname.eq.'NEWVEL') then
    cname='TUNING'
  else
    cname=rname
  endif
  call rec_resttooutput(rname,rtune%frest,freq_axis%main,rsou,foutput,error)
  if (error) return
  write (mess,'(a,1x,f0.3,1x,a,1x,f0.3)') trim(cname),foutput/1d3,sideband(rtune%sb_code),newfcent
  call astro_message(seve%r,rname,mess)
  !
end subroutine astro_tunegrid
!
subroutine noema_ifproc(rname,rec,noema,error)
  use gbl_message
  use gkernel_interfaces
  use astro_interfaces, except_this=>noema_ifproc
  use astro_types
  !-----------------------------------------------------------------------
  ! @ private
  ! define the basebands brought to the backends
  ! no flxibility with noema for the time being
  !-----------------------------------------------------------------------
  character(len=*), intent(in) :: rname
  type(receiver_t), intent(inout) :: rec
  type(noema_t), intent(inout) :: noema
  logical, intent(inout)        :: error
  ! local
  logical :: dopchanged
  integer(kind=4)       :: it,is,ip,ib,icable
  integer(kind=4), parameter :: outer_code = 1
  integer(kind=4), parameter :: inner_code = 2
  integer(kind=4), parameter :: polh_code = 1
  integer(kind=4), parameter :: polv_code = 2
  integer(kind=4)       :: polcode(rec%desc%n_polar), bbcode(rec%desc%n_bbands),sb_code(rec%desc%n_sbands)
  character(len=200)    :: comm
  !
  ! Check that a tuning has been set
  if (rec%n_tunings.eq.0) then
    call astro_message(seve%e,rname,'Please define a tuning before setting the IF Processor')
    error = .true.
    return
  endif
  !
  polcode(1) = polh_code
  polcode(2) = polv_code
  !
  icable = 0
  do it = 1 ,rec%n_tunings
    do ip = 1,rec%desc%n_polar
      if (rec%desc%n_sbands.eq.2) then
        sb_code(1) = lsb_code !lsb first to match the agreement about ordering (VP 2017)
        sb_code(2) = usb_code
      else if (rec%desc%n_sbands.eq.1) then
        sb_code(1) = rec%tune(it)%sb_code
      else
        call astro_message(seve%e,rname,'Problem with number of sidebands')
        error = .true.
        return
      endif
      do is = 1,rec%desc%n_sbands
        !sort basebands by frequency to match the agreement about ordering (VP 2017)
        if (sb_code(is).eq.lsb_code) then
          bbcode(1) = outer_code
          bbcode(2) = inner_code
        else if (sb_code(is).eq.usb_code) then
          bbcode(1) = inner_code
          bbcode(2) = outer_code
        else
          call astro_message(seve%e,rname,'Problem with sideband determination')
          error = .true.
          return
        endif
        do ib = 1,rec%desc%n_bbands
          icable = icable+1
          if (icable.gt.noema%ifproc%n_ifcables) then
            call astro_message(seve%e,rname,'More basebands than available if cables')
            error=.true.
            return
          endif
          noema%ifproc%ifc(icable)%iband = rec%tune(it)%iband
          noema%ifproc%ifc(icable)%sb_code = sb_code(is)
          noema%ifproc%ifc(icable)%bb_code = bbcode(ib)
          noema%ifproc%ifc(icable)%pol_code = polcode(ip)
          write (noema%ifproc%ifc(icable)%label,'(a,i0,a,a,a)')                 &
                  rec%desc%bandname(rec%tune(it)%iband)(1:1),                   &
                  rec%tune(it)%iband,rec%desc%polname(ip)(1:1),                 &
                  sideband(sb_code(is))(1:1),rec%desc%bbname(bbcode(ib))(1:1)
        enddo ! basebands
      enddo ! sidebands
    enddo ! polar
  enddo ! receiver tunings
  !
  noema%ifproc%defined = .true.
  !
  do icable=1,noema%ifproc%n_ifcables
    write (comm,'(a,1x,i0,1x,a,1x,a)') 'Correlator input #',icable,'contains',noema%ifproc%ifc(icable)%label
    call astro_message(seve%d,rname,comm)
  enddo
  !
end subroutine noema_ifproc
!
subroutine noema_ifproc_plot(rname,noema,molecules,rec,cplot,drawaxis,error)
  use gbl_message
  use gkernel_interfaces
  use astro_interfaces, except_this=>noema_ifproc_plot
  use astro_types
  !-----------------------------------------------------------------------
  ! @ private
  ! identifies the correlator inputs on the plot
  !-----------------------------------------------------------------------
  character(len=*), intent(in) :: rname
  type(noema_t), intent(in) :: noema
  type(plot_molecules_t), intent(in) :: molecules
  type(receiver_t), intent(inout) :: rec
  type(current_boxes_t), intent(inout) :: cplot
  type(frequency_axis_t), intent(in) :: drawaxis
  logical, intent(inout) :: error
  ! local
  integer(kind=4) :: i,j,k,sb_code(m_sideband),ip,ib,ic
  character(len=200) :: comm,defchar,smallchar,molchar
  character(len=5) :: cablecode
  real(kind=8) :: fif1,frf,frest,yt,l1,l2,l3,l4,b1,b2,b3,b4
  type(draw_line_t)     :: linebb
  !
  sb_code(1) = usb_code
  sb_code(2) = lsb_code
  !
!   !Clear and redraw in case the frequency axis were changed since last command
!   call gr_execl('CLEAR DIRECTORY')
!   !
!   select case (cplot%desc%plotmode)
!   case ('RECEIVER_M')
!     call rec_plot_mbands(rec,molecules,noema%comm%fixed_scale,cplot,error)
!     if (error) return
!   case ('RECEIVER_N')
!     call rec_plot_nbands(rec,molecules,noema%comm%fixed_scale,cplot,error)
!     if (error) return
!   case ('RECEIVER_S')
!     call rec_get_itune(rec,cplot%box(1)%iband,izoom,error)
!     if (error) return
!     call rec_plot_sidebands(rec,izoom,cplot,molecules,error)
!     if (error) return
!   case ('RECEIVER_Z')
!     call rec_zoom(molecules,noema%comm%fz1,noema%comm%fz2,rec,cplot,error)
! !     if (error) return
!   end select
  if (.not.(gtexist('<GREG<BOX1'))) then
    call gr_execl('CLEAR DIRECTORY')
    call rec_plot_mbands(rec,molecules,noema%comm%fixed_scale,cplot,drawaxis,error)
    if (error) return
  endif
  !
  write (defchar,'(a,1x,f0.3)') 'SET CHARACTER',cplot%desc%defchar
  write (smallchar,'(a,1x,f0.3)') 'SET CHARACTER',cplot%desc%smallchar
  write (molchar,'(a,1x,f0.3)') 'SET CHARACTER',cplot%desc%molchar
  !
  call gr_exec1(defchar)
  call gr_pen(colour=adefcol,error=error)
  if (error) return
  !
  write (comm,'(a)') 'CHANGE DIRECTORY'
  call gr_execl(comm)
  if (gtexist('IFPROC')) then
    write (comm,'(a)') 'DESTROY DIRECTORY IFPROC'
    call gr_execl(comm)
  endif
  write (comm,'(a)') 'CREATE DIRECTORY IFPROC'
  call gr_execl(comm)
  do i=1,cplot%nbox
    write (comm,'(a,i0)') 'CHANGE DIRECTORY <GREG<BOX',i
    call gr_execl(comm)
    l1 = cplot%box(i)%rest%xmin
    l2 = cplot%box(i)%rest%xmax
    l3 = cplot%box(i)%rest%ymin
    l4 = cplot%box(i)%rest%ymax
    b1 = cplot%box(i)%phys%xmin
    b2 = cplot%box(i)%phys%xmax
    b3 = cplot%box(i)%phys%ymin
    b4 = cplot%box(i)%phys%ymax
    write (comm,'(a)') 'CHANGE DIRECTORY'
    call gr_execl(comm)
    write (comm,'(a)') 'CHANGE DIRECTORY IFPROC'
    call gr_execl(comm)
    write (comm,'(a,4(1x,f0.3))') 'SET BOX_LOCATION',b1,b2,b3,b4
    call gr_exec1(comm)
    write (comm,'(a,4(1x,f0.3))') 'LIMITS',l1,l2,l3,l4
    call gr_exec1(comm)
    do j=1,rec%n_tunings
      if (rec%tune(j)%iband.ne.cplot%box(i)%iband) cycle
      do k=1,rec%desc%n_sbands
          do ib=1,rec%desc%n_bbands
            do ip=1,rec%desc%n_polar
              write (cablecode,'(a,i0,a,a,a,a)') rec%desc%bandname(rec%tune(j)%iband)(1:1),            &
                  rec%tune(j)%iband,rec%desc%polname(ip)(1:1),sideband(sb_code(k))(1:1),rec%desc%bbname(ib)(1:1)
              do ic=1,noema%ifproc%n_ifcables
                if (noema%ifproc%ifc(ic)%label.ne.cablecode) cycle
                fif1 = (rec%desc%bblim(2,ib)+rec%desc%bblim(1,ib))/2d0
                yt = l3+((l4-l3)/6)*ip
                call if1torf(rec%tune(j)%flo1,fif1,sb_code(k),frf,error)
                if (error) return
                call rftorest(rec%source%dopshift,frf,frest,error)
                if (error) return
                call gr_pen(colour=abbifproccol,error=error)
                if (error) return
                call gr_exec1(smallchar)
                write (comm,'(a,1x,f0.6,1x,f0.3,1x,a,a,1x,a)')               &
                  'DRAW TEXT',frest,yt,'"',trim(cablecode),'" 5 0 /USER /CLIP'
                call gr_exec1(comm)
                call gr_exec(defchar)
                call noema_draw_confusion(rname,rec,j,ib,sb_code(k),cplot%box(i)%rest,cplot%desc,error)
                call gr_pen(colour=alowrescol,error=error)
                if (error) return
                if (error) return
                !draw the limit of the baseband
                call if1torf(rec%tune(j)%flo1,rec%desc%bblim(1,ib),sb_code(k),frf,error)
                if (error) return
                call rftorest(rec%source%dopshift,frf,linebb%xmin,error)
                if (error) return
                call if1torf(rec%tune(j)%flo1,rec%desc%bblim(2,ib),sb_code(k),frf,error)
                if (error) return
                call rftorest(rec%source%dopshift,frf,linebb%xmax,error)
                if (error) return
                linebb%ymin = cplot%box(i)%rest%ymin+(cplot%box(i)%rest%ymax-cplot%box(i)%rest%ymin)/10
                linebb%ymax = linebb%ymin
                linebb%col   = abbifproccol
                linebb%dash  = 1
                call gr_exec1(smallchar)
                call rec_draw_arrow(linebb,cplot%box(i)%rest,error)
                if (error) return
                call gr_exec1(defchar)
                frest = linebb%xmin
                linebb%xmin = linebb%xmax
                linebb%xmax = frest
                call rec_draw_arrow(linebb,cplot%box(i)%rest,error)
                if (error) return
                call gr_pen(colour=alowrescol,error=error)
                if (error) return
                call gr_exec1(defchar)
              enddo ! cable
            enddo ! polar
            !
          enddo !baseband
!        endif
      enddo !sideband
      !plot again molecules
      call gr_exec1(molchar)
      call rec_draw_molecules(molecules,cplot%box(i)%rest,error)
      if (error) return
      call gr_exec1(defchar)
      ! plot tuning frequency
      call rec_draw_linetune(rec%tune(j),cplot%box(i)%rest,error)
      if (error) return
      !
       write (comm,'(a)') 'CHANGE DIRECTORY'
      call gr_execl(comm)
    enddo !tuning
    write (comm,'(a)') 'CHANGE DIRECTORY'
    call gr_execl(comm)
  enddo ! boxes
  call gr_pen(colour=adefcol,error=error)
  if (error) return
  !
end subroutine noema_ifproc_plot
!
subroutine noema_draw_confusion(rname,rec,itune,bb_code,sb_code,fbox,cpdesc,error)
  use gbl_message
  use gkernel_interfaces
  use astro_interfaces, except_this=>noema_draw_confusion
  use astro_types
  !-----------------------------------------------------------------------
  ! @ private
  ! identifies the confusion zone of NOEMA
  ! S18 version: only the 10MHz of the LO2 parasite
  !-----------------------------------------------------------------------
  character(len=*), intent(in) :: rname
  type(receiver_t), intent(in) :: rec
  integer(kind=4), intent(in)   :: itune
  integer(kind=4), intent(in)   :: bb_code
  integer(kind=4), intent(in)   :: sb_code
  type(frequency_box_user_t), intent(in) :: fbox ! must be rest
  type(current_boxes_desc_t), intent(in) :: cpdesc 
  logical, intent(inout) :: error
  ! Local
  real(kind=8)  :: frf,xt,yt
  type(draw_rect_t) :: confrect
  character(len=200) :: comm,smallchar,defchar
  integer(kind=4) :: iplotmode,ipos
  integer(kind=4), parameter :: outer_code=1
  integer(kind=4), parameter :: inner_code=2
  !
  iplotmode=cpdesc%plotmode
  !Compute Confusion zone in rest frame to draw rectangle
  call if1torf(rec%tune(itune)%flo1,rec%desc%if1conf(1),sb_code,frf,error)
  if (error) return
  call rftorest(rec%source%dopshift,frf,confrect%xmin,error)
  if (error) return
  call if1torf(rec%tune(itune)%flo1,rec%desc%if1conf(2),sb_code,frf,error)
  if (error) return
  call rftorest(rec%source%dopshift,frf,confrect%xmax,error)
  if (error) return
  confrect%ymin = fbox%ymin
  confrect%ymax = fbox%ymax
  confrect%col=aconfucol
  ! Draw rectangle
  call gr_pen(colour=aconfucol,error=error)
  if (error) return
  call rec_draw_frect(confrect,fbox,error)
  if (error) return
  ! Write label (depends on plotmode)
  write (defchar,'(a,1x,f0.3)') 'SET CHARACTER',cpdesc%defchar
  write (smallchar,'(a,1x,f0.3)') 'SET CHARACTER',cpdesc%smallchar
  !
  call gr_exec1(smallchar)
  if (iplotmode.eq.pm_basebands) then
    if (sb_code.eq.lsb_code.and.bb_code.eq.outer_code) ipos = 3
    if (sb_code.eq.lsb_code.and.bb_code.eq.inner_code) ipos = 1
    if (sb_code.eq.usb_code.and.bb_code.eq.inner_code) ipos = 3
    if (sb_code.eq.usb_code.and.bb_code.eq.outer_code) ipos = 1
    write (comm,'(a,1x,i0)') 'DRAW TEXT 0 -0.5 "LO2" 5 0 /character',ipos
    call gr_exec1(comm)
  else if (iplotmode.eq.pm_receiver) then
    write (comm,'(a)') 'DRAW TEXT 0 -0.75 "LO2" 5 0 /character 2'
    call gr_exec1(comm)
  else if (iplotmode.eq.pm_allbands.or.iplotmode.eq.pm_tunedbands) then
    xt=(confrect%xmax+confrect%xmin)/2d0
    yt=confrect%ymin-(confrect%ymax-confrect%ymin)/20d0
    write (comm,'(a,1x,f0.3,1x,f0.3,1x,a)') 'DRAW TEXT',xt,yt,'"LO2" 5 0 /user'
    call gr_exec1(comm)
  else if (iplotmode.eq.pm_proposal) then
    if (bb_code.eq.1) then
      xt=(confrect%xmax+confrect%xmin)/2d0
      yt=confrect%ymin-(confrect%ymax-confrect%ymin)/20d0
      write (comm,'(a,1x,f0.3,1x,f0.3,1x,a)') 'DRAW TEXT',xt,yt,'"LO2" 5 0 /user'
      call gr_exec1(comm)
    endif
  else
    call astro_message(seve%e,rname,'Problem with plot mode')
    error=.true.
    return
  endif
  call gr_pen(colour=adefcol,error=error)
  if (error) return
  call gr_exec1(defchar)
  !
end subroutine noema_draw_confusion
!
subroutine noema_baseband(line,error)
  use gbl_message
  use gkernel_interfaces
  use astro_interfaces, except_this=>noema_baseband
  use astro_types
  use ast_astro
  use my_receiver_globals
  use plot_molecules_globals
  use string_parser_types
  use frequency_axis_globals
  !-----------------------------------------------------------------------
  ! @ private
  ! Baseband and mode selection (in preparation for SPW definition)
  !
  !-----------------------------------------------------------------------
  character(len=*), intent(inout) :: line        ! command line
  logical, intent(inout) :: error
  ! local
  character(len=*), parameter :: rname='BASEBAND'
  type(string_parser_t) :: sp
  integer(kind=4) :: iarg,lpar,narg,ncorrmode,selmode
  integer(kind=4) :: pol_sel,bb_sel,sb_sel,selband,isel,iu,it
  integer(kind=4),parameter :: m_corrmodes = 3
  character(len=12) :: par,corrmode,selcode
  character(len=12) :: corrmodes(m_corrmodes)
  character(len=200) :: mess
  integer(kind=4), parameter :: optreset=1
  logical       :: doreset,dopchanged,gotcode,gotmode,gottype,dooff
  data corrmodes/'62.5','250','OFF'/
  integer(kind=4), parameter :: ipband=1
  integer(kind=4), parameter :: ippol=2
  integer(kind=4), parameter :: ipsb=3
  integer(kind=4), parameter :: ipbb=4
  integer(kind=4), parameter :: nband=3
  integer(kind=4), parameter :: npol=2
  integer(kind=4), parameter :: nsb=2
  integer(kind=4), parameter :: nbb=2
  integer(kind=4)       :: parsecode(4)
  character(len=2) :: bb_band(nband)
  character(len=1) :: bb_pol(npol),bb_sb(nsb),bb_bb(nbb)
  data bb_band/'B1','B2','B3'/
  data bb_pol/'H','V'/
  data bb_sb/'U','L'/
  data bb_bb/'O','I'/
  !
  !Preliminary checks
  if (rec%n_tunings.eq.0) then
    call astro_message(seve%e,rname,'Please define a tuning (TUNING command) before setting up backends')
    error = .true.
    return
  endif
  if (.not.noema%ifproc%defined) then
    call astro_message(seve%e,rname,'Problem with IF Processor part')
    error = .true.
    return
  endif
  !
  !check that doppler did not changed since last tuning
  call rec_check_doppler(rec%source,rec%desc%redshift,dopchanged,error)
  if (dopchanged) then
    call astro_message(seve%e,rname,'Source changed since last tuning')
    call astro_message(seve%i,rname,'You should set again the tuning before working with backends')
    call rec_display_error('Source changed since last tuning',error)
    error = .true.
    return
  endif
  !
  narg=sic_narg(0)
  doreset=sic_present(optreset,0)
  !
  !Defaults codes
  pol_sel=0
  bb_sel=0
  sb_sel=0
  selband=0
  selcode=''
  selmode=1
  do iu=1,m_backunits
    noema%selunit%usel(iu)=-1
  enddo
  !
  !Reading arguments
  gotmode=.false.
  gotcode=.false.
  if (narg.gt.2) then
    call astro_message(seve%e,rname,'At most 2 arguments')
    error=.true.
    return
  else if (narg.gt.0) then
    do iarg=1,sic_narg(0)
      par = 'NULL'
      call sic_ke(line,0,iarg,par,lpar,.false.,error)
      if (error) return
      if (par.eq.'NULL') exit
      call sic_ambigs_sub(rname,par,corrmode,ncorrmode,corrmodes,m_corrmodes,error)
      if (error) return
      if (ncorrmode.eq.0) then
        if (gotcode) then
          call astro_message(seve%e,rname,'Selection Code already entered. Please enter all codes without spaces')
          error=.true.
          return
        endif
        selcode=par
        gotcode=.true.
      else
        if (gotmode) then
          call astro_message(seve%e,rname,'Only one mode at a once !')
          error=.true.
          return
        endif
        selmode=ncorrmode
        gotmode=.true.
      endif
    enddo
  endif
  !
  ! Check mode selection
  dooff=.false.
  corrmode=corrmodes(selmode)
  if (corrmode.eq.'OFF') then
    ! OFF CASE
    selmode=-1
    dooff=.true.
  else if (selmode.ne.1) then
    call astro_message(seve%e,rname,'Mode not yet implemented (only mode 1)')
    error=.true.
    return
  endif
  !
  !Parse the user code
  if (len_trim(selcode).ne.0) then
    call string_parser_addlist(sp,ipband,bb_band,error)
    if (error) return
    call string_parser_addlist(sp,ippol,bb_pol,error)
    if (error) return
    call string_parser_addlist(sp,ipsb,bb_sb,error)
    if (error) return
    call string_parser_addlist(sp,ipbb,bb_bb,error)
    if (error) return
    call string_parser_parse('BASEBAND',sp,selcode,.false.,parsecode,error)
    if (error) return
  else 
    parsecode(ipband)=0
    parsecode(ippol)=0
    parsecode(ipsb)=0
    parsecode(ipbb)=0
  endif
  !
  if (parsecode(ippol).eq.0) then
    noema%selunit%polmode = "B" ! BOTH polar same config
  else
    noema%selunit%polmode = "S"
  endif
  !
  ! Loop over pfx%units to put the right numbers in ifsel() and define the mode when needed
  isel=0
  do iu=1,pfx%n_units
    if (pfx%unit(iu)%iband.ne.parsecode(ipband).and.parsecode(ipband).ne.0) cycle
    if (pfx%unit(iu)%pol_code.ne.parsecode(ippol).and.parsecode(ippol).ne.0) cycle
    if (pfx%unit(iu)%sb_code.ne.parsecode(ipsb).and.parsecode(ipsb).ne.0) cycle
    if (pfx%unit(iu)%bb_code.ne.parsecode(ipbb).and.parsecode(ipbb).ne.0) cycle
    isel=isel+1
    noema%selunit%usel(isel)=iu
    if (dooff.or.doreset.or.selmode.ne.pfx%unit(iu)%imode) then
      write (mess,'(a,1x,i0)') 'Resetting unit ',iu
      call astro_message(seve%i,rname,mess)
      call pfx_reset_unit(pfx%unit(iu),spw%out,error)
      if (error) return
      if (selmode.ne.pfx%unit(iu)%imode) then
        pfx%unit(iu)%imode=selmode
      endif
      if (.not.dooff) then
        call pfx_fixed_spw(rname,rec,pfx%unit(iu),spw,error)
        if (error) return
      endif
    endif
  enddo !iu
  if (isel.eq.0) then
    call astro_message(seve%e,rname,'The selection returns nothing')
    error = .true.
    return
  endif
  ! remove empty spw (if there was a reset)
  call noema_compress_spw(spw%out,error)
  if (error) return
  ! sort spw
  call noema_sort_spw(spw%out,error)
  if (error) return
  ! look for conflicts
  call noema_check_conflicts(rname,spw%out,pfx,error)
  if (error) return
  !
  noema%selunit%n_ifsel=isel
  write (mess,'(a)') 'Selected Basebands are:'
  do iu=1,m_backunits
    if (noema%selunit%usel(iu).ne.-1) then
      write (mess,'(a,1x,a)') trim(mess),trim(noema%ifproc%ifc(noema%selunit%usel(iu))%label)
    endif
  enddo !iu
  call astro_message(seve%i,rname,mess)
  ! Find if flexible spw can be configured
  spw%comm%imode=selmode
  gottype=.false.
  do it=1,pfx%unit(noema%selunit%usel(1))%mode(selmode)%n_types
    if (.not.(pfx%unit(noema%selunit%usel(1))%mode(selmode)%chtype(it)%move_chunk)) cycle
    if (gottype) then
      call astro_message(seve%e,rname,'Several kinds of flexible chunks ??')
      error=.true.
      return
    endif
    spw%comm%itype=it 
    gottype = .true.
  enddo
  spw%comm%resol= &
    pfx%unit(noema%selunit%usel(1))%mode(pfx%unit(noema%selunit%usel(1))%imode)%chtype(spw%comm%itype)%df_chunks*1000 ! in kHz
  !
  call noema_plot_selpfx(rname,rec,noema,pfx,spw,cplot,molecules,freq_axis,error)
  if (error) return
  !
  if (rec%source%sourcetype.eq.soukind_full) then
    call noema_oplot_dopminmax(spw%out,pfx,rec,cplot,freq_axis,error)
    if (error) return
  endif
  !
  ! Let the user with the limits in main axis frequency frame
  call rec_set_limits(rname,cplot,freq_axis%main,error)
  if (error) return
  !
end subroutine noema_baseband
!
subroutine noema_baseband_online(line,error)
  use gbl_message
  use gkernel_interfaces
  use astro_interfaces, except_this=>noema_baseband_online
  use astro_types
  use my_receiver_globals
  use plot_molecules_globals
  use string_parser_types
  use frequency_axis_globals
  !-----------------------------------------------------------------------
  ! @ private
  ! online Baseband and mode selection: as done in OBS
  ! Give a mode to the PFX units
  !-----------------------------------------------------------------------
  character(len=*), intent(inout) :: line        ! command line
  logical, intent(inout) :: error
  ! local
  character(len=*), parameter :: rname='BASEBAND'
  type(string_parser_t) :: sp
  integer(kind=4) :: iarg,lpar,narg,ncorrmode,selmode
  integer(kind=4) :: pol_sel,bb_sel,sb_sel,selband,isel,iu,inputrec,izoom
  integer(kind=4),parameter :: m_corrmodes = 3
  character(len=12) :: par,corrmode,selcode
  character(len=12) :: corrmodes(m_corrmodes)
  character(len=200) :: mess
  integer(kind=4), parameter :: optrec=1
  logical       :: dopchanged,gotcode,gotmode,dorec
  data corrmodes/'1','2','OFF'/
  integer(kind=4), parameter :: ipband=1
  integer(kind=4), parameter :: ippol=2
  integer(kind=4), parameter :: ipsb=3
  integer(kind=4), parameter :: ipbb=4
  integer(kind=4), parameter :: nband=3
  integer(kind=4), parameter :: npol=2
  integer(kind=4), parameter :: nsb=2
  integer(kind=4), parameter :: nbb=2
  integer(kind=4), parameter :: m_keys=1
  integer(kind=4)       :: parsecode(4),nkey
  character(len=2) :: bb_band(nband)
  character(len=1) :: bb_pol(npol),bb_sb(nsb),bb_bb(nbb)
  character(len=1) :: keys(m_keys),key
  data keys/'*'/
  data bb_band/'B1','B2','B3'/
  data bb_pol/'H','V'/
  data bb_sb/'U','L'/
  data bb_bb/'O','I'/
  !
  !Preliminary checks
  if (rec%n_tunings.eq.0) then
    call astro_message(seve%e,rname,'Please define a tuning (TUNING command) before setting up backends')
    error = .true.
    return
  endif
  if (.not.noema%ifproc%defined) then
    call astro_message(seve%e,rname,'Problem with IF Processor part')
    error = .true.
    return
  endif
  !
  !check that doppler did not changed since last tuning ! SHOULD DISAPPEAR in the FUTURE
  call rec_check_doppler(rec%source,rec%desc%redshift,dopchanged,error)
  if (dopchanged) then
    call astro_message(seve%e,rname,'Source changed since last tuning')
    call astro_message(seve%i,rname,'You should set again the tuning before working with backends')
    call rec_display_error('Source changed since last tuning',error)
    error = .true.
    return
  endif
  !
  narg=sic_narg(0)
  dorec=sic_present(optrec,0)
  inputrec=0
  call sic_i4(line,optrec,1,inputrec,.false.,error)
  if (error) return
  !
  !Defaults codes
  pol_sel=0
  bb_sel=0
  sb_sel=0
  selband=0
  selcode=''
  selmode=1 ! defaults is mode 1
  corrmode='1'
  do iu=1,m_backunits
    noema%selunit%usel(iu)=-1
  enddo
  !
  ! Input checks
  if (narg.ne.2) then
    call astro_message(seve%i,rname,'BASEBAND needs 2 arguments. Current state:')
    call noema_list_pfx(rname,pfx,freq_axis%main,rec%source,error)
    if (error) return
    return
  endif
  !Reading arguments
  gotmode=.false.
  gotcode=.false.
  do iarg=1,sic_narg(0)
    call sic_ke(line,0,iarg,par,lpar,.true.,error)
    if (error) return
    ! First try to recognize a correlator mode
    call sic_ambigs_sub(rname,par,corrmode,ncorrmode,corrmodes,m_corrmodes,error)
    if (error) return
    if (ncorrmode.eq.0) then ! correlator mode not recognized
      if (gotcode) then
        call astro_message(seve%e,rname,'Selection Code already entered. Please enter all codes without spaces')
        error=.true.
        return
      endif
      call sic_ambigs_sub(rname,par,key,nkey,keys,m_keys,error)
      if (error) return
      if (nkey.eq.0) then !key '*' not recognized
        selcode=par ! parameter is thus a selection code 
      else
        selcode=''
      endif
      gotcode=.true.
    else
      ! correlator mode recognized
      if (gotmode) then
        call astro_message(seve%e,rname,'Only one mode at a once !')
        error=.true.
        return
      endif
      selmode=ncorrmode
      gotmode=.true.
    endif
  enddo
  !
  ! Check mode selection
  corrmode=corrmodes(selmode)
  if (corrmode.eq.'OFF') then
    ! OFF CASE
    selmode=-1
  else if (corrmode.ne.'1') then
    call astro_message(seve%e,rname,'Mode not yet implemented (only mode 1)')
    error=.true.
    return
  endif
  !
  !Parse the user code
  if (len_trim(selcode).ne.0) then
    call string_parser_addlist(sp,ipband,bb_band,error)
    if (error) return
    call string_parser_addlist(sp,ippol,bb_pol,error)
    if (error) return
    call string_parser_addlist(sp,ipsb,bb_sb,error)
    if (error) return
    call string_parser_addlist(sp,ipbb,bb_bb,error)
    if (error) return
    call string_parser_parse('BASEBAND',sp,selcode,.false.,parsecode,error)
    if (error) return
  else 
    parsecode(ipband)=0
    parsecode(ippol)=0
    parsecode(ipsb)=0
    parsecode(ipbb)=0
  endif
  !
  ! Case with /RECEIVER option
  if (dorec) then
    if (parsecode(ipband).ne.0) then
      if (parsecode(ipband).ne.inputrec) then
        call astro_message(seve%e,rname,'Inconsistency in Band selection')
        error=.true.
        return
      endif
    else if (parsecode(ipband).eq.0) then
      ! use the option input instead of the code
      parsecode(ipband)=inputrec
    endif
  endif
  !
  ! Loop over pfx%units to put the right numbers in ifsel() and define the mode when needed
  isel=0
  do iu=1,pfx%n_units
    if (pfx%unit(iu)%iband.ne.parsecode(ipband).and.parsecode(ipband).ne.0) cycle
    if (pfx%unit(iu)%pol_code.ne.parsecode(ippol).and.parsecode(ippol).ne.0) cycle
    if (pfx%unit(iu)%sb_code.ne.parsecode(ipsb).and.parsecode(ipsb).ne.0) cycle
    if (pfx%unit(iu)%bb_code.ne.parsecode(ipbb).and.parsecode(ipbb).ne.0) cycle
    isel=isel+1
    write (mess,'(a,1x,i0)') 'Resetting unit ',iu
    call astro_message(seve%i,rname,mess)
    ! Reset unit
    call pfx_reset_unit(pfx%unit(iu),spw%out,error)
    if (error) return
    if (selmode.ne.pfx%unit(iu)%imode) then
      pfx%unit(iu)%imode=selmode
    endif
    if (selmode.ne.-1) then
      call pfx_fixed_spw(rname,rec,pfx%unit(iu),spw,error)
      if (error) return
    endif
  enddo !iu
  if (isel.eq.0) then
    call astro_message(seve%e,rname,'The selection returns nothing')
    error = .true.
    return
  endif
  ! remove empty spw (if there was a reset)
  call noema_compress_spw(spw%out,error)
  if (error) return
  ! sort spw
  call noema_sort_spw(spw%out,error)
  if (error) return
  ! look for conflicts
  call noema_check_conflicts(rname,spw%out,pfx,error)
  if (error) return
  !
  ! Now the plotting
  ! PLOT
  !Select all the units for the plot
  noema%selunit%n_ifsel=pfx%n_units
  noema%selunit%polmode='B'
  do iu=1,pfx%n_units
    noema%selunit%usel(iu)=iu
  enddo
  !
  call noema_plot_selpfx(rname,rec,noema,pfx,spw,cplot,molecules,freq_axis,error)
  if (error) return
  ! Let the user with the limits in main axis frequency frame
  call rec_set_limits(rname,cplot,freq_axis%main,error)
  if (error) return
  !
end subroutine noema_baseband_online
