subroutine sic_diff(line,error)
  use gildas_def
  use gbl_message
  use sic_dependencies_interfaces
  use sic_interfaces, except_this=>sic_diff
  use sic_dictionaries
  !---------------------------------------------------------------------
  ! @ private
  !  Support routine for command
  !    SIC\DIFF Header1 Header2
  !---------------------------------------------------------------------
  character(len=*), intent(in)    :: line   !
  logical,          intent(inout) :: error  !
  ! Local
  character(len=*), parameter :: rname='DIFF'
  integer(kind=4), parameter :: typenone=0,typevar=1,typefile=2
  integer(kind=4) :: type1,type2
  type(sic_descriptor_t) :: desc1,desc2
  !
  type1 = typenone
  type2 = typenone
  !
  call getarg(1,type1,desc1,error)
  if (error)  return
  call getarg(2,type2,desc2,error)
  if (error)  goto 10
  !
  call sic_diff_descriptors(desc1,desc2,error)
  if (error)  goto 10
  !
10 continue
  ! Cleaning
  call closearg(type1,desc1)
  call closearg(type2,desc2)
  !
contains
  !
  subroutine getarg(iarg,argtype,desc,error)
    use sic_types
    !-------------------------------------------------------------------
    ! If the argument provides a header, return a pointer to this
    ! header, or return a null pointer if not.
    !-------------------------------------------------------------------
    integer(kind=4),        intent(in)    :: iarg     ! Argument number on command line
    integer(kind=4),        intent(inout) :: argtype  !
    type(sic_descriptor_t), intent(out)   :: desc     !
    logical,                intent(inout) :: error    !
    ! Local
    character(len=filename_length) :: name
    integer(kind=4) :: nc,ier,ivar
    type(sic_identifier_t) :: var
    !
    call sic_ch(line,0,iarg,name,nc,.true.,error)
    if (error) return
    !
    ! Guess if a Variable or a File
    var%name = name
    var%lname = len_trim(name)
    var%level = var_level
    ier = sic_hasfin(maxvar,pfvar,pnvar,dicvar,var,ivar)
    if (ier.ne.1) then
      var%level = 0
      ier = sic_hasfin(maxvar,pfvar,pnvar,dicvar,var,ivar)
    endif
    if (ier.eq.1) then
      ! Found a variable
      argtype = typevar
      call copy_descr(dicvar(ivar)%desc,desc,error)
      if (error)  return
    else
      ! Must be a file name. Will raise an error if not.
      ! Build a hand-made descriptor, will need special cleaning later on
      !
      allocate(desc%head)           ! Allocated, will be cleaned after use
      call gildas_null(desc%head)
      call gdf_read_gildas(desc%head,name,'.gdf',error,data=.true.,rank=0)
      if (error) then
        deallocate(desc%head)
        return
      endif
      desc%addr = desc%head%loca%addr
      desc%type = desc%head%gil%form
      desc%ndim = desc%head%gil%ndim
      desc%dims(1:desc%ndim) = desc%head%gil%dim(1:desc%ndim)
      ! Other elements not needed in sic_diff_datadescr
      !
      argtype = typefile
      !
    endif
  end subroutine getarg
  !
  subroutine closearg(argtype,desc)
    !-------------------------------------------------------------------
    ! Perform the necessary cleaning depending on argument type
    !-------------------------------------------------------------------
    integer(kind=4),        intent(in)    :: argtype  !
    type(sic_descriptor_t), intent(inout) :: desc     !
    ! Local
    logical :: error2
    !
    select case (argtype)
    case (typefile)
      error2 = .false.
      call gdf_close_image(desc%head,error2)
      if (error2) continue
      if (associated(desc%head%r1d))  deallocate(desc%head%r1d)
      ! ZZZ if image format is not R*4, we have a leak here.
      deallocate(desc%head)  ! Trick here: was allocated
    end select
    !
  end subroutine closearg
  !
end subroutine sic_diff
!
subroutine sic_diff_descriptors(desc1,desc2,error)
  use gbl_message
  use sic_types
  use sic_interfaces, except_this=>sic_diff_descriptors
  !---------------------------------------------------------------------
  ! @ private
  ! Show the differences between two SIC variables
  !---------------------------------------------------------------------
  type(sic_descriptor_t), intent(in)    :: desc1,desc2
  logical,                intent(inout) :: error
  ! Local
  logical :: do1,do2
  !
  do1 = associated(desc1%head)
  do2 = associated(desc2%head)
  if (do1 .and. do2) then
    call sic_diff_headers(desc1%head,desc2%head,error)
  elseif (do1 .and. .not.do2) then
    call sic_message(seve%r,'DIFF','Only first argument provides a header')
  elseif (.not.do1 .and. do2) then
    call sic_message(seve%r,'DIFF','Only second argument provides a header')
  endif  ! Else both have no header => no message
  !
  do1 = desc1%addr.ne.ptr_null
  do2 = desc2%addr.ne.ptr_null
  if (do1 .and. do2) then
    call sic_diff_datadescr(desc1,desc2,error)
  elseif (do1 .and. .not.do2) then
    call sic_message(seve%r,'DIFF','Only first argument provides data')
  elseif (.not.do1 .and. do2) then
    call sic_message(seve%r,'DIFF','Only second argument provides data')
  endif  ! Else both have no data => no message
  !
end subroutine sic_diff_descriptors
!
subroutine sic_diff_headers(h1,h2,error)
  use image_def
  use sic_interfaces, except_this=>sic_diff_headers
  !---------------------------------------------------------------------
  ! @ private
  ! Show the differences between two GDF headers
  !---------------------------------------------------------------------
  type(gildas), intent(in)    :: h1,h2
  logical,      intent(inout) :: error
  !
  call sic_diff_general(h1,h2,error)
  if (error)  return
  call sic_diff_dimension(h1,h2,error)
  if (error)  return
  call sic_diff_blanking(h1,h2,error)
  if (error)  return
  call sic_diff_extrema(h1,h2,error)
  if (error)  return
  call sic_diff_coordinate(h1,h2,error)
  if (error)  return
  call sic_diff_description(h1,h2,error)
  if (error)  return
  call sic_diff_position(h1,h2,error)
  if (error)  return
  call sic_diff_projection(h1,h2,error)
  if (error)  return
  call sic_diff_spectroscopy(h1,h2,error)
  if (error)  return
  call sic_diff_resolution(h1,h2,error)
  if (error)  return
  call sic_diff_noise(h1,h2,error)
  if (error)  return
  call sic_diff_astrometry(h1,h2,error)
  if (error)  return
  call sic_diff_telescope(h1,h2,error)
  if (error)  return
  call sic_diff_uvdata(h1,h2,error)
  if (error)  return
  !
end subroutine sic_diff_headers
!
subroutine sic_diff_general(h1,h2,error)
  use gbl_message
  use image_def
  use sic_dependencies_interfaces
  use sic_interfaces, except_this=>sic_diff_general
  !---------------------------------------------------------------------
  ! @ private
  !  Print the differences between 2 headers (section GENERAL)
  !---------------------------------------------------------------------
  type(gildas), intent(in)    :: h1,h2
  logical,      intent(inout) :: error
  ! Local
  character(len=*), parameter :: rname='DIFF'
  character(len=*), parameter :: secname='General'
  logical :: secwarned
  !
  secwarned = .false.
  call gag_diff_elem(rname,secname,secwarned,'TYPE:',       h1%char%type,      h2%char%type)
  call gag_diff_elem(rname,secname,secwarned,'VERSION_GDF:',h1%gil%version_gdf,h2%gil%version_gdf)
  call gag_diff_elem(rname,secname,secwarned,'TYPE_GDF:',   h1%gil%type_gdf,   h2%gil%type_gdf)
  call gag_diff_elem(rname,secname,secwarned,'FORMAT:',     h1%gil%form,       h2%gil%form)
  !
end subroutine sic_diff_general
!
subroutine sic_diff_dimension(h1,h2,error)
  use gbl_message
  use image_def
  use sic_dependencies_interfaces
  use sic_interfaces, except_this=>sic_diff_dimension
  !---------------------------------------------------------------------
  ! @ private
  !  Print the differences between 2 headers (section DIMENSION)
  !---------------------------------------------------------------------
  type(gildas), intent(in)    :: h1,h2
  logical,      intent(inout) :: error
  ! Local
  character(len=*), parameter :: rname='DIFF'
  character(len=*), parameter :: secname='Dimension'
  logical :: secwarned
  character(len=64) :: elname
  integer(kind=4) :: idime
  !
  if (sic_diff_presec(secname,h1%gil%dim_words,h2%gil%dim_words))  return
  !
  secwarned = .false.
  call gag_diff_elem(rname,secname,secwarned,'NDIM:',h1%gil%ndim,h2%gil%ndim)
  do idime=1,min(h1%gil%ndim,h2%gil%ndim)
    write(elname,'(A,I0,A)') 'DIM[',idime,']:'
    call gag_diff_elem(rname,secname,secwarned,elname,h1%gil%dim(idime),h2%gil%dim(idime))
  enddo
  !
end subroutine sic_diff_dimension
!
subroutine sic_diff_blanking(h1,h2,error)
  use gbl_message
  use image_def
  use sic_dependencies_interfaces
  use sic_interfaces, except_this=>sic_diff_blanking
  !---------------------------------------------------------------------
  ! @ private
  !  Print the differences between 2 headers (section BLANKING)
  !---------------------------------------------------------------------
  type(gildas), intent(in)    :: h1,h2
  logical,      intent(inout) :: error
  ! Local
  character(len=*), parameter :: rname='DIFF'
  character(len=*), parameter :: secname='Blanking'
  logical :: secwarned
  !
  if (sic_diff_presec(secname,h1%gil%blan_words,h2%gil%blan_words))  return
  !
  secwarned = .false.
  call gag_diff_elem(rname,secname,secwarned,'BLANK[1]:',h1%gil%bval,h2%gil%bval)
  call gag_diff_elem(rname,secname,secwarned,'BLANK[2]:',h1%gil%eval,h2%gil%eval)
  !
end subroutine sic_diff_blanking
!
subroutine sic_diff_extrema(h1,h2,error)
  use gbl_message
  use image_def
  use sic_dependencies_interfaces
  use sic_interfaces, except_this=>sic_diff_extrema
  !---------------------------------------------------------------------
  ! @ private
  !  Print the differences between 2 headers (section EXTREMA)
  !---------------------------------------------------------------------
  type(gildas), intent(in)    :: h1,h2
  logical,      intent(inout) :: error
  ! Local
  character(len=*), parameter :: rname='DIFF'
  character(len=*), parameter :: secname='Extrema'
  logical :: secwarned
  character(len=64) :: elname
  integer(kind=4) :: idime
  !
  if (sic_diff_presec(secname,h1%gil%extr_words,h2%gil%extr_words))  return
  !
  secwarned = .false.
  call gag_diff_elem(rname,secname,secwarned,'MIN:',h1%gil%rmin,h2%gil%rmin)
  call gag_diff_elem(rname,secname,secwarned,'MAX:',h1%gil%rmax,h2%gil%rmax)
  do idime=1,min(h1%gil%ndim,h2%gil%ndim)
    write(elname,'(A,I0,A)') 'MINLOC[',idime,']:'
    call gag_diff_elem(rname,secname,secwarned,elname,h1%gil%minloc(idime),h2%gil%minloc(idime))
    write(elname,'(A,I0,A)') 'MAXLOC[',idime,']:'
    call gag_diff_elem(rname,secname,secwarned,elname,h1%gil%maxloc(idime),h2%gil%maxloc(idime))
  enddo
  !
end subroutine sic_diff_extrema
!
subroutine sic_diff_coordinate(h1,h2,error)
  use gbl_message
  use image_def
  use sic_dependencies_interfaces
  use sic_interfaces, except_this=>sic_diff_coordinate
  !---------------------------------------------------------------------
  ! @ private
  !  Print the differences between 2 headers (section COORDINATE)
  !---------------------------------------------------------------------
  type(gildas), intent(in)    :: h1,h2
  logical,      intent(inout) :: error
  ! Local
  character(len=*), parameter :: rname='DIFF'
  character(len=*), parameter :: secname='Coordinate'
  logical :: secwarned
  character(len=64) :: elname
  integer(kind=4) :: idime
  !
  if (sic_diff_presec(secname,h1%gil%coor_words,h2%gil%coor_words))  return
  !
  secwarned = .false.
  do idime=1,min(h1%gil%ndim,h2%gil%ndim)
    write(elname,'(A,I0,A,I0,A)') 'CONVERT[1,',idime,'] (REF',idime,'):'
    call gag_diff_elem(rname,secname,secwarned,elname,h1%gil%convert(1,idime),h2%gil%convert(1,idime))
    write(elname,'(A,I0,A,I0,A)') 'CONVERT[2,',idime,'] (VAL',idime,'):'
    call gag_diff_elem(rname,secname,secwarned,elname,h1%gil%convert(2,idime),h2%gil%convert(2,idime))
    write(elname,'(A,I0,A,I0,A)') 'CONVERT[3,',idime,'] (INC',idime,'):'
    call gag_diff_elem(rname,secname,secwarned,elname,h1%gil%convert(3,idime),h2%gil%convert(3,idime))
  enddo
  !
end subroutine sic_diff_coordinate
!
subroutine sic_diff_description(h1,h2,error)
  use gbl_message
  use image_def
  use sic_dependencies_interfaces
  use sic_interfaces, except_this=>sic_diff_description
  !---------------------------------------------------------------------
  ! @ private
  !  Print the differences between 2 headers (section DESCRIPTION)
  !---------------------------------------------------------------------
  type(gildas), intent(in)    :: h1,h2
  logical,      intent(inout) :: error
  ! Local
  character(len=*), parameter :: rname='DIFF'
  character(len=*), parameter :: secname='Description'
  logical :: secwarned
  character(len=64) :: elname
  integer(kind=4) :: idime
  !
  if (sic_diff_presec(secname,h1%gil%desc_words,h2%gil%desc_words))  return
  !
  secwarned = .false.
  call gag_diff_elem(rname,secname,secwarned,'UNIT:',h1%char%unit,h2%char%unit)
  do idime=1,min(h1%gil%ndim,h2%gil%ndim)
    write(elname,'(A,I0,A)') 'UNIT',idime,':'
    call gag_diff_elem(rname,secname,secwarned,elname,h1%char%code(idime),h2%char%code(idime))
  enddo
  !
end subroutine sic_diff_description
!
subroutine sic_diff_position(h1,h2,error)
  use gbl_message
  use image_def
  use sic_dependencies_interfaces
  use sic_interfaces, except_this=>sic_diff_position
  !---------------------------------------------------------------------
  ! @ private
  !  Print the differences between 2 headers (section POSITION)
  !---------------------------------------------------------------------
  type(gildas), intent(in)    :: h1,h2
  logical,      intent(inout) :: error
  ! Local
  character(len=*), parameter :: rname='DIFF'
  character(len=*), parameter :: secname='Position'
  logical :: secwarned
  !
  if (sic_diff_presec(secname,h1%gil%posi_words,h2%gil%posi_words))  return
  !
  secwarned = .false.
  call gag_diff_elem(rname,secname,secwarned,'SOURCE:', h1%char%name,h2%char%name)
  call gag_diff_elem(rname,secname,secwarned,'SYSTEM:', h1%char%syst,h2%char%syst)
  call gag_diff_elem(rname,secname,secwarned,'RA:',     h1%gil%ra,   h2%gil%ra)
  call gag_diff_elem(rname,secname,secwarned,'DEC:',    h1%gil%dec,  h2%gil%dec)
  call gag_diff_elem(rname,secname,secwarned,'LII:',    h1%gil%lii,  h2%gil%lii)
  call gag_diff_elem(rname,secname,secwarned,'BII:',    h1%gil%bii,  h2%gil%bii)
  call gag_diff_elem(rname,secname,secwarned,'EQUINOX:',h1%gil%epoc, h2%gil%epoc)
  !
end subroutine sic_diff_position
!
subroutine sic_diff_projection(h1,h2,error)
  use gbl_message
  use image_def
  use sic_dependencies_interfaces
  use sic_interfaces, except_this=>sic_diff_projection
  !---------------------------------------------------------------------
  ! @ private
  !  Print the differences between 2 headers (section PROJECTION)
  !---------------------------------------------------------------------
  type(gildas), intent(in)    :: h1,h2
  logical,      intent(inout) :: error
  ! Local
  character(len=*), parameter :: rname='DIFF'
  character(len=*), parameter :: secname='Projection'
  logical :: secwarned
  !
  if (sic_diff_presec(secname,h1%gil%proj_words,h2%gil%proj_words))  return
  !
  secwarned = .false.
  call gag_diff_elem(rname,secname,secwarned,'A0:',    h1%gil%a0,  h2%gil%a0)
  call gag_diff_elem(rname,secname,secwarned,'D0:',    h1%gil%d0,  h2%gil%d0)
  call gag_diff_elem(rname,secname,secwarned,'ANGLE:', h1%gil%pang,h2%gil%pang)
  call gag_diff_elem(rname,secname,secwarned,'PTYPE:', projnam(h1%gil%ptyp),projnam(h2%gil%ptyp))
  call gag_diff_elem(rname,secname,secwarned,'X_AXIS:',h1%gil%xaxi,h2%gil%xaxi)
  call gag_diff_elem(rname,secname,secwarned,'Y_AXIS:',h1%gil%yaxi,h2%gil%yaxi)
  !
end subroutine sic_diff_projection
!
subroutine sic_diff_spectroscopy(h1,h2,error)
  use gbl_message
  use image_def
  use sic_dependencies_interfaces
  use sic_interfaces, except_this=>sic_diff_spectroscopy
  !---------------------------------------------------------------------
  ! @ private
  !  Print the differences between 2 headers (section SPECTROSCOPY)
  !---------------------------------------------------------------------
  type(gildas), intent(in)    :: h1,h2
  logical,      intent(inout) :: error
  ! Local
  character(len=*), parameter :: rname='DIFF'
  character(len=*), parameter :: secname='Spectroscopy'
  logical :: secwarned
  !
  if (sic_diff_presec(secname,h1%gil%spec_words,h2%gil%spec_words))  return
  !
  secwarned = .false.
  call gag_diff_elem(rname,secname,secwarned,'LINE:',   h1%char%line,h2%char%line)
  call gag_diff_elem(rname,secname,secwarned,'FREQRES:',h1%gil%fres, h2%gil%fres)
  call gag_diff_elem(rname,secname,secwarned,'IMAGFRE:',h1%gil%fima, h2%gil%fima)
  call gag_diff_elem(rname,secname,secwarned,'RESTFRE:',h1%gil%freq, h2%gil%freq)
  call gag_diff_elem(rname,secname,secwarned,'VELRES:', h1%gil%vres, h2%gil%vres)
  call gag_diff_elem(rname,secname,secwarned,'VELOFF:', h1%gil%voff, h2%gil%voff)
  call gag_diff_elem(rname,secname,secwarned,'F_AXIS:', h1%gil%faxi, h2%gil%faxi)
  call gag_diff_elem(rname,secname,secwarned,'DOPPLER:',h1%gil%dopp, h2%gil%dopp)
  call gag_diff_elem(rname,secname,secwarned,'VTYPE:',  h1%gil%vtyp, h2%gil%vtyp)
  !
end subroutine sic_diff_spectroscopy
!
subroutine sic_diff_resolution(h1,h2,error)
  use gbl_message
  use image_def
  use sic_dependencies_interfaces
  use sic_interfaces, except_this=>sic_diff_resolution
  !---------------------------------------------------------------------
  ! @ private
  !  Print the differences between 2 headers (section RESOLUTION)
  !---------------------------------------------------------------------
  type(gildas), intent(in)    :: h1,h2
  logical,      intent(inout) :: error
  ! Local
  character(len=*), parameter :: rname='DIFF'
  character(len=*), parameter :: secname='Resolution'
  logical :: secwarned
  !
  if (sic_diff_presec(secname,h1%gil%reso_words,h2%gil%reso_words))  return
  !
  secwarned = .false.
  call gag_diff_elem(rname,secname,secwarned,'MAJOR:',h1%gil%majo,h2%gil%majo)
  call gag_diff_elem(rname,secname,secwarned,'MINOR:',h1%gil%mino,h2%gil%mino)
  call gag_diff_elem(rname,secname,secwarned,'PA:',   h1%gil%posa,h2%gil%posa)
  !
end subroutine sic_diff_resolution
!
subroutine sic_diff_noise(h1,h2,error)
  use gbl_message
  use image_def
  use sic_dependencies_interfaces
  use sic_interfaces, except_this=>sic_diff_noise
  !---------------------------------------------------------------------
  ! @ private
  !  Print the differences between 2 headers (section NOISE)
  !---------------------------------------------------------------------
  type(gildas), intent(in)    :: h1,h2
  logical,      intent(inout) :: error
  ! Local
  character(len=*), parameter :: rname='DIFF'
  character(len=*), parameter :: secname='Noise'
  logical :: secwarned
  !
  if (sic_diff_presec(secname,h1%gil%nois_words,h2%gil%nois_words))  return
  !
  secwarned = .false.
  call gag_diff_elem(rname,secname,secwarned,'NOISE:',h1%gil%noise,h2%gil%noise)
  call gag_diff_elem(rname,secname,secwarned,'RMS:',  h1%gil%rms,  h2%gil%rms)
  !
end subroutine sic_diff_noise
!
subroutine sic_diff_astrometry(h1,h2,error)
  use gbl_message
  use image_def
  use sic_dependencies_interfaces
  use sic_interfaces, except_this=>sic_diff_astrometry
  !---------------------------------------------------------------------
  ! @ private
  !  Print the differences between 2 headers (section ASTROMETRY)
  !---------------------------------------------------------------------
  type(gildas), intent(in)    :: h1,h2
  logical,      intent(inout) :: error
  ! Local
  character(len=*), parameter :: rname='DIFF'
  character(len=*), parameter :: secname='Astrometry'
  logical :: secwarned
  !
  if (sic_diff_presec(secname,h1%gil%astr_words,h2%gil%astr_words))  return
  !
  secwarned = .false.
  call gag_diff_elem(rname,secname,secwarned,'MU[1]:',   h1%gil%mura,    h2%gil%mura)
  call gag_diff_elem(rname,secname,secwarned,'MU[2]:',   h1%gil%mudec,   h2%gil%mudec)
  call gag_diff_elem(rname,secname,secwarned,'PARALLAX:',h1%gil%parallax,h2%gil%parallax)
  !
end subroutine sic_diff_astrometry
!
subroutine sic_diff_telescope(h1,h2,error)
  use gbl_message
  use image_def
  use sic_dependencies_interfaces
  use sic_interfaces, except_this=>sic_diff_telescope
  !---------------------------------------------------------------------
  ! @ private
  !  Print the differences between 2 headers (section TELESCOPE)
  !---------------------------------------------------------------------
  type(gildas), intent(in)    :: h1,h2
  logical,      intent(inout) :: error
  ! Local
  character(len=*), parameter :: rname='DIFF'
  character(len=*), parameter :: secname='Telescope'
  logical :: secwarned
  character(len=64) :: elname
  integer(kind=4) :: itel
  !
  if (sic_diff_presec(secname,h1%gil%tele_words,h2%gil%tele_words))  return
  !
  secwarned = .false.
  call gag_diff_elem(rname,secname,secwarned,'NTEL:',h1%gil%nteles,h2%gil%nteles)
  do itel=1,min(h1%gil%nteles,h2%gil%nteles)
    write(elname,'(A,I0,A)') 'TEL',itel,'%NAME:'
    call gag_diff_elem(rname,secname,secwarned,elname,h1%gil%teles(itel)%ctele,h2%gil%teles(itel)%ctele)
  enddo
  !
end subroutine sic_diff_telescope
!
subroutine sic_diff_uvdata(h1,h2,error)
  use gbl_message
  use image_def
  use sic_dependencies_interfaces
  use sic_interfaces, except_this=>sic_diff_uvdata
  !---------------------------------------------------------------------
  ! @ private
  !  Print the differences between 2 headers (section UV_DATA)
  !---------------------------------------------------------------------
  type(gildas), intent(in)    :: h1,h2
  logical,      intent(inout) :: error
  ! Local
  character(len=*), parameter :: rname='DIFF'
  character(len=*), parameter :: secname='UV Data'
  logical :: secwarned
  !
  if (sic_diff_presec(secname,h1%gil%uvda_words,h2%gil%uvda_words))  return
  !
  secwarned = .false.
  call gag_diff_elem(rname,secname,secwarned,'VERSION_UV:',h1%gil%version_uv,h2%gil%version_uv)
  call gag_diff_elem(rname,secname,secwarned,'NCHAN:',     h1%gil%nchan,     h2%gil%nchan)
  call gag_diff_elem(rname,secname,secwarned,'NVISI:',     h1%gil%nvisi,     h2%gil%nvisi)
  call gag_diff_elem(rname,secname,secwarned,'NSTOKES:',   h1%gil%nstokes,   h2%gil%nstokes)
  call gag_diff_elem(rname,secname,secwarned,'NATOM:',     h1%gil%natom,     h2%gil%natom)
  call gag_diff_elem(rname,secname,secwarned,'BASEMIN:',   h1%gil%basemin,   h2%gil%basemin)
  call gag_diff_elem(rname,secname,secwarned,'BASEMAX:',   h1%gil%basemax,   h2%gil%basemax)
  !
end subroutine sic_diff_uvdata
!
function sic_diff_presec(secname,len1,len2)
  use gbl_message
  use sic_interfaces, except_this=>sic_diff_presec
  !---------------------------------------------------------------------
  ! @ private
  ! Return .true. if the section is absent from 1 or from the 2 headers
  !---------------------------------------------------------------------
  logical :: sic_diff_presec
  character(len=*), intent(in) :: secname    ! Section name
  integer(kind=4),  intent(in) :: len1,len2  ! Section lengths
  !
  if (len1.le.0 .and. len2.le.0) then
    sic_diff_presec = .true.
  elseif (len1.gt.0 .and. len2.le.0) then
    call sic_message(seve%r,'DIFF','Only in first header: '//(secname)//' section')
    sic_diff_presec = .true.
  elseif (len1.le.0 .and. len2.gt.0) then
    call sic_message(seve%r,'DIFF','Only in second header: '//(secname)//' section')
    sic_diff_presec = .true.
  else
    sic_diff_presec = .false.
  endif
  !
end function sic_diff_presec
!
subroutine sic_diff_datadescr(desc1,desc2,error)
  use gbl_message
  use sic_dependencies_interfaces, no_interface=>gag_diff_datar4
  use sic_interfaces, except_this=>sic_diff_datadescr
  use sic_types
  !---------------------------------------------------------------------
  ! @ private
  ! Show the differences between the data of two SIC variables
  !---------------------------------------------------------------------
  type(sic_descriptor_t), intent(in)    :: desc1,desc2
  logical,                intent(inout) :: error
  ! Global
  include 'gbl_memory.inc'
  ! Local
  character(len=*), parameter :: rname='DIFF'
  character(len=*), parameter :: secname='Data descriptor'
  logical :: secwarned
  integer(kind=4) :: idime
  character(len=64) :: elname
  integer(kind=size_length) :: ndata
  integer(kind=address_length) :: ipnt1,ipnt2
  !
  secwarned = .false.
  call gag_diff_elem(rname,secname,secwarned,'TYPE:',desc1%type,desc2%type)
  call gag_diff_elem(rname,secname,secwarned,'NDIM:',desc1%ndim,desc2%ndim)
  do idime=1,min(desc1%ndim,desc2%ndim)
    write(elname,'(A,I0,A)') 'DIM[',idime,']:'
    call gag_diff_elem(rname,secname,secwarned,elname,desc1%dims(idime),desc2%dims(idime))
  enddo
  !
  if (secwarned)  return  ! Only compare same data kind and size
  !
  ipnt1 = gag_pointer(desc1%addr,memory)
  ipnt2 = gag_pointer(desc2%addr,memory)
  ndata = desc_nelem(desc1)
  select case (desc1%type)
  case (fmt_r4)
    call gag_diff_datar4(rname,memory(ipnt1),memory(ipnt2),ndata)
  case default
    call sic_message(seve%w,rname,'Data not compared (only R*4 implemented)')
  end select
  if (error)  return
  !
end subroutine sic_diff_datadescr
