subroutine map_copy_par(in,out)
  use clean_def
  !-----------------------------------------------------------------------
  !
  ! @ private
  !   Copy a MAP structure to another one, but avoid erasing
  !   the number of fields in it. The result must have an intent(inout)
  !   to avoid erasing the allocatable array in the derived type.
  !
  !-----------------------------------------------------------------------
  type(uvmap_par), intent(in) :: in
  type(uvmap_par), intent(inout) :: out
  !
  out%taper = in%taper         ! UV Taper
  out%size = in%size           ! Map size
  out%xycell = in%xycell       ! Map cell
  out%uvcell = in%uvcell       ! UV cell
  out%uniform = in%uniform     ! Weighting parameters
  out%wcol = in%wcol           ! Weighting channel
  out%ctype = in%ctype         ! Convolution mode
  out%support = in%support     ! Support of gridding function
  out%beam = in%beam           ! One beam every N channels
  out%field = in%field         ! Field of view in arcsecond
  out%precis = in%precis       ! Precision at map edges
  out%mode = in%mode           ! Weighting mode
  out%shift = in%shift         ! Shift or rotate UV data
  out%ra_c = in%ra_c           ! Right ascension
  out%dec_c = in%dec_c         ! Declination
  out%angle = in%angle         ! New position angle of UV data
  out%truncate = in%truncate   ! Truncation radius
  !
end subroutine map_copy_par
!
subroutine map_prepare(task,map,error)
  use clean_def
  use clean_default
  use gkernel_interfaces
  use gbl_message
  !
  ! @ private
  !
  character(len=*), intent(in) :: task
  type(uvmap_par), intent(inout) :: map
  logical, intent(out) :: error
  ! character(len=*), parameter :: obsol='obsolete'   ! Later
  character(len=*), parameter :: obsol='obsolescent'
  !
  ! Check version
  if (map_version.lt.-1 .or. map_version.gt.1) then
    call map_message(seve%e,task,'Invalid MAP_VERSION, should be -1,0 or 1')
    error = .true.
    return
  endif
  !
  ! Check old syntax
  if (old_map%uniform(1).ne.save_map%uniform(1)) then
    call map_message(seve%w,task,'UV_CELL is '//obsol//', use MAP_UVCELL instead')
    default_map%uniform(1) = old_map%uniform(1)
  endif
  if (old_map%uniform(2).ne.save_map%uniform(2)) then
    call map_message(seve%w,task,'UV_CELL is '//obsol//', use MAP_ROBUST instead')
    default_map%uniform(2) = old_map%uniform(2)
  endif
  if (old_map%taper(4).ne.save_map%taper(4)) then
    call map_message(seve%w,task,'TAPER_EXPO is '//obsol//', use MAP_TAPER_EXPO instead')
    save_map%taper(4) = old_map%taper(4)
    default_map%taper(4) = old_map%taper(4)
  endif
  if (any(old_map%taper.ne.save_map%taper)) then
    call map_message(seve%w,task,'UV_TAPER is '//obsol//', use MAP_UVTAPER instead')
    default_map%taper = old_map%taper
  endif
  if (old_map%ctype.ne.save_map%ctype) then
    call map_message(seve%w,task,'CONVOLUTION is '//obsol//', use MAP_CONVOLUTION instead')
    default_map%ctype = old_map%ctype
  endif
  if (old_map%mode.ne.save_map%mode) then
    call map_message(seve%w,task,'WEIGHT_MODE is '// &
      & 'obsolete, set MAP_ROBUST=0 instead') 
    ! Attempt to catch the Natural case
    call get_weightmode(task,old_map%mode,error)
    if (old_map%mode.eq.'NATURAL') default_map%uniform(2) = 0.0
    default_map%mode = old_map%mode
  endif
  if (old_map%shift.neqv.save_map%shift) then
    call map_message(seve%w,task,'UV_SHIFT is '//obsol//', use MAP_SHIFT instead')
    default_map%shift = old_map%shift
  endif
  !
  ! Copy the current default to the actual structure
  call map_copy_par(default_map,map)
  !
  ! Weighting scheme
  map%uniform = max(0.0,map%uniform)
  error = .false.
  !
  ! Here, we do not care about the fields issue
  old_map = default_map
  save_map = default_map
end subroutine map_prepare
!
subroutine sub_mosaic(name,error)
  use gkernel_interfaces
  use imager_interfaces, except_this=>sub_mosaic
  use clean_def
  use clean_arrays
  use gbl_message
  !----------------------------------------------------------------------
  ! @ private
  !
  ! MAPPING     MOSAIC ON|OFF
  !             Activates or desactivates the mosaic mode
  !----------------------------------------------------------------------
  character(len=*), intent(in) :: name
  logical, intent(out) :: error
  !
  real(8), parameter :: pi=3.14159265358979323846d0
  !
  integer nf
  logical mosaic, old_mosaic
  real prim_beam
  character(len=message_length) :: mess
  character(len=6) :: rname = 'MOSAIC'
  !
  old_mosaic = user_method%mosaic
  mosaic = name.eq.'ON'
  !
  if (mosaic) then
    if (old_mosaic) then
      call map_message(seve%i,rname,'Already in MOSAIC mode')
    else
      call map_message(seve%i,rname,'Switch to MOSAIC mode')
      call gprompt_set('MOSAIC')
    endif
    if (user_method%trunca.ne.0.) then
      nf = hprim%gil%dim(1)
      ! Should that be replaced by something else ? e.g. in the Telescope Section ?
      prim_beam = hprim%gil%convert(3,4)   ! convention
      write(mess,100) 'Last mosaic loaded: ', nf,' fields'
      call map_message(seve%i,rname,mess)
      write(mess,101) 'Primary beam (arcsec) = ',prim_beam*180*3600/pi
      call map_message(seve%i,rname,mess)
      write(mess,101) 'Truncation level B_MIN = ',user_method%trunca
      call map_message(seve%i,rname,mess)
    else
      call map_message(seve%w,rname,'No mosaic loaded so far')
    endif
    write(mess,101) 'Current value: SEARCH_W = ',user_method%search
    call map_message(seve%i,rname,mess)
    write(mess,101) 'Current value: RESTORE_W = ',user_method%restor
    call map_message(seve%i,rname,mess)
    user_method%mosaic = .true.
  else
    if (.not.old_mosaic) then
      call map_message(seve%i,rname,'Already in NORMAL mode')
    else
      call map_message(seve%i,rname,'Switch to NORMAL mode')
      call gprompt_set('IMAGER')
      user_method%trunca = 0.0
      call sic_delvariable('PRIMARY',.false.,error)
      hprim%gil%dim(1) = 1
    endif
    user_method%mosaic = .false.
  endif
  !
100 format(a,i3,a)
101 format(a,f5.2)
end subroutine sub_mosaic
!
subroutine get_bsize(h,rname,line,bsize,error,otrunc,btrunc)
  use image_def
  use gkernel_interfaces
  use imager_interfaces, except_this => get_bsize
  use gbl_message
  !---------------------------------------------------------------------
  ! @ private-mandatory
  !
  ! MAPPING  Support for command
  !   PRIMARY [BeamSize] [/TRUNCATE Percent]
  !   UV_MAP  [BeamSize] [/TRUNCATE Percent]
  !
  ! Return the primary beam size in radian
  !---------------------------------------------------------------------
  type(gildas), intent(in) :: h          ! UV data header
  character(len=*), intent(in) :: rname  ! Caller's name
  character(len=*), intent(in) :: line   ! Command line
  real(4), intent(out) :: bsize          ! Beam size in radian
  logical, intent(out) :: error          ! Error flag
  integer, optional, intent(in) ::  otrunc         ! Truncation option number
  real(4), optional, intent(out) :: btrunc         ! Truncation level [0,1]
  !
  real(8), parameter :: pi=3.14159265358979323846d0
  character(len=60) :: chain
  integer :: nc
  real :: beamsize
  !
  ! Verify if any Deconvolved data
  !
  if (h%loca%size.eq.0) then
    call map_message(seve%e,rname,'No input data')
    error = .true.
    return
  endif
  !
  beamsize = telescope_beam (rname,h)
  !
  ! NO Backwards compatibility: use command line argument
!  if (sic_present(0,1).and.line.ne.' ') then
!    call sic_r4(line,0,1,bsize,.true.,error)
!    if (error) return
!    bsize = bsize*pi/180/3600
!  else    if (beamsize.eq.0) then
  if (beamsize.eq.0) then
    nc = len_trim(rname)+6
    chain(1:nc) = ' '
    chain(nc:) = 'Use command "SPECIFY TELESCOPE Name" to add one'
    call map_message(seve%e,rname,'No primary beam from data') 
    call map_message(seve%r,rname,chain) 
    error = .true.
    return
  else
    bsize = beamsize
  endif
  !
  if (abs(beamsize-bsize).gt.0.02*bsize) then
    write(chain,'(A,F8.1,A)') 'Specified beam differs from value in data ', &
    & beamsize*180*3600/pi,'"'
    call map_message(seve%w,rname,chain)
  endif
  write(chain,'(A,F8.1,A)') 'Primary beam ',bsize*180*3600/pi,'"'
  call map_message(seve%i,rname,chain)
  !
  ! Truncation
  if (present(btrunc)) then
    call sic_get_real('MAP_TRUNCATE',btrunc,error)
    if (present(otrunc)) call sic_r4(line,otrunc,1,btrunc,.true.,error)
    if (error) return
    btrunc = btrunc*0.01
  else if (present(otrunc)) then
    call map_message(seve%f,rname,'Programming Error: OTRUNC present, but not BTRUNC')
    error = .true.
  endif
  !
end subroutine get_bsize
!
subroutine primary_comm(line,error)
  use gkernel_interfaces
  use imager_interfaces, except_this=>primary_comm
  use clean_def
  use clean_arrays
  use clean_types
  use clean_default
  use gbl_message
  !------------------------------------------------------------------------
  ! @ private
  !
  ! MAPPING  Support for command
  !   PRIMARY [BeamSize] [/TRUNCATE Percent]
  !
  ! Correct for primary beam
  !------------------------------------------------------------------------
  character(len=*), intent(in) :: line ! Command line
  logical, intent(out) :: error
  !
  character(len=*), parameter :: rname='PRIMARY'
  real(8), parameter :: pi=3.14159265358979323846d0
  integer, parameter :: o_trunc=1
  !
  real, allocatable :: dflat(:,:)
  integer :: i, ier, nc
  real :: btrunc
  real :: bsize
  character(len=80) :: mess
  !
  ! Verify if any Deconvolved data and get Beam size
  !
  if (sic_present(o_trunc,0)) then
    call get_bsize(hclean,rname,line,bsize,error,OTRUNC=o_trunc,BTRUNC=btrunc)
  else
    call get_bsize(hclean,rname,line,bsize,error,BTRUNC=btrunc)
  endif
  if (error) return
  !
  write(mess,'(a,f10.2,a,f6.0,a)') 'Correcting for a beam size of ',&
    & bsize/pi*180*3600,'" down to ',100*btrunc,'% '
  call map_message(seve%i,rname,mess)
  !
  if (allocated(dsky)) then
    if (any(hsky%gil%dim.ne.hclean%gil%dim)) then
      Print *,'Programming error - Sky & Clean mismatch'
      deallocate(dsky)
    else
      call map_message(seve%i,rname,'Re-using sky memory')
      call sic_delvariable('SKY',.false.,error)
    endif
  endif
  !
  if (.not.allocated(dsky)) then
    allocate (dsky(hclean%gil%dim(1), hclean%gil%dim(2), hclean%gil%dim(3)), stat=ier)
    if (ier.ne.0) then
      call map_message(seve%e,rname,'Memory allocation error')
      error = .true.
      return
    endif
  endif
  !
  call gildas_null(hsky)
  call gdf_copy_header(hclean,hsky,error)
  !
  allocate (dflat(hclean%gil%dim(1),hclean%gil%dim(2)),stat=ier)
  if (ier.ne.0) then
    call map_message(seve%e,rname,'Memory allocation error')
    error = .true.
    return
  endif
  !
  call mos_primary (hsky,dflat,bsize)
  where (dflat.lt.btrunc)
    dflat = 0
  else where
    dflat = 1.0/dflat
  end where
  !
  do i=1,hclean%gil%dim(3)
    dsky(:,:,i) = dclean(:,:,i) * dflat
  enddo
  hsky%loca%addr = locwrd(dsky)
  call gdf_get_extrema(hsky,error)
  hclean%gil%extr_words = def_extr_words
  !
  call sic_def_real ('SKY',dsky,hsky%gil%ndim,hsky%gil%dim,.true.,error)
  save_data(code_save_sky) = .true.
end subroutine primary_comm
!
subroutine mosaic_uvmap(task,line,error)
  use gkernel_interfaces
  use imager_interfaces, except_this=>mosaic_uvmap
  use clean_def
  use clean_arrays
  use clean_types
  use clean_default
  use gbl_message
  !------------------------------------------------------------------------
  ! @ private
  !
  ! TASK  Compute a Mosaic from a CLIC UV Table with
  ! pointing offset information.
  !
  ! Input :
  !     a UV table with pointing offset information
  !
  ! Ouput
  !   NX NY are the image sizes
  !   NC is the number of channels
  !   NF is the number of different frequencies
  !   NP is the number of pointing centers
  !
  ! 'NAME'.LMV  a 3-d cube containing the uniform noise
  !     combined mosaic, i.e. the sum of the product
  !     of the fields by the primary beam. (NX,NY,NC)
  ! 'NAME'.LOBE the primary beams pseudo-cube (NP,NX,NY,NB)
  ! 'NAME'.WEIGHT the sum of the square of the primary beams (NX,NY,NB)
  ! 'NAME'.BEAM a 4-d cube where each cube contains the synthesised
  !     beam for one field (NX,NY,NB,NP)
  !
  ! All images have the same X,Y sizes
  !------------------------------------------------------------------------
  character(len=*), intent(in) :: task ! Caller (MOSAIC)
  character(len=*), intent(in) :: line ! Command line
  logical, intent(out) :: error
  !
  real(8), parameter :: pi=3.14159265358979323846d0
  real(8), parameter :: f_to_k = 2.d0*pi/299792458.d-6
  character(len=*), parameter :: rname='UV_MOSAIC'
  !
  real, allocatable :: w_mapu(:), w_mapv(:), w_grid(:,:)
  real, allocatable :: res_uv(:,:)
  real(8) new(3)
  real(4) rmega,uvmax,uvmin,uvma
  integer wcol,mcol(2),nfft,sblock
  integer n,ier
  logical one, sorted, shift
  character(len=24) ra_c,dec_c
  character(len=message_length) :: chain
  real cpu0, cpu1
  real(8) :: freq
  real, allocatable :: fft(:)
  real, allocatable :: noises(:)
  integer nx,ny,nu,nv,nc,np,mp
  !
  logical limits
  logical debug
  real ylimn,ylimp
  integer ipen
  !
  real :: thre, btrunc, bsize ! To be initialized
  real :: beamx, beamy
  !
  real, allocatable, target :: dmap(:,:,:), dtmp(:,:,:,:)
  real, allocatable :: dweight(:,:)
  real, allocatable :: dtrunc(:,:)
  real, allocatable :: doff(:,:)
  real, pointer :: my_dirty(:,:,:)
  !
  integer, allocatable :: voff(:)
  real, allocatable :: factorx(:)
  real :: offx, offy, factory, xm, xp, off_range(2)
  integer :: ifield, jfield, ic, j, fstart, fend
  integer :: ib, nb, old_ib
  integer, parameter :: o_trunc=1
  integer, parameter :: o_field=2
  type(projection_t) :: proj
  real(8) :: pos(2)
  integer(kind=index_length) :: dim(4)
  !
  debug = sic_present(0,2) 
  !
  ! Get beam size from data or command line
  if (sic_present(o_trunc,0)) then
    call get_bsize(huv,rname,line,bsize,error,OTRUNC=o_trunc,BTRUNC=btrunc)
  else
    call get_bsize(huv,rname,line,bsize,error,BTRUNC=btrunc)
  endif
  if (error) return
  write(chain,'(a,f10.2,a,f6.0,a)') 'Correcting for a beam size of ',&
    & bsize/pi*180*3600,'" down to ',100*btrunc,'% '
  call map_message(seve%i,rname,chain)
  !
  call map_prepare(task,themap,error)
  if (error) return
  !
  one = .true.
  call sic_get_inte('WCOL',wcol,error)
  call sic_get_inte('MCOL[1]',mcol(1),error)
  call sic_get_inte('MCOL[2]',mcol(2),error)
  !
  call map_center(line,task,huv,shift,new,error)
  if (error) return
  call gag_cpu(cpu0)
  !
  ! Note: the sorting should have FIELD ID as primary (slowest varying) key
  ! It needs a mandatory interface for Doffx & Doffy
  ! It may be better to have a separate counting of NP before, as
  ! such a thing would anyway be needed by UV_STAT
  !
  call mosaic_sort (error,sorted,shift,new,uvmax,uvmin, &
    & huv%gil%column_pointer(code_uvt_xoff), &
    & huv%gil%column_pointer(code_uvt_yoff), &
    & mp,doff,voff)
  if (error) return
  !
  xm = minval(doff(1,1:mp))
  xp = maxval(doff(1,1:mp))
  off_range(1) = xp-xm
  xm = minval(doff(2,1:mp))
  xp = maxval(doff(2,1:mp))
  off_range(2) = xp-xm
  !
  if (.not.sorted) then
    !! Print *,'Done mosaic_sort UV range ',uvmin,uvmax,' sorted ',sorted
    !! call uv_dump_buffers('UV_MOSAIC')
    ! Redefine SIC variables (mandatory)
    call sic_delvariable ('UV',.false.,error)
    call sic_mapgildas ('UV',huv,error,duv) 
  else
    !! Print *,'Mosaic was sorted ',uvmin,uvmax,' sorted ',sorted
  endif
  call gag_cpu(cpu1)
  write(chain,102) 'Finished sorting ',cpu1-cpu0
  call map_message(seve%i,task,chain)
  !
  ! This may need to be revised 
  call map_parameters(task,themap,freq,uvmax,uvmin,error)
  if (error) return
  uvma = uvmax/(freq*f_to_k)
  !
  themap%xycell = themap%xycell*pi/180.0/3600.0
  !
  ! Get work space, ideally before mapping first image, for
  ! memory contiguity reasons.
  !
  nx = themap%size(1)
  ny = themap%size(2)
  nu = huv%gil%dim(1)
  nv = huv%gil%dim(2)
  nc = huv%gil%nchan
  !
  allocate(w_mapu(nx),w_mapv(ny),w_grid(nx,ny),stat=ier)
  if (ier.ne.0) then
    call map_message(seve%e,task,'Gridding allocation error')
    goto 98
  endif
  !
  do_weig = .true.
  if (do_weig) then
    call map_message(seve%i,task,'Computing weights ')
    if (allocated(g_weight)) deallocate(g_weight)
    if (allocated(g_v)) deallocate(g_v)
    allocate(g_weight(nv),g_v(nv),stat=ier)
    if (ier.ne.0) then
      call map_message(seve%e,task,'Weight allocation error')
      goto 98
    endif
  else
    call map_message(seve%i,task,'Re-using weights')
  endif
  nfft = 2*max(nx,ny)
  allocate(fft(nfft),stat=ier)
  if (ier.ne.0) then
    call map_message(seve%e,task,'FFT allocation error')
    goto 98
  endif
  !
  rmega = 8.0
  ier = sic_ramlog('SPACE_MAPPING',rmega)
  sblock = max(int(256.0*rmega*1024.0)/(nx*ny),1)
  !
  ! New Beam place
  if (allocated(dbeam)) then
    call sic_delvariable ('BEAM',.false.,error)
    deallocate(dbeam)
  endif
  call gildas_null(hbeam)
  !
  ! New dirty image
  if (task.ne.'MOSAIC_RESTORE') then 
    if (allocated(ddirty)) then
      call sic_delvariable ('DIRTY',.false.,error)
      deallocate(ddirty)
    endif
    allocate(ddirty(nx,ny,nc),dmap(nx,ny,nc),dtrunc(nx,ny),stat=ier)
    my_dirty => ddirty
  else
    if (allocated(dresid)) then
      call sic_delvariable ('RESIDUAL',.false.,error)
      deallocate(dresid)
    endif
    allocate(dresid(nx,ny,nc),dmap(nx,ny,nc),dtrunc(nx,ny),stat=ier)
    my_dirty => dresid
  endif
  if (ier.ne.0) then
    call map_message(seve%e,task,'Map allocation error')
    goto 98
  endif
  !
  call gildas_null(hdirty)
  hdirty%gil%ndim = 3
  hdirty%gil%dim(1:3) = (/nx,ny,nc/)
  !
  ! Compute the primary beams and weight image
  call gildas_null(hprim)
  if (allocated(dprim)) then
    call sic_delvariable ('PRIMARY',.false.,error)
    deallocate(dprim)
  endif
  if (allocated(dfields)) then
    deallocate(dfields)
  endif
  !
  ! Find out how many beams are required
  call map_beams(task,themap%beam,huv,nx,ny,nb,nc)
  !
  !!Print *,'Done MAP_BEAMS ',themap%beam,nb
  !
  call sic_delvariable('FIELDS%N_SELECT',.false.,error)
  call sic_delvariable('FIELDS%SELECTED',.false.,error)
  error = .false.
  !
  ! Get the field lists from the /FIELD option if any
  if (sic_present(o_field,0)) then
    call select_fields(rname,line,o_field,mp,np,error)
    if (error) return
  else
    np = mp
    allocate(selected_fields(mp),stat=ier)
    do jfield=1,np
      selected_fields(jfield) = jfield
    enddo
    selected_fieldsize = 0    ! Means all
  endif
  !
  ! An issue is that the map characteristics are defined
  ! in "one_beam". We need to call the proper routines before...
  call map_headers (rname,themap,huv,hbeam,hdirty,hprim,nb,np,mcol)
  !
  !!Print *,'Done MAP_HEADERS '
  !
  ! Define the projection about the Phase center
  call gwcs_projec(huv%gil%a0,huv%gil%d0,-huv%gil%pang,huv%gil%ptyp,proj,error)
  ! POS is here the Offset of the Pointing center relative to the Phase center
  call abs_to_rel (proj,huv%gil%ra,huv%gil%dec,pos(1),pos(2),1)
  !
  if (map_version.lt.0) then
    !
    ! Older MOSAIC code
    call map_message(seve%i,task,'Producing a single beam for all channels')
    !
    hbeam%gil%ndim = 3
    hbeam%gil%dim(1:3)=(/nx,ny,np/)
    hbeam%char%code(3) = 'FIELD'
    hbeam%gil%convert(:,3) = 1.d0
    allocate(dbeam(nx,ny,np,1),stat=ier)
    if (ier.ne.0) then
      call map_message(seve%e,task,'Beam allocation error')
      goto 98
    endif
!    call sic_def_real ('BEAM',dbeam,hbeam%gil%ndim,   &
!         &    hbeam%gil%dim,.true.,error)
    call sic_mapgildas('BEAM',hbeam,error,dbeam)
!
    hbeam%r4d => dbeam
    !
    allocate (factorx(nx), dprim(np,nx,ny,1), dweight(nx,ny), &
      & dtmp(nx,ny,1,1), stat=ier)
    if (ier.ne.0) then
      call map_message(seve%e,task,'Primary beam allocation error')
      goto 98
    endif
    hprim%r4d => dprim
    beamx = hprim%gil%inc(2)/bsize*2.0*sqrt(log(2.0))
    beamy = hprim%gil%inc(3)/bsize*2.0*sqrt(log(2.0))
    !
    my_dirty = 0.0    !! ddirty = 0.0
    !
    ! Create the primary beams, lobe and weight images
    do jfield = 1,np
      ifield = jfield
      if (selected_fieldsize.ne.0) ifield = selected_fields(jfield)    
      !
      ! POS is here the Offset of the Pointing center relative to the Phase center
      ! DOFF is the Offset of the Field relative to the Pointing center
      ! So OFFX/Y is the offset of the Field relative to the Phase center in pixel units
      offx = (doff(1,ifield)+pos(1))/hprim%gil%inc(2)
      offy = (doff(2,ifield)+pos(2))/hprim%gil%inc(3)
      !
      do j=1,nx
        factorx(j) = exp(-((j-hprim%gil%ref(2)-offx)*beamx)**2)
      enddo
      do j=1,ny
        factory = exp(-((j-hprim%gil%ref(3)-offy)*beamy)**2)
        dprim(jfield,:,j,1) = factorx(:) * factory
      enddo
    enddo
    deallocate (factorx)
    !
    ! Loop on fields for imaging
    ! Use Dtmp and Dmap as work arrays for beam and image
    hbeam%r3d => dtmp(:,:,:,1)
    hbeam%gil%dim(3) = 1
    hdirty%r3d => dmap
    call map_message(seve%i,task,'Producing a single beam for all channels')
    do jfield = np,1,-1
      ifield = jfield
      if (selected_fieldsize.ne.0) ifield = selected_fields(jfield)    
      !
      do_weig = .true.
      fstart = voff(ifield)      ! Starting Visibility of field
      fend   = voff(ifield+1)-1  ! Ending Visibility of field
      nv = fend-fstart+1
      !
      ! Process sorted UV Table according to the type of beam produced
      call one_beam (task,themap,   &
         &    huv, hbeam, hdirty,   &
         &    nx,ny,nu,nv, duv(:,fstart:fend),   &
         &    w_mapu, w_mapv, w_grid, g_weight, g_v, do_weig,  &
         &    wcol,mcol,fft,   &
         &    sblock,cpu0,error,uvma)
      !
      ! Add it to the "mosaic dirty" image, by multiplying by
      ! the truncated primary beam
      dtrunc = dprim(jfield,:,:,1)
      where(dtrunc.lt.btrunc) dtrunc = 0
      !
      do ic=1,nc
        my_dirty(:,:,ic) = my_dirty(:,:,ic) + dmap(:,:,ic)*dtrunc(:,:)
      enddo
      !
      ! Save the beam
      dbeam(:,:,jfield,1) = dtmp(:,:,1,1)
    enddo
    !
    ! Reset the good pointers and sizes
    hdirty%r3d => my_dirty
    hbeam%r4d => dbeam
    hbeam%gil%ndim = 3
    hbeam%gil%dim(3) = np
    !
    ! Create the Weight image
    !!Print *,'BMIN ',btrunc
    dweight = 0
    call mos_addsq (nx*ny,np,dweight,dprim)
    thre = btrunc**2
    call mos_inverse (nx*ny,dweight,thre)
    !!Print *,'Done MOS_INVERSE'
    !
  else
    !!Print *,'Using untested code, use MAP_BEAM_STEP = -2 if failed'
    !
    ! Code ready but only tested for one Beam for all Frequencies.
    hbeam%gil%ndim = 4
    hbeam%gil%dim(1:4)=(/nx,ny,nb,np/)
    allocate(dbeam(nx,ny,nb,np),stat=ier)
    !
    allocate (dprim(np,nx,ny,nb), dweight(nx,ny), &
      & dtmp(nx,ny,nb,1), factorx(nx), stat=ier)
    if (ier.ne.0) then
      call map_message(seve%e,task,'Primary beam allocation error')
      goto 98
    endif
    hprim%r4d => dprim
    !
    ! Create the primary beams, lobe and weight images
    do jfield = 1,np
      ifield = jfield
      if (selected_fieldsize.ne.0) ifield = selected_fields(jfield)    
      ! bsize is Frequency dependent TBC
      !
      do ib=1,nb
        beamx = hprim%gil%inc(2)/bsize*2.0*sqrt(log(2.0))
        beamy = hprim%gil%inc(3)/bsize*2.0*sqrt(log(2.0))
        offx = (doff(1,ifield)+pos(1))/hprim%gil%inc(2)
        offy = (doff(2,ifield)+pos(2))/hprim%gil%inc(3)
        !
        do j=1,nx
          factorx(j) = exp(-((j-hprim%gil%ref(2)-offx)*beamx)**2)
        enddo
        do j=1,ny
          factory = exp(-((j-hprim%gil%ref(3)-offy)*beamy)**2)
          dprim(jfield,:,j,ib) = factorx(:) * factory
        enddo
      enddo
    enddo
    deallocate (factorx)
    !
    ! Loop on fields for imaging
    ! Use Dtmp and Dmap as work arrays for beam and image
    hbeam%r3d => dtmp(:,:,:,1)
    hbeam%gil%dim(4) = 1
    !
    ! Y a un os pour le mode parallele
    hdirty%r3d => dmap
    !
    my_dirty = 0
    dbeam = 0
    allocate(noises(np))  ! To remember the noise
    !
    !$OMP PARALLEL DEFAULT(none) &
    !$OMP   & SHARED(np,debug,task,themap,huv,dprim,dbeam,my_dirty) & 
    !$OMP   & FIRSTPRIVATE(hdirty,hbeam) &
    !$OMP   & SHARED(voff, selected_fields, noises) &
    !$OMP   & SHARED(nx,ny,nu,nc,duv,wcol,mcol,sblock,cpu0,uvma,btrunc) &
    !$OMP   & PRIVATE(g_weight,g_v,ifield,jfield) &
    !$OMP   & PRIVATE(fstart,fend,nv,do_weig,error,chain) &
    !$OMP   & PRIVATE(old_ib,ic,ib,dtrunc,dmap,dtmp)
    !
    !$OMP DO
    do jfield = np,1,-1
      ifield = selected_fields(jfield)
      ! Pour le mode parallele
      hdirty%r3d => dmap
      hbeam%r3d => dtmp(:,:,:,1)
      !
      do_weig = .true.
      fstart = voff(ifield)      ! Starting Visibility of field
      fend   = voff(ifield+1)-1  ! Ending Visibility of field
      nv = fend-fstart+1
      if (debug) then
        Print *,'Ifield ',ifield,fstart,fend
        Print *,'Cols ',wcol,mcol
        Print *,'Sizes ',nx,ny,nu,nv,np,nc
        Print *,'Calling many_beams_para with SBLOCK ',sblock
      endif
      call many_beams_para (task,themap, huv, hbeam, hdirty,   &
         &    nx,ny,nu,nv, duv(:,fstart:fend),   &
         &    g_weight, g_v, do_weig,  &
         &    wcol,mcol,sblock,cpu0,error,uvma,ifield)
      noises(jfield) = hdirty%gil%noise  ! Remember the noise
      !
      old_ib = 0
      !
      do ic=1,nc
        ib = beam_for_channel(ic,hdirty,hbeam)
        if (debug) Print *,'Selected beam ',ib
        ! Add it to the "mosaic dirty" image, by multiplying by
        ! the truncated primary beam
        if (ib.ne.old_ib) then
          dtrunc(:,:) = dprim(jfield,:,:,ib)
          if (debug) Print *,'Set DTRUNC ',ib,' # ',old_ib
          where (dtrunc.lt.btrunc) dtrunc = 0
          old_ib = ib
        endif
        !$OMP CRITICAL
        my_dirty(:,:,ic) = my_dirty(:,:,ic) + dmap(:,:,ic)*dtrunc(:,:)
        !$OMP END CRITICAL
      enddo
      !
      ! Save the beam - Transposition could be done here if needed
      dbeam(:,:,:,jfield) = dtmp(:,:,:,1)
    enddo
    !$OMP END DO
    !$OMP END PARALLEL
    hdirty%gil%noise = sum(noises)/np
    !
    hbeam%r4d => dbeam
    !
    ! What about the weights here ?
    if (nb.eq.1) then
      dweight = 0
      call mos_addsq (nx*ny,np,dweight,dprim)
      thre = btrunc**2
      call mos_inverse (nx*ny,dweight,thre)
      if (debug) Print *,'Done MOS_INVERSE'
      !
      ! Repack the Beam to 3-Dimensions only
      hbeam%gil%dim(1:4) = [nx,ny,np,1]
      hbeam%gil%convert(:,3) = 1.d0
      hbeam%char%code(3) = 'FIELD'
      hbeam%gil%ndim = 3
    else
      call map_message(seve%e,task,'More than one beam per field - Weights not computed')
    endif
    !
    call sic_mapgildas('BEAM',hbeam,error,dbeam)
  endif
  !
  ! OK we are done (apart from details like Extrema)
  if (.not.error) then
    call message_colour(2)
    call map_message(seve%i,task,'Successful completion')
    call message_colour(-1)
  endif
  !
  hprim%gil%inc(1) = btrunc  ! Convention to store the truncation level
  call sic_mapgildas('PRIMARY',hprim,error,dprim)
  !
  ! Reset the Dirty pointer
  hdirty%r3d => my_dirty
  ! Correct the noise for the approximate gain at mosaic center
  ! for HWHM hexagonal spacing (normally it is sqrt(1+6/4)) 
  hdirty%gil%noise = hdirty%gil%noise/1.5
  if (task.ne.'MOSAIC_RESTORE') then 
    call sic_mapgildas('DIRTY',hdirty,error,ddirty)
    !
    save_data(code_save_beam) = .true.
    save_data(code_save_dirty) = .true.
    save_data(code_save_primary) = .true.
    save_data(code_save_fields) = .true.
    !
    call new_dirty_beam
    !
    ! Define Min Max
    call map_minmax(hdirty)
    !
    d_max = hdirty%gil%rmax
    if (hdirty%gil%rmin.eq.0) then
      d_min = -0.03*hdirty%gil%rmax
    else
      d_min = hdirty%gil%rmin
    endif
  else
    ! Restore the DIRTY image pointer
    hdirty%r3d => ddirty
    call map_minmax(hdirty)
    ! And define the RESIDUAL
    call gdf_copy_header(hdirty,hresid,error)
    hresid%r3d => dresid
    call map_minmax(hresid)
    call sic_mapgildas('RESIDUAL',hresid,error,dresid)
  endif
  !
  error = .false.
  !
  ! Backward compatibility with previous methods
  user_method%trunca = btrunc     ! By convention
  hprim%gil%convert(3,4) = bsize  ! Primary beam size convention
  call sub_mosaic('ON',error)
  !
99 continue
  if (allocated(w_mapu)) deallocate(w_mapu)
  if (allocated(w_mapv)) deallocate(w_mapv)
  if (allocated(w_grid)) deallocate(w_grid)
  if (allocated(fft)) deallocate(fft)
  return
  !
98 call map_message(seve%e,task,'Memory allocation failure')
  error = .true.
  return
  !
102 format(a,f9.2)
end subroutine mosaic_uvmap
!
subroutine map_headers (rname,map,huv,hbeam,hdirty,hprim,nb,nf,mcol)
  use gkernel_interfaces
  use imager_interfaces, except_this=>map_headers
  use clean_def
  use image_def
  use gbl_message
  !------------------------------------------------------------------------
  ! @ private
  !
  ! MAPPING
  !   Define the image headers
  !------------------------------------------------------------------------
  character(len=*), intent(in) :: rname   ! Calling Task name
  type (uvmap_par), intent(inout) :: map  ! Mapping parameters
  type (gildas), intent(inout) :: huv     ! UV data set
  type (gildas), intent(inout) :: hbeam   ! Dirty beam data set
  type (gildas), intent(inout) :: hdirty  ! Dirty image data set
  type (gildas), intent(inout) :: hprim   ! Primary beam data set
  integer, intent(in) :: nb   ! Number of beams per field
  integer, intent(in) :: nf   ! Number of fields
  integer, intent(inout) :: mcol(2)  ! First and last channel
  ! Global variables:
  !
  real(kind=8), parameter :: clight=299792458d-6 ! Frequency in  MHz
  !
  type(gildas) :: htmp
  integer :: nx   ! X size
  integer :: ny   ! Y size
  integer :: nc   ! Number of channels
  integer :: nd   ! Size of data
  integer icol,lcol,fcol
  real vref,voff,vinc,schunk
  real(kind=4) :: loff,boff
  character(len=message_length) :: chain
  logical :: error
  character(len=4) :: code
  !
  !------------------------------------------------------------------------
  !
  ! Code:
  nx = map%size(1)
  ny = map%size(2)
  !
  vref = huv%gil%ref(1)
  voff = huv%gil%voff
  vinc = huv%gil%vres
  !
  nc = huv%gil%nchan
  if (mcol(1).eq.0) then
    mcol(1) = 1
  else
    mcol(1) = max(1,min(mcol(1),nc))
  endif
  if (mcol(2).eq.0) then
    mcol(2) = nc
  else
    mcol(2) = max(1,min(mcol(2),nc))
  endif
  fcol = min(mcol(1),mcol(2))
  lcol = max(mcol(1),mcol(2))
  nc = lcol-fcol+1
  !
  ! Make beam, not normalized
  call gdf_copy_header(huv,hbeam,error)
  hbeam%gil%dopp = 0.0 ! Nullify the Doppler factor
  !
  ! Is that right ?
  schunk = nc/nb
  !
  hbeam%gil%ndim = 4
  hbeam%gil%dim(1) = nx
  hbeam%gil%dim(2) = ny
  hbeam%gil%dim(3) = nb
  hbeam%gil%dim(4) = nf
  hbeam%gil%convert(1,1) = nx/2+1
  hbeam%gil%convert(1,2) = ny/2+1
  hbeam%gil%convert(2,1) = 0
  hbeam%gil%convert(2,2) = 0
  hbeam%gil%convert(3,1) = -map%xycell(1)  ! Assume EQUATORIAL system
  hbeam%gil%convert(3,2) = map%xycell(2)
  hbeam%gil%convert(1,3) = (2.d0*(vref-fcol)+schunk+1.d0)/2/schunk ! Correct
  hbeam%gil%convert(2,3) = voff
  hbeam%gil%convert(:,4) = 1.d0
  hbeam%gil%proj_words = 0
  hbeam%gil%extr_words = 0
  hbeam%gil%reso_words = 0
  hbeam%gil%uvda_words = 0
  hbeam%gil%type_gdf = code_gdf_image
  !
  hbeam%char%code(1) = 'ANGLE'
  hbeam%char%code(2) = 'ANGLE'
  hbeam%char%code(3) = 'VELOCITY'
  hbeam%char%code(4) = 'FIELD'
  hbeam%gil%majo = 0.0
  hbeam%loca%size = nx*ny*nb*nf
  !
  ! Prepare the dirty map header
  call gdf_copy_header(hbeam,hdirty,error)
  hdirty%gil%ndim = 3
  hdirty%gil%dim(1) = nx
  hdirty%gil%dim(2) = ny
  hdirty%gil%dim(3) = nc
  hdirty%gil%dim(4) = 1
  hdirty%gil%convert(1,3) = vref-fcol+1
  hdirty%gil%convert(2,3) = voff
  hdirty%gil%convert(3,3) = vinc
  hdirty%gil%proj_words = def_proj_words
  hdirty%gil%uvda_words = 0
  hdirty%gil%type_gdf = code_gdf_image
  hdirty%char%code(1) = 'RA'
  hdirty%char%code(2) = 'DEC'
  hdirty%char%code(3) = 'VELOCITY'
  call equ_to_gal(hdirty%gil%ra,hdirty%gil%dec,0.0,0.0,   &
                  hdirty%gil%epoc,hdirty%gil%lii,hdirty%gil%bii,loff,boff,error)
  if (huv%gil%ptyp.eq.p_none) then
    hdirty%gil%ptyp = p_azimuthal  ! Azimuthal (Sin)
    hdirty%gil%pang = 0.d0     ! Defined in table.
    hdirty%gil%a0 = hdirty%gil%ra
    hdirty%gil%d0 = hdirty%gil%dec
  else
    hdirty%gil%ptyp = p_azimuthal
    hdirty%gil%pang = huv%gil%pang ! Defined in table.
    hdirty%gil%a0 = huv%gil%a0
    hdirty%gil%d0 = huv%gil%d0
  endif
  hdirty%char%syst = 'EQUATORIAL'
  hdirty%gil%xaxi = 1
  hdirty%gil%yaxi = 2
  hdirty%gil%faxi = 3
  hdirty%gil%extr_words = 0          ! extrema not computed
  hdirty%gil%reso_words = 0          ! no beam defined
  hdirty%gil%nois_words = 2
  hdirty%gil%majo = 0
  hdirty%char%unit = 'Jy/beam'
  hdirty%loca%size = nx*ny*nc
  !
  call gildas_null(hprim)
  if (nf.ge.1) then
    call gildas_null(htmp)
    ! Prepare the primary beam cube header
    call gdf_copy_header(hdirty,htmp,error)
    htmp%gil%dim(4) = nf
    htmp%gil%convert(:,4) = 1.d0
    htmp%char%unit = ' '
    htmp%char%code(4) = 'FIELD'
    ! Also reset the Number of Beams in Frequency
    htmp%gil%dim(3) = nb
    code = '4123'
    call gdf_transpose_header(htmp,hprim,code,error)    
  endif
end subroutine map_headers
!!
subroutine mosaic_sort (error,sorted,shift,new,uvmax,uvmin, &
  & ixoff,iyoff,nf,doff,voff)
  use gkernel_interfaces
  use imager_interfaces, except_this=>mosaic_sort
  use clean_def
  use clean_arrays
  use gbl_message
  !----------------------------------------------------------------------
  ! @ private-mandatory
  !
  ! MAPPING
  !   Sort the input UV table
  !----------------------------------------------------------------------
  logical, intent(inout) :: sorted      ! Is table sorted ?
  logical, intent(inout) :: shift       ! Do we shift phase center ?
  logical, intent(out) :: error
  real(kind=8), intent(inout) :: new(3) ! New phase center and PA
  real, intent(out) :: uvmin            ! Min baseline
  real, intent(out) :: uvmax            ! Max baseline
  integer, intent(in) :: ixoff, iyoff   ! Offset pointers
  integer, intent(out) :: nf            ! Number of fields
  real, intent(out), allocatable :: doff(:,:)  ! Field offsets
  integer, allocatable :: voff(:)       ! Field visibility offsets
  !
  ! Global variables:
  real(8), parameter :: pi=3.14159265358979323846d0
  real(8), parameter :: f_to_k = 2.d0*pi/299792458.d-6
  real(kind=8) freq, off(3)
  real pos(2), cs(2)
  integer nu,nv,ier
  real, pointer :: duv_previous(:,:), duv_next(:,:)
  !
  ! The UV table is available in HUV%
  if (huv%loca%size.eq.0) then
    call map_message(seve%e,'UV_MAP','No UV data loaded')
    error = .true.
    return
  endif
  nu = huv%gil%dim(1)
  nv = huv%gil%dim(2)
  !
  ! Correct for new phase center if required
  if (shift) then
    if (huv%gil%ptyp.eq.p_none) then
      call map_message(seve%w,'SHIFT','No previous phase center info')
      huv%gil%a0 = huv%gil%ra
      huv%gil%d0 = huv%gil%dec
      huv%gil%pang = 0.d0
      huv%gil%ptyp = p_azimuthal
    elseif (huv%gil%ptyp.ne.p_azimuthal) then
      call map_message(seve%w,'SHIFT','Previous projection type not SIN')
      huv%gil%ptyp = p_azimuthal
    endif
    call uv_shift (new,huv%gil%a0,huv%gil%d0,huv%gil%pang,   &
        &      off,shift)
    huv%gil%posi_words = def_posi_words
    huv%gil%proj_words = def_proj_words
  endif
  !
  sorted = .false.
  if (.not.shift) then
    call check_order_mosaic (duv,nu,nv,ixoff,iyoff,sorted)
  endif
  !
  ! Get center frequency
  freq = gdf_uv_frequency(huv,huv%gil%ref(1))
  !
  if (sorted) then
    !
    ! If already sorted, use it
    call map_message(seve%i,'UV_MOSAIC','UV table is already sorted')
    !
    ! Load Field coordinates and compute UVMAX
    call mosaic_loadfield (duv,nu,nv,ixoff,iyoff,nf,doff,voff,uvmax,uvmin)
  else
    !
    ! Else, create another copy
    call map_message(seve%i,'UV_MOSAIC','Sorting UV table...')
    !
    ! Compute observing frequency, and new phase center in wavelengths
    if (shift) then
      huv%gil%a0 = new(1)
      huv%gil%d0 = new(2)
      huv%gil%pang = new(3)
      cs(1)  =  cos(off(3))
      cs(2)  = -sin(off(3))
      ! Note that the new phase center is counter-rotated because rotations
      ! are applied before phase shift.
      pos(1) = - freq * f_to_k * ( off(1)*cs(1) - off(2)*cs(2) )
      pos(2) = - freq * f_to_k * ( off(2)*cs(1) + off(1)*cs(2) )
    else
      pos(1) = 0.0
      pos(2) = 0.0
      cs(1) = 1.0
      cs(2) = 0.0
    endif
    !
    ! OK, rotate, shift, sort and copy...
    !
    nullify (duv_previous, duv_next)
    !
    call uv_find_buffers ('UV_MOSAIC',nu,nv,duv_previous,duv_next,error)
    if (error) return
    !! call uv_dump_buffers ('UV_MOSAIC - After Find')
    !
    call mosaic_sortuv (nu,nv,huv%gil%ntrail,duv_previous,duv_next,   &
           &        pos,cs,uvmax,uvmin,error,ixoff,iyoff,nf,doff,voff)
    call uv_clean_buffers (duv_previous, duv_next, error)
    if (error) return
    !! call uv_dump_buffers ('UV_MOSAIC- After Clean')
  endif
  !
  ! Now transform UVMAX in kiloWavelength (including 2 pi factor)
  uvmax = uvmax*freq*f_to_k
  uvmin = uvmin*freq*f_to_k
  error = .false.
end subroutine mosaic_sort
!
subroutine mosaic_sortuv (np,nv,ntrail,vin,vout,xy,cs,uvmax,uvmin, &
  & error,ixoff,iyoff,nf,doff,voff)
  use gildas_def
  use gkernel_interfaces
  use imager_interfaces, except_this=> mosaic_sortuv
  !---------------------------------------------------------------------
  ! @ public-mandatory
  !
  ! MAPPING
  !     Sort a UV table by fields
  !     Rotate, Shift and Sort a UV table for map making
  !     Differential precession should have been applied before.
  !---------------------------------------------------------------------
  integer, intent(in) :: np        ! Size of a visibility
  integer, intent(in) :: nv        ! Number of visibilities
  integer, intent(in) :: ntrail    ! Number of trailing daps
  real, intent(in) :: vin(np,nv)   ! Input visibilities
  real, intent(out) :: vout(np,nv) ! Output visibilities
  real, intent(in) :: xy(2)        ! Phase shift
  real, intent(in) :: cs(2)        ! Frame Rotation
  real, intent(out) :: uvmax       ! Max UV value
  real, intent(out) :: uvmin       ! Min UV value
  integer, intent(in) :: ixoff, iyoff
  integer, intent(out) :: nf
  real, intent(out), allocatable :: doff(:,:)
  integer, intent(out), allocatable :: voff(:)
  logical, intent(out) :: error    !
  ! Local
  real(8), parameter :: pi=3.14159265358979323846d0
  logical, allocatable :: ips(:)       ! Sign of visibility
  real, allocatable :: rpu(:), rpv(:)  ! U,V coordinates
  real, allocatable :: spv(:)          ! Sorted V coordinates
  integer, allocatable :: ipi(:)       ! Index
  real(8), allocatable :: dtr(:)       ! Sorting number
  logical :: sorted
  integer :: ier, ifi, iv
  !
  ! Load U,V coordinates, applying possible rotation (CS),
  ! and making all V negative
  allocate (ips(nv),rpu(nv),rpv(nv),ipi(nv),dtr(nv),stat=ier)
  if (ier.ne.0) then
    error = .true.
    return
  endif
  call loaduv (vin,np,nv,cs,rpu,rpv,ips,uvmax,uvmin)
  !!Print *,'UVMIN ',uvmin,' UVMAX ',uvmax,' NP ',np,' NV ',nv
  !
  ! Modify the uv coordinates to minimize
  ! the projection errors ... See Sault et al 1996 Appendix 1
  ! Key question here
  ! - modification must be done before sorting
  ! - but should we use the modified or intrinsic UV coordinates ?
  !
  ! For the rotation above, it does not matter, actually: the
  ! matrix commutes (I think so - That can be check later...)
  !
  !  call remapuv(nv,cs,rpu,rpv,ixoff,iyoff,uvmax,uvmin)
  !
  ! Identify number of fields
  call loadfiuv (vin,np,nv,dtr,ipi,sorted,ixoff,iyoff,rpv,nf,doff)
  !
  ! Sort by fields (major number) then V (fractionary part)
  if (.not.sorted) then
    !!Print *,'Sorting UV data '
    call gr8_trie (dtr,ipi,nv,error)
    deallocate (dtr,stat=ier)
    allocate (spv(nv),stat=ier)
    !
    ! One must sort RPV here to use SORTUV later...
    do iv=1,nv
      spv(iv) = rpv(ipi(iv))
    enddo
    rpv(:) = spv(:)
    deallocate (spv,stat=ier)
  else
    deallocate (dtr,stat=ier)
    !!Print *,'UV Data is already sorted '
  endif
  !!Read(5,*) ifi
  !
  ! Apply phase shift and copy to output visibilities
  call sortuv (vin,vout,np,nv,ntrail,xy,rpu,rpv,ips,ipi)
  !
  !
  ! Compute start & end of fields
  allocate(voff(nf+1),stat=ier)
  ifi = 1
  voff(ifi) = 1
  do iv=1,nv
    if ( (doff(1,ifi).ne.vout(ixoff,iv)) .or. &
      &  (doff(2,ifi).ne.vout(iyoff,iv)) ) then
      ifi = ifi+1
      voff(ifi) = iv
    endif
  enddo
  voff(nf+1) = nv+1
  !
  !!Print *,'XOFF ',ixoff,' YOFF ',iyoff
  do ifi=1,min(nf,20)
    write(*,'(I4,A,2F12.4,2I10)') ifi,' DOFF ', &
      & doff(1,ifi)*180.*3600./pi, &
      & doff(2,ifi)*180.*3600./pi, &
      & voff(ifi), voff(ifi+1)-1
  enddo
  !
  error = .false.
  !
end subroutine mosaic_sortuv
!
subroutine check_order_mosaic(visi,np,nv,ixoff,iyoff,sorted)
  !----------------------------------------------------------
  ! @ private
  !
  ! MAPPING
  !   Check if visibilites are sorted.
  !   Chksuv does a similar job, but using V values and an index.
  !----------------------------------------------------------
  integer, intent(in) :: np       ! Size of a visibility
  integer, intent(in) :: nv       ! Number of visibilities
  real, intent(in) :: visi(np,nv) ! Visibilities
  integer, intent(in) :: ixoff    ! X pointing column
  integer, intent(in) :: iyoff    ! Y pointing column
  logical, intent(out) :: sorted
  !
  real vmax,xoff,yoff
  integer iv
  !
  vmax = visi(2,1)
  xoff = visi(ixoff,1)
  yoff = visi(iyoff,1)
  !
  do iv=2,nv
    if (visi(2,iv).lt.vmax) then
      if (visi(ixoff,iv).eq.xoff .and. visi(iyoff,iv).eq.yoff) then
        !!Print *,'Unsorted V at ',iv,visi(2,iv),vmax
        sorted = .false.
        return
      endif
      ! else, this is a new offset
      xoff = visi(ixoff,iv)
      yoff = visi(iyoff,iv)
    else if (visi(ixoff,iv).eq.xoff .and. visi(iyoff,iv).eq.yoff) then
      ! ok, things progress normally
      continue
    else
      ! Unsorted offset
      !!Print *,'Unsorted Position offset at ',iv
      sorted = .false.
      return
    endif
    vmax = visi(2,iv)
  enddo
  sorted = .true.
end subroutine check_order_mosaic
!
subroutine loadfiuv (visi,np,nv,dtr,it,sorted,ixoff,iyoff,rpv,nf,doff)
  !----------------------------------------------------------------------
  ! @ public-mandatory
  !
  ! MAP    UVSORT routines
  !     Load field numbers into work arrays for sorting.
  !----------------------------------------------------------------------
  integer, intent(in)  :: np                   ! Size of a visibility
  integer, intent(in)  :: nv                   ! Number of visibilities
  real, intent(in) :: visi(np,nv)              ! Input visibilities
  real(8), intent(out) :: dtr(nv)              ! Output field number
  integer, intent(out) :: it(nv)               ! Indexes
  logical, intent(out) :: sorted               !
  integer, intent(in)  :: ixoff                ! X pointer
  integer, intent(in)  :: iyoff                ! Y pointer
  real(4), intent(in)  :: rpv(nv)              ! V Values
  integer, intent(out) :: nf                   ! Number of fields
  real(kind=4), intent(out), allocatable :: doff(:,:)  ! Fields offsets
  !
  integer :: iv, itime, iant, jant
  integer :: ifi, mfi, kfi, nfi, ier
  real(8) :: vmax, d0
  real(kind=4), allocatable :: dtmp(:,:)
  !
  ! Scan how many fields
  nfi = 1
  mfi = 100
  !
  ! V are negative values, so this 1 + max(abs(V))
  vmax = 1.0d0-minval(rpv)
  !
  allocate(dtmp(2,mfi),stat=ier)
  dtmp(1,1) = visi(ixoff,1)
  dtmp(2,1) = visi(iyoff,1)
  dtr(1) = 1.d0+rpv(1)/vmax ! We have here 0 =< dtr < 1
  !
  do iv=2,nv
    kfi = 0
    do ifi=1,nfi
      if (visi(ixoff,iv).eq.dtmp(1,ifi) .and. &
      & visi(iyoff,iv).eq.dtmp(2,ifi) ) then
        dtr(iv) = dble(ifi)+rpv(iv)/vmax
        kfi = ifi
        exit
      endif
    enddo
    !
    ! New field
    if (kfi.eq.0) then
      if (nfi.eq.mfi) then
        allocate(doff(2,mfi),stat=ier)
        doff(:,:) = dtmp(:,:)
        deallocate(dtmp)
        allocate(dtmp(2,2*mfi),stat=ier)
        dtmp(:,1:mfi) = doff
        deallocate(doff)
        mfi = 2*mfi
      endif
      nfi = nfi+1
      dtmp(1,nfi) = visi(ixoff,iv)
      dtmp(2,nfi) = visi(iyoff,iv)
      dtr(iv) = dble(nfi)+rpv(iv)/vmax   ! nfi-1 =< dtr < nfi
    endif
  enddo
  !
  nf = nfi
  allocate(doff(2,nfi),stat=ier)
  doff(1:2,:) = dtmp(1:2,1:nfi)
  !
  do iv=1,nv
    it(iv) = iv
  enddo
  !
  ! DTR must in the end be ordered and increasing.
  vmax = dtr(1)
  do iv = 1,nv
    if (dtr(iv).lt.vmax) then
      sorted = .false.
      return
    endif
    vmax = dtr(iv)
  enddo
  sorted = .true.
  !
end subroutine loadfiuv
!
subroutine uvmap_headers(huv,nx,ny,nb,ns,map,mcol,hbeam,hdirty,error)
  use clean_def
  use image_def
  use gkernel_interfaces
  !---------------------------------------------------------------------
  ! IMAGER
  !
  !   Define the Beam and Dirty image headers 
  !---------------------------------------------------------------------
  integer, intent(in) :: nx,ny   ! Map size
  integer, intent(in) :: ns      ! Number of channels per single beam
  integer, intent(in) :: nb      ! Number of beams
  integer, intent(in) :: mcol(2) ! Channel range
  type (uvmap_par), intent(in) :: map    ! Mapping parameters
  type(gildas), intent(in) :: huv        ! UV headers
  type(gildas), intent(inout) :: hbeam   ! Beam headers
  type(gildas), intent(inout) :: hdirty  ! Dirty image header
  logical, intent(inout) :: error
  !
  integer :: nc   ! Number of channels
  integer :: nd   ! Size of data
  integer :: fcol,lcol,wcol, mycols(2)
  real(8) :: vref,voff,vinc
  real(4) :: loff,boff
  integer ier
  !
  nc = huv%gil%nchan
  vref = huv%gil%ref(1)
  voff = huv%gil%voff
  vinc = huv%gil%vres
  !
  wcol = 0
  mycols = mcol
  call uvmap_cols(mycols,nc,fcol,lcol,wcol)
  nc = lcol-fcol+1
  !
  ! Make beam, not normalized
  call gdf_copy_header(huv,hbeam,error)
  hbeam%gil%dopp = 0    ! Nullify the Doppler factor
  !
  hbeam%gil%ndim = 3
  hbeam%gil%dim(1) = nx
  hbeam%gil%dim(2) = ny
  hbeam%gil%dim(3) = nb
  hbeam%gil%dim(4) = 1
  hbeam%gil%convert(1,1) = nx/2+1
  hbeam%gil%convert(1,2) = ny/2+1
  hbeam%gil%convert(2,1) = 0
  hbeam%gil%convert(2,2) = 0
  hbeam%gil%convert(3,1) = -map%xycell(1)  ! Assume EQUATORIAL system
  hbeam%gil%convert(3,2) = map%xycell(2)
!    hbeam%gil%convert(1,3) = vref-fcol+1     ! for 1 per channel
! From UV_COMPRESS
!    uvout%gil%inc(1) = uvout%gil%inc(1)*nc
!    uvout%gil%ref(1) = (2.0*uvout%gil%ref(1)+nc-1.0)/2/nc
!    uvout%gil%vres = nc*uvout%gil%vres
!    uvout%gil%fres = nc*uvout%gil%fres
!
  hbeam%gil%convert(1,3) = (2.d0*(vref-fcol)+ns+1.d0)/2/ns ! Correct
  hbeam%gil%convert(2,3) = voff
  hbeam%gil%extr_words = 0
  hbeam%gil%reso_words = 0
  hbeam%gil%uvda_words = 0
  hbeam%gil%type_gdf = code_gdf_image
  !
  hbeam%char%code(1) = 'ANGLE'
  hbeam%char%code(2) = 'ANGLE'
  hbeam%char%code(3) = 'VELOCITY'
  hbeam%gil%majo = 0.0
  hbeam%loca%size = nx*ny*nb
  !
  ! Also define the Projection, for a better "show beam"
  hbeam%gil%proj_words = def_proj_words
  hbeam%char%code(1) = 'RA'
  hbeam%char%code(2) = 'DEC'
  hbeam%char%code(3) = 'VELOCITY'
  call equ_to_gal(hbeam%gil%ra,hbeam%gil%dec,0.0,0.0,   &
                  hbeam%gil%epoc,hbeam%gil%lii,hbeam%gil%bii,loff,boff,error)
  if (huv%gil%ptyp.eq.p_none) then
    hbeam%gil%ptyp = p_azimuthal  ! Azimuthal (Sin)
    hbeam%gil%pang = 0.d0     ! Defined in table.
    hbeam%gil%a0 = hbeam%gil%ra
    hbeam%gil%d0 = hbeam%gil%dec
  else
    hbeam%gil%ptyp = p_azimuthal
    hbeam%gil%pang = huv%gil%pang ! Defined in table.
    hbeam%gil%a0 = huv%gil%a0
    hbeam%gil%d0 = huv%gil%d0
  endif
  hbeam%char%syst = 'EQUATORIAL'
  hbeam%gil%xaxi = 1
  hbeam%gil%yaxi = 2
  hbeam%gil%faxi = 3  
  !
  ! Prepare the dirty map header
  call gdf_copy_header(hbeam,hdirty,error)
  hdirty%gil%ndim = 3
  hdirty%gil%dim(1) = nx
  hdirty%gil%dim(2) = ny
  hdirty%gil%dim(3) = nc
  hdirty%gil%dim(4) = 1
  hdirty%gil%convert(1,3) = vref-fcol+1
  hdirty%gil%convert(2,3) = voff
  hdirty%gil%convert(3,3) = vinc
  hdirty%gil%proj_words = def_proj_words
  hdirty%gil%uvda_words = 0
  hdirty%gil%type_gdf = code_gdf_image
  hdirty%char%code(1) = 'RA'
  hdirty%char%code(2) = 'DEC'
  hdirty%char%code(3) = 'VELOCITY'
  call equ_to_gal(hdirty%gil%ra,hdirty%gil%dec,0.0,0.0,   &
                  hdirty%gil%epoc,hdirty%gil%lii,hdirty%gil%bii,loff,boff,error)
  if (huv%gil%ptyp.eq.p_none) then
    hdirty%gil%ptyp = p_azimuthal  ! Azimuthal (Sin)
    hdirty%gil%pang = 0.d0         ! Defined in table.
    hdirty%gil%a0 = hdirty%gil%ra
    hdirty%gil%d0 = hdirty%gil%dec
  else
    hdirty%gil%ptyp = p_azimuthal
    hdirty%gil%pang = huv%gil%pang ! Defined in table.
    hdirty%gil%a0 = huv%gil%a0
    hdirty%gil%d0 = huv%gil%d0
  endif
  hdirty%char%syst = 'EQUATORIAL'
  hdirty%gil%xaxi = 1
  hdirty%gil%yaxi = 2
  hdirty%gil%faxi = 3
  hdirty%gil%extr_words = 0        ! extrema not computed
  hdirty%gil%reso_words = 0        ! no beam defined
  hdirty%gil%nois_words = 2
  hdirty%gil%majo = 0
  hdirty%char%unit = 'Jy/beam'
  hdirty%loca%size = nx*ny*nc
  !
  ! Smooth the beam in Frequency
  hbeam%gil%convert(3,3) = vinc*ns
  hbeam%gil%vres = ns*vinc
  hbeam%gil%fres = ns*hbeam%gil%fres
  !
end subroutine uvmap_headers
!
subroutine uvmap_cols(mcol,nc,fcol,lcol,wcol)
  !---------------------------------------------------------------------
  !
  ! IMAGER
  !   Define the Active columns (First, Last and Weight)
  !---------------------------------------------------------------------
  integer, intent(inout) :: mcol(2) ! Default channel range
  integer, intent(inout) :: nc      ! Resulting number of channels
  integer, intent(inout) :: wcol    ! Weight channel
  integer, intent(out) :: fcol,lcol ! First and last channel
  !
  ! Local
  if (mcol(1).eq.0) then
    mcol(1) = 1
  else
    mcol(1) = max(1,min(mcol(1),nc))
  endif
  if (mcol(2).eq.0) then
    mcol(2) = nc
  else
    mcol(2) = max(1,min(mcol(2),nc))
  endif
  fcol = min(mcol(1),mcol(2))
  lcol = max(mcol(1),mcol(2))
  if (wcol.eq.0) then
    wcol = (fcol+lcol)/3
  endif
  wcol = max(1,wcol)
  wcol = min(wcol,nc)
end subroutine uvmap_cols
!
subroutine select_fields(rname,line,o_field,mp,np,error)
  use gkernel_interfaces
  use imager_interfaces, only : get_i4list_fromsic, map_message
  use clean_arrays
  use gbl_message
  !
  ! @ private
  !   Select a list of fields from a Mosaic
  !
  character(len=*), intent(in) :: rname
  character(len=*), intent(in) :: line
  integer, intent(in) :: o_field
  integer, intent(in) :: mp ! Number of fields in UV data
  integer, intent(out) :: np ! Number of fields selected 
  logical, intent(inout) :: error
  !
  integer :: ifield, jfield
  integer(kind=index_length) :: dim(4)
  character(len=80) chain
  !
  np = 0
  call get_i4list_fromsic(rname,line,o_field,np,selected_fields,error)  
  if (np.gt.mp) then
    call map_message(seve%e,rname,'More selected fields than available')
    error = .true.
  else 
    do jfield=1,np
      ifield = selected_fields(jfield)
      if (ifield.le.0 .or. ifield.gt.mp) then
        write(chain,'(A,I0,I0,A,I0,A)') 'Selected field ',jfield,& 
        & ifield,' out of range [1,',mp,']'
        call map_message(seve%e,rname,chain)
        error = .true.
      endif
    enddo
  endif
  if (error) return
  selected_fieldsize = np
  write(chain,'(I0,A,I0,A)') np,' fields selected:' 
  call map_message(seve%i,rname,chain)
  write(*,'(10X,6(2X,I0))') selected_fields(1:np)
  !
  call sic_def_inte('FIELDS%N_SELECT',selected_fieldsize,0,0,.true.,error)
  dim(1) = selected_fieldsize
  call sic_def_inte('FIELDS%SELECTED',selected_fields,1,dim,.true.,error)    
end subroutine select_fields
