program uv_extract
  use gildas_def
  use gkernel_interfaces
  !---------------------------------------------------------------------
  ! GILDAS
  !	Extract channels from an input multi-channel UV table
  !---------------------------------------------------------------------
  character(len=filename_length) :: uvdata,uvsort
  real(kind=8) :: drange(2)
  logical :: error
  character(len=9) :: ctype
  !
  ! Code:
  call gildas_open
  call gildas_char('UV_INPUT$',uvdata)
  call gildas_char('UV_OUTPUT$',uvsort)
  call gildas_dble('RANGES$',drange,2)
  call gildas_char('CTYPE$',ctype)
  call gildas_close
  error = .false.
  call sub_uv_extract(uvdata,uvsort,drange,ctype,error)
  if (error) call sysexi(fatale)
end program uv_extract
!
subroutine sub_uv_extract(uvdata,uvsort,drange,ctype,error)
  use gildas_def
  use gkernel_interfaces
  use image_def
!  use gbl_format
  use gbl_message
  !---------------------------------------------------------------------
  ! GILDAS
  !	Extract channels from an input multi-channel UV table
  !---------------------------------------------------------------------
  character(len=*), intent(in) :: uvdata,uvsort
  real(kind=8), intent(inout) :: drange(2)
  character(len=*), intent(inout) :: ctype
  logical, intent(out) :: error
  !
  character(len=*), parameter :: rname='UV_EXTRACT'
  ! Local
  type(gildas) :: uvin, uvou
  integer(kind=index_length) :: nvisi, ib
  integer(kind=4) :: nc(2),ier,nn,itype,nblock
  character(len=message_length) :: mess
  integer(kind=4), parameter :: mtype=3
  character(len=9) :: types(mtype),mytype
  data types /'CHANNEL','VELOCITY','FREQUENCY'/
  !
  error = .true.
  if (len_trim(uvdata).eq.0) return 
  if (len_trim(uvsort).eq.0) return
  error = .false.
  !
  ! Input file
  call gildas_null(uvin, type= 'UVT')
  call gdf_read_gildas (uvin, uvdata, '.uvt', error, data=.false.)
  if (error) then
    call map_message(seve%f,rname,'Cannot read input table')
    return
  endif
  !
  call sic_upper(ctype)
  error = .false.
  call sic_ambigs('UV_EXTRACT',ctype,mytype,itype,types,mtype,error)
  if (error)  return 
  !
  if (mytype.eq.'CHANNEL') then
    nc(:) = nint(drange)
  else if (mytype.eq.'VELOCITY') then
    ! drange = nc - uvin%gil%ref(uvin%gil%faxi) ) * uvin%gil%vres + uvin%gil%voff 
    nc(:) = (drange(:) - uvin%gil%voff) / uvin%gil%vres + uvin%gil%ref(uvin%gil%faxi)
  else if (mytype.eq.'FREQUENCY') then
    ! drange = nc - uvin%gil%ref(uvin%gil%faxi) ) * uvin%gil%vres + uvin%gil%voff 
    nc(:) = (drange(:) - uvin%gil%freq) / uvin%gil%fres + uvin%gil%ref(uvin%gil%faxi)
  else
    call map_message(seve%f,rname,'Type of value '''//trim(mytype)//''' not supported')
    error = .true.
    return
  endif
  !
  if (nc(1).gt.nc(2)) then
    nn = nc(2)
    nc(2) = nc(1)
    nc(1) = nn
  endif
  !
  ! Sanity: uv_extract behaviour is to produce the intersection of old range
  ! and new range. Must be an error if there is no overlap!
  if (nc(2).lt.1 .or. nc(1).gt.uvin%gil%nchan) then
    write(mess,'(A,I0,A,I0)')  &
      'Channel range ',nc(1),' to ',nc(2),' does not intersect the original range'
    call map_message(seve%f,rname,mess)
    error = .true.
    return
  endif
  !
  ! Intersection
  ! NB: gdf_range interprets negative channel as channels counted from the end
  ! (e.g. [10:-10] will discard 10 channels one the left and 10 channels on the
  ! right). However, this never worked in uv_extract as the range is sorted as
  ! [-10:10] (just before), which gives misordered resulting range. Assume no
  ! support for negative-from-the-end channels.
  if (nc(1).lt.1)              nc(1) = 1
  if (nc(2).gt.uvin%gil%nchan) nc(2) = uvin%gil%nchan
  ! gdf_range is of no real help...
  ! ier = gdf_range (nc, uvin%gil%nchan) 
  ! if (ier.ne.0) return
  write(mess,'(4(A,I0))')   &
    'Extracting channels ',nc(1),' to ',nc(2),' (from ',uvin%gil%nchan,' original channels)'
  call map_message(seve%i,rname,mess)
  !
  call gildas_null(uvou, type= 'UVT')
  call gdf_copy_header(uvin, uvou, error)
  if (error) return
  call sic_parse_file(uvsort,' ','.uvt',uvou%file)
  !
  uvou%gil%nchan = nc(2)-nc(1)+1
  uvou%gil%ref(1) = uvin%gil%ref(1)-nc(1)+1
  uvou%gil%dim(1) = uvou%gil%nlead + uvou%gil%natom*uvou%gil%nchan + uvou%gil%ntrail
  !
  ! Define blocking factor, on largest data file, usually the input one
  ! but not always...  
  call gdf_nitems('SPACE_GILDAS',nblock,uvin%gil%dim(1)) ! Visibilities at once
  nblock = min(nblock,uvin%gil%dim(2))
  ! Allocate respective space for each file
  allocate (uvin%r2d(uvin%gil%dim(1),nblock), uvou%r2d(uvou%gil%dim(1),nblock), stat=ier)
  if (ier.ne.0) then
    write(mess,*) 'Memory allocation error ',uvin%gil%dim(1), nblock
    call map_message(seve%e,rname,mess)
    error = .true.
    return
  endif
  !
  call gdf_create_image(uvou,error)
  if (error) return
  !
  ! Loop over line table 
  uvou%blc = 0
  uvou%trc = 0
  uvin%blc = 0
  uvin%trc = 0
  do ib = 1,uvou%gil%dim(2),nblock
    write(mess,*) ib,' / ',uvou%gil%dim(2),nblock
    call map_message(seve%d,rname,mess) 
    uvin%blc(2) = ib
    uvin%trc(2) = min(uvin%gil%dim(2),ib-1+nblock) 
    uvou%blc(2) = ib
    uvou%trc(2) = uvin%trc(2)
    call gdf_read_data(uvin,uvin%r2d,error)
    nvisi = uvin%trc(2)-uvin%blc(2)+1
    !
    ! Here do the job
    call sub_extract_block(uvou, uvou%r2d, uvin, uvin%r2d, nvisi, nc) 
    call gdf_write_data(uvou, uvou%r2d, error)
  enddo
  !
  call gdf_close_image(uvou,error)
  call gdf_close_image(uvin,error)
  call map_message(seve%i,rname,'Successful completion')
end subroutine sub_uv_extract
!
subroutine sub_extract_block(out, dout, in, din, nvisi, nc)
  use image_def
  type(gildas), intent(in) :: out
  type(gildas), intent(in) :: in
  integer(kind=size_length) :: nvisi
  real, intent(in) :: din(in%gil%dim(1),nvisi)
  real, intent(out) :: dout(out%gil%dim(1),nvisi)
  integer, intent(in) :: nc(2)
  !
  integer :: iv, icf, icl, jcf, jcl
  !
  icf = in%gil%nlead+(nc(1)-1)*in%gil%natom+1
  icl = in%gil%nlead+nc(2)*in%gil%natom
  jcf = out%gil%nlead+1
  jcl = out%gil%nlead+out%gil%nchan*out%gil%natom
  !
  do iv=1,nvisi
    dout(1:in%gil%nlead,iv) = din(1:in%gil%nlead,iv)
    dout(jcf:jcl,iv) = din(icf:icl,iv)
    if (out%gil%ntrail.gt.0) then
      dout(out%gil%dim(1)-out%gil%ntrail+1:out%gil%dim(1),iv) =   &
      &     din(in%gil%dim(1)-in%gil%ntrail+1:in%gil%dim(1),iv)  
    endif
  enddo
end subroutine sub_extract_block
