/*
 * Decompiled with CFR 0.152.
 */
package ghidra.util.data;

import ghidra.app.plugin.core.datamgr.util.DataTypeUtils;
import ghidra.app.services.DataTypeQueryService;
import ghidra.program.database.data.DataTypeUtilities;
import ghidra.program.database.data.ProgramDataTypeManager;
import ghidra.program.model.data.AbstractStringDataType;
import ghidra.program.model.data.ArrayDataType;
import ghidra.program.model.data.BitFieldDataType;
import ghidra.program.model.data.BuiltInDataTypeManager;
import ghidra.program.model.data.CategoryPath;
import ghidra.program.model.data.DataType;
import ghidra.program.model.data.DataTypeManager;
import ghidra.program.model.data.Dynamic;
import ghidra.program.model.data.FactoryDataType;
import ghidra.program.model.data.InvalidDataTypeException;
import ghidra.program.model.data.PointerDataType;
import ghidra.program.model.data.TypeDef;
import ghidra.util.exception.AssertException;
import ghidra.util.exception.CancelledException;
import java.util.ArrayList;
import java.util.List;
import org.apache.commons.lang3.StringUtils;

public class DataTypeParser {
    private DataTypeManager sourceDataTypeManager;
    private DataTypeManager destinationDataTypeManager;
    private DataTypeQueryService dataTypeManagerService;
    private AllowedDataTypes allowedTypes;

    public DataTypeParser(DataTypeQueryService dataTypeManagerService, AllowedDataTypes allowedTypes) {
        this.dataTypeManagerService = dataTypeManagerService;
        this.allowedTypes = allowedTypes;
    }

    public DataTypeParser(DataTypeManager sourceDataTypeManager, DataTypeManager destinationDataTypeManager, DataTypeQueryService dataTypeManagerService, AllowedDataTypes allowedTypes) {
        this.sourceDataTypeManager = sourceDataTypeManager;
        this.destinationDataTypeManager = destinationDataTypeManager;
        this.dataTypeManagerService = dataTypeManagerService;
        this.allowedTypes = allowedTypes;
    }

    public DataType parse(String dataTypeString) throws InvalidDataTypeException, CancelledException {
        return this.parse(dataTypeString, (CategoryPath)null);
    }

    public DataType parse(String dataTypeString, CategoryPath category) throws InvalidDataTypeException, CancelledException {
        String dataTypeName = DataTypeParser.getBaseString(dataTypeString = dataTypeString.replaceAll("\\s+", " ").trim());
        DataType namedDt = this.getNamedDataType(dataTypeName, category);
        if (namedDt == null) {
            throw new InvalidDataTypeException("Valid data-type not specified");
        }
        return this.parseDataTypeModifiers(namedDt, dataTypeString.substring(dataTypeName.length()));
    }

    public DataType parse(String dataTypeString, DataType suggestedBaseDataType) throws InvalidDataTypeException, CancelledException {
        DataType namedDt;
        String dataTypeName = DataTypeParser.getBaseString(dataTypeString = dataTypeString.replaceAll("\\s+", " ").trim());
        if (StringUtils.isBlank((CharSequence)dataTypeName)) {
            throw new InvalidDataTypeException("missing base data-type name");
        }
        if (suggestedBaseDataType != null && dataTypeName.equals(suggestedBaseDataType.getName())) {
            namedDt = suggestedBaseDataType;
            if (namedDt.getDataTypeManager() != this.destinationDataTypeManager) {
                namedDt = namedDt.clone(this.destinationDataTypeManager);
            }
        } else {
            namedDt = this.getNamedDataType(dataTypeName, null);
            if (namedDt == null) {
                throw new InvalidDataTypeException("valid data-type not specified");
            }
        }
        return this.parseDataTypeModifiers(namedDt, dataTypeString.substring(dataTypeName.length()));
    }

    public static void ensureIsAllowableType(DataType dt, AllowedDataTypes allowedTypes) throws InvalidDataTypeException {
        if (dt instanceof BitFieldDataType) {
            if (allowedTypes != AllowedDataTypes.SIZABLE_DYNAMIC_AND_BITFIELD) {
                throw new InvalidDataTypeException("Bitfield data-type not allowed");
            }
            return;
        }
        switch (allowedTypes) {
            case DYNAMIC: {
                if (!(dt instanceof FactoryDataType)) break;
                throw new InvalidDataTypeException("Factory data-type not allowed");
            }
            case SIZABLE_DYNAMIC: 
            case SIZABLE_DYNAMIC_AND_BITFIELD: {
                if (dt instanceof FactoryDataType) {
                    throw new InvalidDataTypeException("Factory data-type not allowed");
                }
                if (!(dt instanceof Dynamic) || ((Dynamic)dt).canSpecifyLength()) break;
                throw new InvalidDataTypeException("non-sizable data-type not allowed");
            }
            case FIXED_LENGTH: {
                if (dt.getLength() >= 0) break;
                throw new InvalidDataTypeException("Fixed-length data-type required");
            }
            case STRINGS_AND_FIXED_LENGTH: {
                if (dt.getLength() >= 0 || dt instanceof AbstractStringDataType) break;
                throw new InvalidDataTypeException("Fixed-length or string data-type required");
            }
            case BITFIELD_BASE_TYPE: {
                if (BitFieldDataType.isValidBaseDataType((DataType)dt)) break;
                throw new InvalidDataTypeException("Enum or integer derived data-type required");
            }
            case ALL: {
                break;
            }
            default: {
                throw new InvalidDataTypeException("Unknown data type allowance specified: " + allowedTypes);
            }
        }
    }

    private DataType parseDataTypeModifiers(DataType namedDataType, String dataTypeModifiers) throws InvalidDataTypeException {
        List<DtPiece> modifiers = this.parseModifiers(dataTypeModifiers);
        Object dt = namedDataType;
        int elementLength = dt.getLength();
        try {
            for (DtPiece modifier : modifiers) {
                if (modifier instanceof PointerSpecPiece) {
                    int pointerSize = ((PointerSpecPiece)modifier).getPointerSize();
                    dt = new PointerDataType(dt, pointerSize, this.destinationDataTypeManager);
                    elementLength = dt.getLength();
                    continue;
                }
                if (modifier instanceof ElementSizeSpecPiece) {
                    if (elementLength > 0) continue;
                    elementLength = ((ElementSizeSpecPiece)modifier).getElementSize();
                    continue;
                }
                if (modifier instanceof ArraySpecPiece) {
                    int elementCount = ((ArraySpecPiece)modifier).getElementCount();
                    dt = this.createArrayDataType((DataType)dt, elementLength, elementCount);
                    elementLength = dt.getLength();
                    continue;
                }
                if (!(modifier instanceof BitfieldSpecPiece)) continue;
                if (this.allowedTypes != AllowedDataTypes.SIZABLE_DYNAMIC_AND_BITFIELD) {
                    throw new InvalidDataTypeException("Bitfield not permitted");
                }
                if (this.destinationDataTypeManager == null) {
                    throw new AssertException("Bitfields require destination datatype manager to be specified");
                }
                int bitSize = ((BitfieldSpecPiece)modifier).getBitSize();
                dt = new ProxyBitFieldDataType(dt.clone(this.destinationDataTypeManager), bitSize);
            }
        }
        catch (IllegalArgumentException e) {
            throw new InvalidDataTypeException(e.getMessage());
        }
        DataTypeParser.ensureIsAllowableType(dt, this.allowedTypes);
        return dt;
    }

    private List<DtPiece> parseModifiers(String dataTypeModifiers) throws InvalidDataTypeException {
        int arrayStartIndex = -1;
        ArrayList<DtPiece> modifiers = new ArrayList<DtPiece>();
        boolean terminalModifier = false;
        for (String piece : DataTypeParser.splitDataTypeModifiers(dataTypeModifiers)) {
            piece = piece.trim();
            if (terminalModifier) {
                throw new InvalidDataTypeException("Invalid data type modifier");
            }
            if (piece.startsWith("*")) {
                modifiers.add(new PointerSpecPiece(piece));
                arrayStartIndex = -1;
                continue;
            }
            if (piece.startsWith("[")) {
                ArraySpecPiece arraySpec = new ArraySpecPiece(piece);
                if (arrayStartIndex >= 0) {
                    modifiers.add(arrayStartIndex, arraySpec);
                    continue;
                }
                arrayStartIndex = modifiers.size();
                modifiers.add(arraySpec);
                continue;
            }
            if (piece.startsWith(":")) {
                terminalModifier = true;
                modifiers.add(new BitfieldSpecPiece(piece));
                continue;
            }
            if (!piece.startsWith("{")) continue;
            modifiers.add(new ElementSizeSpecPiece(piece));
            arrayStartIndex = -1;
        }
        return modifiers;
    }

    private DataType getNamedDataType(String baseName, CategoryPath category) throws InvalidDataTypeException, CancelledException {
        ArrayList<DataType> results = new ArrayList<DataType>();
        DataType dt = this.findDataType(this.sourceDataTypeManager, baseName, category, results);
        if (dt != null) {
            return dt;
        }
        if (results.isEmpty() && DataType.DEFAULT.getDisplayName().equals(baseName)) {
            dt = DataType.DEFAULT;
        } else if (category == null) {
            dt = this.findDataTypeInAllDataTypeManagers(baseName, results);
        }
        if (dt == null) {
            String msg = "Unrecognized data type of \"" + baseName + "\"";
            throw new InvalidDataTypeException(msg);
        }
        return dt.clone(this.destinationDataTypeManager);
    }

    private DataType findDataTypeInAllDataTypeManagers(String baseName, List<DataType> results) throws CancelledException {
        if (results.isEmpty() && this.dataTypeManagerService != null) {
            results.addAll(DataTypeUtils.getExactMatchingDataTypes(baseName, this.dataTypeManagerService));
        }
        if (results.isEmpty()) {
            return null;
        }
        DataType dt = DataTypeParser.pickFromPossibleEquivalentDataTypes(results);
        if (dt != null) {
            return dt;
        }
        return this.proptUserForType(baseName);
    }

    private DataType proptUserForType(String baseName) throws CancelledException {
        if (this.dataTypeManagerService == null) {
            return null;
        }
        DataType dt = this.dataTypeManagerService.getDataType(baseName);
        if (dt == null) {
            throw new CancelledException();
        }
        return dt;
    }

    private DataType findDataType(DataTypeManager dtm, String baseName, CategoryPath category, List<DataType> list) {
        BuiltInDataTypeManager builtInDTM = BuiltInDataTypeManager.getDataTypeManager();
        if (dtm == null) {
            return this.findDataType((DataTypeManager)builtInDTM, baseName, category, list);
        }
        if (category != null) {
            DataType dt = dtm.getDataType(category, baseName);
            if (dt != null) {
                list.add(dt);
                return dt;
            }
        } else {
            DataType dataType = DataTypeUtilities.getCPrimitiveDataType((String)baseName);
            if (dataType != null) {
                return dataType.clone(dtm);
            }
            dtm.findDataTypes(baseName, list);
            if (list.size() == 1) {
                return list.get(0);
            }
        }
        if (list.isEmpty() && dtm != builtInDTM) {
            return this.findDataType((DataTypeManager)builtInDTM, baseName, category, list);
        }
        return null;
    }

    private static DataType pickFromPossibleEquivalentDataTypes(List<DataType> dtList) {
        DataType programDataType = null;
        for (DataType dataType : dtList) {
            DataTypeManager manager = dataType.getDataTypeManager();
            if (manager instanceof BuiltInDataTypeManager) {
                programDataType = dataType;
                continue;
            }
            if (!(manager instanceof ProgramDataTypeManager)) continue;
            programDataType = dataType;
            break;
        }
        if (programDataType == null) {
            return null;
        }
        for (DataType dataType : dtList) {
            if (programDataType.isEquivalent(dataType)) continue;
            return null;
        }
        return programDataType;
    }

    private static String getBaseString(String dataTypeString) {
        int nextIndex = 0;
        int templateCount = 0;
        while (nextIndex < dataTypeString.length()) {
            char c = dataTypeString.charAt(nextIndex);
            if (c == '<') {
                ++templateCount;
            } else if (c == '>') {
                --templateCount;
            }
            if (templateCount != 0) {
                ++nextIndex;
                continue;
            }
            if (c == '*' || c == '[' || c == ':' || c == '{') {
                return dataTypeString.substring(0, nextIndex).trim();
            }
            ++nextIndex;
        }
        return dataTypeString;
    }

    private static String[] splitDataTypeModifiers(String dataTypeModifiers) {
        int nextIndex;
        if ((dataTypeModifiers = dataTypeModifiers.replaceAll(":[ \\t]", "")).length() == 0) {
            return new String[0];
        }
        ArrayList<String> list = new ArrayList<String>();
        int startIndex = 0;
        for (nextIndex = 1; nextIndex < dataTypeModifiers.length(); ++nextIndex) {
            char c = dataTypeModifiers.charAt(nextIndex);
            if (c != '*' && c != '[' && c != ':' && c != '{') continue;
            list.add(dataTypeModifiers.substring(startIndex, nextIndex));
            startIndex = nextIndex;
        }
        list.add(dataTypeModifiers.substring(startIndex, nextIndex));
        String[] pieces = new String[list.size()];
        list.toArray(pieces);
        return pieces;
    }

    private DataType createArrayDataType(DataType baseDataType, int elementLength, int elementCount) throws InvalidDataTypeException {
        DataType dt = baseDataType;
        if (dt instanceof TypeDef) {
            dt = ((TypeDef)dt).getBaseDataType();
        }
        if (elementLength <= 0) {
            throw new InvalidDataTypeException("Only a datatype with a positive size be used for an array: " + baseDataType.getName() + "; " + elementLength);
        }
        return new ArrayDataType(baseDataType, elementCount, elementLength, this.destinationDataTypeManager);
    }

    private static int parseSize(String size) {
        if (StringUtils.isBlank((CharSequence)size)) {
            throw new NumberFormatException();
        }
        if (StringUtils.startsWithIgnoreCase((CharSequence)(size = size.trim()), (CharSequence)"0x")) {
            return Integer.parseInt(size.substring(2), 16);
        }
        return Integer.parseInt(size);
    }

    public static enum AllowedDataTypes {
        ALL,
        DYNAMIC,
        SIZABLE_DYNAMIC,
        SIZABLE_DYNAMIC_AND_BITFIELD,
        FIXED_LENGTH,
        STRINGS_AND_FIXED_LENGTH,
        BITFIELD_BASE_TYPE;

    }

    private static interface DtPiece {
    }

    private static class PointerSpecPiece
    implements DtPiece {
        int pointerSize = -1;

        PointerSpecPiece(String piece) throws InvalidDataTypeException {
            if (!piece.startsWith("*")) {
                throw new InvalidDataTypeException("Invalid pointer specification: " + piece);
            }
            if (piece.length() == 1) {
                return;
            }
            try {
                this.pointerSize = Integer.parseInt(piece.substring(1));
            }
            catch (NumberFormatException e) {
                throw new InvalidDataTypeException("Invalid pointer specification: " + piece);
            }
            int mod = this.pointerSize % 8;
            this.pointerSize /= 8;
            if (mod != 0 || this.pointerSize <= 0 || this.pointerSize > 8) {
                throw new InvalidDataTypeException("Invalid pointer size: " + piece);
            }
        }

        int getPointerSize() {
            return this.pointerSize;
        }
    }

    private static class ElementSizeSpecPiece
    implements DtPiece {
        int elementSize;

        ElementSizeSpecPiece(String piece) throws InvalidDataTypeException {
            if (piece.startsWith("{") && piece.endsWith("}")) {
                String elementSizeStr = piece.substring(1, piece.length() - 1);
                try {
                    this.elementSize = DataTypeParser.parseSize(elementSizeStr);
                    return;
                }
                catch (NumberFormatException numberFormatException) {
                    // empty catch block
                }
            }
            throw new InvalidDataTypeException("Invalid array element size specification: " + piece);
        }

        int getElementSize() {
            return this.elementSize;
        }
    }

    private static class ArraySpecPiece
    implements DtPiece {
        int elementCount;

        ArraySpecPiece(String piece) throws InvalidDataTypeException {
            if (piece.startsWith("[") && piece.endsWith("]")) {
                try {
                    String elementCountStr = piece.substring(1, piece.length() - 1);
                    this.elementCount = elementCountStr.length() == 0 ? 0 : DataTypeParser.parseSize(elementCountStr);
                    return;
                }
                catch (NumberFormatException numberFormatException) {
                    // empty catch block
                }
            }
            throw new InvalidDataTypeException("Invalid array specification: " + piece);
        }

        int getElementCount() {
            return this.elementCount;
        }
    }

    private static class BitfieldSpecPiece
    implements DtPiece {
        int bitSize;

        BitfieldSpecPiece(String piece) throws InvalidDataTypeException {
            if (piece.startsWith(":")) {
                String bitSizeStr = piece.substring(1);
                try {
                    this.bitSize = DataTypeParser.parseSize(bitSizeStr);
                    if (this.bitSize >= 0) {
                        return;
                    }
                }
                catch (NumberFormatException numberFormatException) {
                    // empty catch block
                }
            }
            throw new InvalidDataTypeException("Invalid bitfield specification: " + piece);
        }

        int getBitSize() {
            return this.bitSize;
        }
    }

    private static class ProxyBitFieldDataType
    extends BitFieldDataType {
        private ProxyBitFieldDataType(DataType baseDataType, int bitSize) throws InvalidDataTypeException {
            super(baseDataType, bitSize);
        }
    }
}

