/* ###
 * IP: GHIDRA
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package agent.dbgeng.manager.cmd;

import agent.dbgeng.dbgeng.DebugClient;
import agent.dbgeng.dbgeng.DebugClient.DebugCreateFlags;
import agent.dbgeng.dbgeng.DebugClient.DebugEngCreateFlags;
import agent.dbgeng.dbgeng.DebugClient.DebugVerifierFlags;
import agent.dbgeng.dbgeng.DebugProcessInfo;
import agent.dbgeng.dbgeng.DebugSystemObjects;
import agent.dbgeng.dbgeng.DebugThreadId;
import agent.dbgeng.dbgeng.DebugThreadInfo;
import agent.dbgeng.manager.DbgEvent;
import agent.dbgeng.manager.DbgProcess;
import agent.dbgeng.manager.DbgThread;
import agent.dbgeng.manager.evt.AbstractDbgCompletedCommandEvent;
import agent.dbgeng.manager.evt.DbgProcessCreatedEvent;
import agent.dbgeng.manager.impl.DbgManagerImpl;
import ghidra.comm.util.BitmaskSet;

/**
 * Implementation of {@link DbgProcess#fileExecAndSymbols(String)}
 */
public class DbgLaunchProcessCommand extends AbstractDbgCommand<DbgThread> {

	private DbgProcessCreatedEvent created = null;
	private boolean completed = false;
	private String args;
	private String initialDirectory;
	private String environment;
	private BitmaskSet<DebugCreateFlags> createFlags;
	private BitmaskSet<DebugEngCreateFlags> engCreateFlags;
	private BitmaskSet<DebugVerifierFlags> verifierFlags;

	public DbgLaunchProcessCommand(DbgManagerImpl manager, String args,
			String initialDirectory, String environment,
			BitmaskSet<DebugCreateFlags> createFlags,
			BitmaskSet<DebugEngCreateFlags> engCreateFlags,
			BitmaskSet<DebugVerifierFlags> verifierFlags) {
		super(manager);
		this.args = args;
		this.initialDirectory = initialDirectory;
		this.environment = environment;
		this.createFlags = createFlags;
		this.engCreateFlags = engCreateFlags;
		this.verifierFlags = verifierFlags;
	}

	@Override
	public boolean handle(DbgEvent<?> evt, DbgPendingCommand<?> pending) {
		if (evt instanceof AbstractDbgCompletedCommandEvent && pending.getCommand().equals(this)) {
			completed = true;
		}
		else if (evt instanceof DbgProcessCreatedEvent) {
			created = (DbgProcessCreatedEvent) evt;
		}
		return completed && (created != null);
	}

	@Override
	public DbgThread complete(DbgPendingCommand<?> pending) {
		DebugProcessInfo info = created.getInfo();
		DebugThreadInfo tinfo = info.initialThreadInfo;
		DebugSystemObjects so = manager.getSystemObjects();
		DebugThreadId tid = so.getThreadIdByHandle(tinfo.handle);
		return manager.getThread(tid);
	}

	@Override
	public void invoke() {
		DebugClient dbgeng = manager.getClient();
		//DebugControl control = dbgeng.getControl();

		initialDirectory = fixPath(initialDirectory);
		environment = fixPath(environment);
		// NB: The intent here is to enable multi-line input via a single dialog field
		if (environment != null) {
			environment = environment.replace("\\0", "\0");
		}

		dbgeng.createProcess(dbgeng.getLocalServer(), args,
			initialDirectory, environment, createFlags, engCreateFlags, verifierFlags);
		manager.waitForEventEx();
	}

	private String fixPath(String input) {
		if (input == null || input.equals("")) {
			return null;
		}
		String output = input;
		if (input.startsWith("/")) {
			output = output.substring(1);
		}
		output = output.replace("/", "\\");
		return output;
	}
}
