module Parsers where


infixr 2 |||
infixr 2 +||
infixr 5 .>.
infixr 5 ...
infixr 5 +..
infixr 5 ..+
infixr 3 ***
infixr 3 *?*
infixr 6 <|
infixr 3 <<<


-- parsing according to Wadler and Hutton. AR 1998

type Parser a b = [a] -> [(b,[a])]

succeed :: b -> Parser a b
succeed v s = [(v,s)]

fails :: Parser a b
fails s = []

(|||) :: Parser a b -> Parser a b -> Parser a b
(p1 ||| p2) s = p1 s ++ p2 s

(.>.) :: Parser a b -> (b -> Parser a c) -> Parser a c
(p1 .>. p2) s = [(b,z) | (a,y) <- p1 s, (b,z) <- p2 a y]



(...) :: Parser a b -> Parser a c -> Parser a (b,c)
p1 ... p2 = p1 .>. (\x -> p2 .>. (\y -> succeed (x,y)))

(+..) :: Parser a b -> Parser a c -> Parser a c
p1 +.. p2 = p1 .>. (\x -> p2 .>. (\y -> succeed y))

(..+) :: Parser a b -> Parser a c -> Parser a b
p1 ..+ p2 = p1 .>. (\x -> p2 .>. (\y -> succeed x))

(***) :: Parser a b -> (b -> c) -> Parser a c
(p *** f) = p .>. (\x -> succeed (f x))

(*?*) :: Parser a b -> (b -> Maybe c) -> Parser a c
(p *?* f) = p .>. (\x -> case f x of Just c -> succeed c
                                     _      -> fails)

(<<<) :: Parser a b -> c -> Parser a c  -- return
p <<< v = p *** (\x -> v)


item :: Parser a a
item [] = []
item (a:x) = [(a,x)]

(|>) :: Parser a b -> (b -> Bool) -> Parser a b
p |> b = p .>. (\x -> if b x then succeed x else fails)

satisfy :: (a -> Bool) -> Parser a a
satisfy b = item |> b

literal :: (Eq a) => a -> Parser a a
literal x = satisfy (==x)

first :: Parser a b -> Parser a b
first p s = case p s of []    -> []
                        (x:l) -> [x]

(+||) :: Parser a b -> Parser a b -> Parser a b
p1 +|| p2 = first (p1 ||| p2)

many :: Parser a b -> Parser a [b]
many p = p .>. (\x -> many p .>. (\y -> succeed (x:y))) ||| succeed []

some :: Parser a b -> Parser a [b]
some p = (p ... many p) *** (\ (x,y) -> x:y)

longestOfMany :: Parser a b -> Parser a [b]
longestOfMany p = 
  guarantee 
   (p .>. (\x -> longestOfMany p .>. (\y -> succeed (x:y))) +|| succeed [])

guarantee :: Parser a b -> Parser a b
guarantee p s = let u = p s in (fst (head u),snd (head u)) : tail u

closure :: (b -> Parser a b) -> (b -> Parser a b)
closure p v = p v .>. closure p ||| succeed v


PJunk :: Parser Char String
PJ :: Parser Char a -> Parser Char a
PTList :: (Eq a) => [a] -> Parser a b -> Parser a [b]
PLookup :: [(String,a)] -> Parser Char a
PElem   :: [String] -> Parser Char String

PSpace = many (satisfy (\x -> elem x " "))
PJunk = many (satisfy (\x -> elem x "\n "))
PSp P = PSpace +.. P ..+ PSpace
PJ P  = PJunk +.. P ..+ PJunk
PTList t P = P ... many (literals t +.. P) *** (\ (x,y) -> x:y)
PLookup C = foldl (+||) fails [literals s *** (\x -> c) | (s,c) <- C]
PElem L = foldl (+||) fails (map literals L)

literals :: (Eq a) => [a] -> Parser a [a]
literals l = case l of []  -> succeed [] 
                       a:l -> literal a ... literals l *** (\ (x,y) -> x:y)


