/*
 * Decompiled with CFR 0.152.
 */
package com.gemstone.gemfire.cache.hdfs.internal.hoplog;

import com.gemstone.gemfire.cache.hdfs.HDFSStore;
import com.gemstone.gemfire.cache.hdfs.internal.hoplog.CompactionStatus;
import com.gemstone.gemfire.cache.hdfs.internal.hoplog.HoplogOrganizer;
import com.gemstone.gemfire.internal.i18n.LocalizedStrings;
import com.gemstone.gemfire.internal.logging.LogService;
import com.gemstone.gemfire.internal.logging.log4j.LocalizedMessage;
import java.io.IOException;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;

public class HDFSCompactionManager {
    private static final ConcurrentHashMap<String, HDFSCompactionManager> storeToManagerMap = new ConcurrentHashMap();
    HDFSStore storeConfig;
    private final CompactionExecutor minorCompactor;
    private final CompactionExecutor majorCompactor;
    private static final Logger logger = LogService.getLogger();
    protected static final String logPrefix = "<HDFSCompactionManager> ";

    private HDFSCompactionManager(HDFSStore config) {
        this.storeConfig = config;
        int capacity = Integer.getInteger("hoplog.compaction.queue.capacity", 500);
        this.minorCompactor = new CompactionExecutor(config.getMinorCompactionThreads(), capacity, "MinorCompactor_" + config.getName());
        this.majorCompactor = new CompactionExecutor(config.getMajorCompactionThreads(), capacity, "MajorCompactor_" + config.getName());
        this.minorCompactor.allowCoreThreadTimeOut(true);
        this.majorCompactor.allowCoreThreadTimeOut(true);
    }

    public static synchronized HDFSCompactionManager getInstance(HDFSStore config) {
        HDFSCompactionManager instance = storeToManagerMap.get(config.getName());
        if (instance == null) {
            instance = new HDFSCompactionManager(config);
            storeToManagerMap.put(config.getName(), instance);
        }
        return instance;
    }

    public synchronized Future<CompactionStatus> submitRequest(CompactionRequest request) {
        if (!request.isForced && request.compactor.isBusy(request.isMajor)) {
            if (logger.isDebugEnabled()) {
                this.fineLog("Compactor is busy. Ignoring ", request);
            }
            return null;
        }
        CompactionExecutor executor = request.isMajor ? this.majorCompactor : this.minorCompactor;
        try {
            return executor.submit(request);
        }
        catch (Throwable e) {
            if (e instanceof CompactionIsDisabled) {
                if (logger.isDebugEnabled()) {
                    this.fineLog("{}" + e.getMessage(), logPrefix);
                }
            } else {
                logger.info((Message)LocalizedMessage.create(LocalizedStrings.ONE_ARG, "Compaction request submission failed"), e);
            }
            return null;
        }
    }

    public void reset() {
        this.minorCompactor.shutdownNow();
        this.majorCompactor.shutdownNow();
        storeToManagerMap.remove(this.storeConfig.getName());
    }

    public ThreadPoolExecutor getMinorCompactor() {
        return this.minorCompactor;
    }

    public ThreadPoolExecutor getMajorCompactor() {
        return this.majorCompactor;
    }

    private void fineLog(Object ... strings) {
        if (logger.isDebugEnabled()) {
            StringBuffer sb = new StringBuffer();
            for (Object str : strings) {
                sb.append(str.toString());
            }
            logger.debug("{}" + sb.toString(), new Object[]{logPrefix});
        }
    }

    public static class CompactionIsDisabled
    extends RejectedExecutionException {
        private static final long serialVersionUID = 1L;

        public CompactionIsDisabled(String name) {
            super(name);
        }
    }

    private class CompactionExecutor
    extends ThreadPoolExecutor
    implements ThreadFactory {
        final AtomicInteger count;
        private String name;

        CompactionExecutor(int max, int capacity, String name) {
            super(max, max, 5L, TimeUnit.SECONDS, new LinkedBlockingDeque<Runnable>(capacity));
            this.count = new AtomicInteger(1);
            this.allowCoreThreadTimeOut(true);
            this.setThreadFactory(this);
            this.name = name;
        }

        private void throwIfStopped(CompactionRequest req, HDFSStore storeConfig) {
            boolean isEnabled = true;
            isEnabled = storeConfig.getMinorCompaction();
            if (req.isMajor) {
                isEnabled = storeConfig.getMajorCompaction();
            }
            if (isEnabled || req.isForced) {
                return;
            }
            throw new CompactionIsDisabled(this.name + " is disabled");
        }

        private void throwIfPoolSizeChanged(CompactionRequest task, HDFSStore config) {
            int threadCount = config.getMinorCompactionThreads();
            if (task.isMajor) {
                threadCount = config.getMajorCompactionThreads();
            }
            if (this.getCorePoolSize() < threadCount) {
                this.setCorePoolSize(threadCount);
            } else if (this.getCorePoolSize() > threadCount) {
                this.setCorePoolSize(threadCount);
            }
            if (!task.isForced && this.getActiveCount() > threadCount) {
                throw new CompactionIsDisabled("Rejecting to reduce the number of threads for " + this.name + ", currently:" + this.getActiveCount() + " target:" + threadCount);
            }
        }

        @Override
        public <T> Future<T> submit(Callable<T> task) {
            this.throwIfStopped((CompactionRequest)task, HDFSCompactionManager.this.storeConfig);
            this.throwIfPoolSizeChanged((CompactionRequest)task, HDFSCompactionManager.this.storeConfig);
            if (logger.isDebugEnabled()) {
                HDFSCompactionManager.this.fineLog(new Object[]{"New:", task, " pool:", this.getPoolSize(), " active:", this.getActiveCount()});
            }
            return super.submit(task);
        }

        @Override
        public Thread newThread(Runnable r) {
            Thread thread = new Thread(r, this.name + ":" + this.count.getAndIncrement());
            thread.setDaemon(true);
            if (logger.isDebugEnabled()) {
                HDFSCompactionManager.this.fineLog(new Object[]{"New thread:", this.name, " poolSize:", this.getPoolSize(), " active:", this.getActiveCount()});
            }
            return thread;
        }
    }

    public static class CompactionRequest
    implements Callable<CompactionStatus> {
        String regionFolder;
        int bucket;
        HoplogOrganizer.Compactor compactor;
        boolean isMajor;
        final boolean isForced;
        final boolean versionUpgrade;

        public CompactionRequest(String regionFolder, int bucket, HoplogOrganizer.Compactor compactor, boolean major) {
            this(regionFolder, bucket, compactor, major, false);
        }

        public CompactionRequest(String regionFolder, int bucket, HoplogOrganizer.Compactor compactor, boolean major, boolean isForced) {
            this(regionFolder, bucket, compactor, major, isForced, false);
        }

        public CompactionRequest(String regionFolder, int bucket, HoplogOrganizer.Compactor compactor, boolean major, boolean isForced, boolean versionUpgrade) {
            this.regionFolder = regionFolder;
            this.bucket = bucket;
            this.compactor = compactor;
            this.isMajor = major;
            this.isForced = isForced;
            this.versionUpgrade = versionUpgrade;
        }

        @Override
        public CompactionStatus call() throws Exception {
            HDFSStore store = this.compactor.getHdfsStore();
            if (!this.isForced) {
                if (this.isMajor && !store.getMajorCompaction()) {
                    if (logger.isDebugEnabled()) {
                        logger.debug("{}Major compaction is disabled. Ignoring request", new Object[]{HDFSCompactionManager.logPrefix});
                    }
                    return new CompactionStatus(this.bucket, false);
                }
                if (!this.isMajor && !store.getMinorCompaction()) {
                    if (logger.isDebugEnabled()) {
                        logger.debug("{}Minor compaction is disabled. Ignoring request", new Object[]{HDFSCompactionManager.logPrefix});
                    }
                    return new CompactionStatus(this.bucket, false);
                }
            }
            try {
                boolean status = this.compactor.compact(this.isMajor, this.versionUpgrade);
                return new CompactionStatus(this.bucket, status);
            }
            catch (IOException e) {
                logger.error((Message)LocalizedMessage.create(LocalizedStrings.HOPLOG_HDFS_COMPACTION_ERROR, this.bucket), (Throwable)e);
                return new CompactionStatus(this.bucket, false);
            }
        }

        public int hashCode() {
            int prime = 31;
            int result = 1;
            result = 31 * result + this.bucket;
            result = 31 * result + (this.regionFolder == null ? 0 : this.regionFolder.hashCode());
            return result;
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            CompactionRequest other = (CompactionRequest)obj;
            if (this.bucket != other.bucket) {
                return false;
            }
            return !(this.regionFolder == null ? other.regionFolder != null : !this.regionFolder.equals(other.regionFolder));
        }

        public String toString() {
            return "CompactionRequest [regionFolder=" + this.regionFolder + ", bucket=" + this.bucket + ", isMajor=" + this.isMajor + ", isForced=" + this.isForced + "]";
        }
    }
}

