
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

#ifdef HAVE_DIRENT_H
# include <dirent.h>
#endif
#ifdef HAVE_PWD_H
#include <pwd.h>
#endif
#ifdef HAVE_STDIO_H
# include <stdio.h>
#endif
#ifdef HAVE_STDLIB_H
# include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
# include <string.h>
#endif
#ifdef HAVE_STRINGS_H
# include <strings.h>
#endif
#ifdef HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

#include "version.h"
#include "gensig.h"
#include "tagdb.h"

/* typedefs & globals */

typedef struct {
	char name[BUFSIZ];
	dev_t device;
	ino_t inode;
	time_t open_time;
	FILE *file;
	DIR *dir;
} filenode;

filenode *filelist = NULL;
int filecount = 0;
int filelistsize = 0;
#define INDEX_STEP 1024

typedef struct {
	FILE *file;
	long offset;
} tagnode;

tagnode *tagindex = NULL;
int tagcount = 0;
int tagindexsize = 0;
#define FILELIST_STEP 10

int done_srand = 0;

/* */

int init_tagdb()
{
   struct timeval tv;

   if (debug) fprintf(stderr, "Initializing tagdb.\n");

   if (!done_srand)
   {
      gettimeofday(&tv, NULL);

#ifdef HAVE_RANDOM
      srandom(tv.tv_sec ^ tv.tv_usec);
#else
      srand(tv.tv_sec ^ tv.tv_usec);
#endif
      done_srand = 1;
   }

   filelistsize = FILELIST_STEP;
   filelist = (filenode*)malloc(filelistsize * sizeof(filenode));
   filecount = 0;

   tagindexsize = INDEX_STEP;
   tagindex = (tagnode*)malloc(tagindexsize * sizeof(tagnode));
   tagcount = 0;

   return 0;
}

int already_in_filelist(dev_t device, ino_t inode)
{
   int i;
   int rv = 0;

   for (i = 0; i < filecount; i++)
   {
      if (filelist[i].file || filelist[i].dir)
      {
         if ( (filelist[i].device == device) && (filelist[i].inode == inode) )
         {
            rv++;
         }
      }
   }

   return rv;
}

int open_tagfile(const char *newfn)
{
   char buffer[BUFSIZ];
   struct stat statbuf;
   struct dirent *mydirent;
   DIR *newdir;
   FILE *newfile;
   time_t open_time;
   int rv = 0;
   int count = 0;

   if (newfn == NULL)
   {
      /* try ~/.taglines first... */

      sprintf(buffer, "%s/.taglines", getpwuid(getuid())->pw_dir);

      rv = stat(buffer, &statbuf);

      /* only try it if it's there, and it's a dir or regular file */
      if ( (rv == 0) && ( (S_ISREG(statbuf.st_mode)) || (S_ISDIR(statbuf.st_mode)) ) )
      {
         if (already_in_filelist(statbuf.st_dev, statbuf.st_ino) == 0)
         {
            if (debug) fprintf(stderr, "Opening user taglines source: %s\n", buffer);
            count += open_tagfile(buffer);
         }
      }

      /* use system default if we don't have any taglines yet... */
      if (count == 0)
      {
         strncpy(buffer, PKGDATADIR "/taglines", BUFSIZ);
         if (debug) fprintf(stderr, "Opening system taglines file: %s\n", buffer);
         count += open_tagfile(buffer);
      }
   }
   else
   {
      if ( (rv = stat(newfn, &statbuf)) == 0 )
      {
         if ( already_in_filelist(statbuf.st_dev, statbuf.st_ino) == 0 )
         {

            /* if it's a directory, recurse */
            if (S_ISDIR(statbuf.st_mode))
            {
               if (debug) fprintf(stderr, "Recursing into %s\n", newfn);

               /* for each file in the directory, recurse... */

               rv = 0;

               open_time = time(NULL);
               newdir = opendir(newfn);

               if (newdir)
               {
                  /* expand filelist possibly */
                  if (filecount >= filelistsize)
                  {
                     filelistsize += FILELIST_STEP;
                     filelist = (filenode*)realloc(filelist, sizeof(filenode) * filelistsize);

                     if (filelist == NULL)
                     {
                        perror("realloc filelist");
                        exit(1);
                     }
                  }

                  /* make new file entry */
                  strncpy(filelist[filecount].name, newfn, BUFSIZ);
                  filelist[filecount].device = statbuf.st_dev;
                  filelist[filecount].inode = statbuf.st_ino;
                  filelist[filecount].file = NULL;
                  filelist[filecount].dir = newdir;
                  filelist[filecount].open_time = open_time;
                  filecount++;

                  while ((mydirent = readdir(newdir)))
                  {
                     /* ignore all dot-files --
			this gets around '.' and '..', too */
                     if ( mydirent->d_name[0] != '.' )
                     {
                        sprintf(buffer, "%s/%s", newfn, mydirent->d_name);
                        count += open_tagfile(buffer);
                     }
                  }

                  seekdir(newdir, 0);
               }
               else
               {
                  perror("dir open");
               }
            }
            /* otherwise if it's a regular file:  open & index... */
            else if (S_ISREG(statbuf.st_mode))
            {
               if (debug) fprintf(stderr, "Adding %s\n", newfn);

               open_time = time(NULL);
               newfile = fopen(newfn, "r");

               if (newfile == NULL)
               {
                  perror("newfile open");
                  return 0;
               }

               /* expand filelist possibly */
               if (filecount >= filelistsize)
               {
                  filelistsize += FILELIST_STEP;
                  filelist = (filenode*)realloc(filelist, sizeof(filenode) * filelistsize);

                  if (filelist == NULL)
                  {
                     perror("realloc filelist");
                     exit(1);
                  }
               }

               rv = 0;

               /* make new file entry */
               strncpy(filelist[filecount].name, newfn, BUFSIZ);
               filelist[filecount].device = statbuf.st_dev;
               filelist[filecount].inode = statbuf.st_ino;
               filelist[filecount].file = newfile;
               filelist[filecount].dir = NULL;
               filelist[filecount].open_time = open_time;
               filecount++;

               tagindex[tagcount].file = newfile;	/* always true */
               tagindex[tagcount].offset = 0;	/* always true */

               while (fgets(buffer, BUFSIZ, newfile) != NULL)
               {
                  tagcount++;
                  rv++;

                  if (tagcount >= tagindexsize)
                  {
                     tagindexsize += INDEX_STEP;
                     tagindex = (tagnode*)realloc(tagindex, sizeof(tagnode) * tagindexsize);

                     if (tagindex == NULL)
                     {
                        perror("realloc tagindex");
                        exit(1);
                     }
                  }

                  tagindex[tagcount].file = newfile;
                  tagindex[tagcount].offset = ftell(newfile);
               }

               count++;
            }
            else
            {
               fprintf(stderr, "not a directory or regular file: %s", newfn);
            }
         }
         else
         {
            if (debug) fprintf(stderr, "Won't add duplicate file: %s\n", newfn);
            rv = 0;
         }
      }
      else
      {
         sprintf(buffer, "stat %s", newfn);
         perror(buffer);
      }
   }

   return count;
}

int open_default_tagfiles()
{
   char buffer[BUFSIZ];
   char *ind, *next;
   int count = 0;

   if (getenv("TAGLINES"))
   {
      strncpy(buffer, getenv("TAGLINES"), BUFSIZ);
      if (debug) fprintf(stderr, "Got env var of '%s'\n", buffer);

      ind = buffer;
      while ((next = index(ind, ':')) != NULL)
      {
         /* terminate the filename */
         *next = '\0';

         /* add it */
         count += open_tagfile(ind);

         /* next place to search from */
         ind = next + 1;
      }

      if (*ind == '\0')
      {
         if (debug) fprintf(stderr, "Adding defaults too!\n");
         count += open_tagfile(NULL);
      }
      else
      {
         count += open_tagfile(ind);
      }

   }

   if (count == 0)
   {
      if (debug) fprintf(stderr, "Nothing good found in $TAGLINES -- defaulting\n");
      count += open_tagfile(NULL);
   }

   return count;
}

int close_tagdb()
{
   int i;

   for ( i=0; i < filecount; i++)
   {
      if (filelist[i].file) fclose(filelist[i].file);
      if (filelist[i].dir) closedir(filelist[i].dir);
   }

   if (tagindex) free(tagindex);
   tagindex = NULL;
   tagindexsize = 0;
   tagcount = 0;

   if (filelist) free(filelist);
   filelist = NULL;
   filelistsize = 0;
   filecount = 0;

   return 0;
}

int pack_filelist()
{
   /* remove entries with null FILE*, and decrement filecount appropriately */
   int i;
   int rv = 0;

   for (i=0; i<filecount; )
   {
      if ( (filelist[i].file == NULL) && (filelist[i].dir == NULL) )
      {
         filecount--;
         filelist[i] = filelist[filecount];
         rv++;
      }
      else
      {
         i++;
      }
   }

   return rv;
}

int close_tagfile(int fileind)
{
   int i;

   /* close the file, null the FILE*, don't decrement filecount */
   if (debug) fprintf(stderr, "Closing %s\n", filelist[fileind].name);

   if (filelist[fileind].file)
   {
      for (i=0; i<tagcount; )
      {
         if (tagindex[i].file == filelist[fileind].file)
         {
            tagcount--;

            if (i != tagcount)
            {
               tagindex[i] = tagindex[tagcount];
            }
            else
            {
               tagindex[i].file = NULL;
            }
         }
         else
         {
            i++;
         }
      }
      fclose(filelist[fileind].file);
      filelist[fileind].file = NULL;
   }
   else if (filelist[fileind].dir)
   {
      closedir(filelist[fileind].dir);
      filelist[fileind].dir = NULL;
   }

   return 0;
}

int reindex_tagfile(int fileind)
{
   int i;
   int startcount, middlecount;
   char buffer[BUFSIZ];

   startcount = tagcount;

   if (debug) fprintf(stderr, "Reindexing %s: starting with %d entries.\n",
		filelist[fileind].name, tagcount);

   /* removing entries in the index for this file */
   for (i=0; i<tagcount; )
   {
      if (tagindex[i].file == filelist[fileind].file)
      {
         tagcount--;
         tagindex[i] = tagindex[tagcount];
      }
      else
      {
         i++;
      }
   }

   /* close the file */
   fclose(filelist[fileind].file);

   middlecount = tagcount;

   if (debug) fprintf(stderr, "Removed %d entries, %d remain\n", (startcount-tagcount), tagcount);

   /* reopen the file, remembering when we've done so */
   filelist[fileind].open_time = time(NULL);
   filelist[fileind].file = fopen(filelist[fileind].name, "r");

   if (filelist[fileind].file == NULL)	/* fopen failed */
   {
      close_tagfile(fileind);
      perror("tagfile open");
   }
   else
   {

      /* re-index the file */
      tagindex[tagcount].file = filelist[fileind].file;        /* always true */
      tagindex[tagcount].offset = 0;    /* always true */

      while (fgets(buffer, BUFSIZ, filelist[fileind].file) != NULL)
      {
         tagcount++;

         if (tagcount >= tagindexsize)
         {
            tagindexsize += INDEX_STEP;
            tagindex = (tagnode*)realloc(tagindex, sizeof(tagnode) * tagindexsize);

            if (tagindex == NULL)
            {
               perror("realloc tagindex");
               exit(1);
            }
         }

         tagindex[tagcount].file = filelist[fileind].file;
         tagindex[tagcount].offset = ftell(filelist[fileind].file);

      }

      if (debug) fprintf(stderr, "Read %d entries, %d total now\n",
	(tagcount-middlecount), tagcount);
   }

   return tagcount;
}

int checkpoint_tagdb()
{
   int i;
   struct stat statbuf;
   struct dirent *mydirent;
   char buffer[BUFSIZ];

   if ( (filelist == NULL) && (tagindex == NULL) ) return 0;

   for (i=0; i<filecount; i++)
   {
      if (debug) fprintf(stderr, "Checkpointing %s\n", filelist[i].name);

      if (filelist[i].file && filelist[i].dir)
      {
         fprintf(stderr, "Removing FILE/DIR screwup!\n");
         filelist[i].file = NULL;
         filelist[i].dir = NULL;
      }
      else if (filelist[i].file)
      {
         if ( stat(filelist[i].name, &statbuf) < 0 )
         {
            if (debug) fprintf(stderr, "Problem stat()ing %s, closing it.\n", filelist[i].name);
            close_tagfile(i);
         }
         else
         {
            if (statbuf.st_mtime >= filelist[i].open_time)
            {
               if (debug) fprintf(stderr, "Tagfile %s has been updated.\n", filelist[i].name);
               reindex_tagfile(i);
            }
         }
      }
      else if (filelist[i].dir)
      {
         if ( stat(filelist[i].name, &statbuf) < 0 )
         {
            if (debug) fprintf(stderr, "Problem stat()ing %s, closing it.\n", filelist[i].name);
            close_tagfile(i);
         }
         else
         {
            seekdir(filelist[i].dir, 0);
            while ((mydirent = readdir(filelist[i].dir)))
            {
               /* ignore all dot-files --
		this gets around '.' and '..', too */
               if ( mydirent->d_name[0] != '.' )
               {
                  sprintf(buffer, "%s/%s", filelist[i].name, mydirent->d_name);
                  open_tagfile(buffer);
               }
            }
            seekdir(filelist[i].dir, 0);
         }
      }
      else
      {
         /* nothing -- shouldn't be here, but it'll
		get cleaned in pack_filelist() */
      }
   }

   pack_filelist();

   return 0;
}

char *get_random_tagline(int with_checkpoint)
{
   int keynum;
   char buffer[BUFSIZ];
   static char rv[BUFSIZ];
   char *ind;

   if (filecount == 0) return NULL;

   if (with_checkpoint) checkpoint_tagdb();

   if (tagcount == 0) return NULL;

#ifdef HAVE_RANDOM
   keynum = random() % tagcount;
#else
   keynum = rand() % tagcount;
#endif

   fseek(tagindex[keynum].file, tagindex[keynum].offset, SEEK_SET);

   fgets(buffer, BUFSIZ, tagindex[keynum].file);

   ind = buffer;
   while ((ind = index(buffer, '\r')) != NULL) *ind++ = '\n';

   if (buffer[strlen(buffer)-1] == '\n') buffer[strlen(buffer)-1] = '\0';

   /* this is here in case I want to print out anything else with it... */
   /* sprintf(rv, "%d/%d: %s", keynum, tagcoung, buffer); */
   strncpy(rv, buffer, BUFSIZ);

   /* more processing still desired... */

   return rv;
}

