/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { AuthType } from '@google/gemini-cli-core';
import { vi } from 'vitest';
import { validateAuthMethod } from './auth.js';
vi.mock('./settings.js', () => ({
    loadEnvironment: vi.fn(),
    loadSettings: vi.fn().mockReturnValue({
        merged: vi.fn().mockReturnValue({}),
    }),
}));
describe('validateAuthMethod', () => {
    beforeEach(() => {
        vi.resetModules();
        vi.stubEnv('GEMINI_API_KEY', undefined);
        vi.stubEnv('GOOGLE_CLOUD_PROJECT', undefined);
        vi.stubEnv('GOOGLE_CLOUD_LOCATION', undefined);
        vi.stubEnv('GOOGLE_API_KEY', undefined);
    });
    afterEach(() => {
        vi.unstubAllEnvs();
    });
    it('should return null for LOGIN_WITH_GOOGLE', () => {
        expect(validateAuthMethod(AuthType.LOGIN_WITH_GOOGLE)).toBeNull();
    });
    it('should return null for CLOUD_SHELL', () => {
        expect(validateAuthMethod(AuthType.CLOUD_SHELL)).toBeNull();
    });
    describe('USE_GEMINI', () => {
        it('should return null if GEMINI_API_KEY is set', () => {
            vi.stubEnv('GEMINI_API_KEY', 'test-key');
            expect(validateAuthMethod(AuthType.USE_GEMINI)).toBeNull();
        });
        it('should return an error message if GEMINI_API_KEY is not set', () => {
            vi.stubEnv('GEMINI_API_KEY', undefined);
            expect(validateAuthMethod(AuthType.USE_GEMINI)).toBeNull();
        });
    });
    describe('USE_VERTEX_AI', () => {
        it('should return null if GOOGLE_CLOUD_PROJECT and GOOGLE_CLOUD_LOCATION are set', () => {
            vi.stubEnv('GOOGLE_CLOUD_PROJECT', 'test-project');
            vi.stubEnv('GOOGLE_CLOUD_LOCATION', 'test-location');
            expect(validateAuthMethod(AuthType.USE_VERTEX_AI)).toBeNull();
        });
        it('should return null if GOOGLE_API_KEY is set', () => {
            vi.stubEnv('GOOGLE_API_KEY', 'test-api-key');
            expect(validateAuthMethod(AuthType.USE_VERTEX_AI)).toBeNull();
        });
        it('should return an error message if no required environment variables are set', () => {
            vi.stubEnv('GOOGLE_CLOUD_PROJECT', undefined);
            vi.stubEnv('GOOGLE_CLOUD_LOCATION', undefined);
            expect(validateAuthMethod(AuthType.USE_VERTEX_AI)).toBe('When using Vertex AI, you must specify either:\n' +
                '• GOOGLE_CLOUD_PROJECT and GOOGLE_CLOUD_LOCATION environment variables.\n' +
                '• GOOGLE_API_KEY environment variable (if using express mode).\n' +
                'Update your environment and try again (no reload needed if using .env)!');
        });
    });
    it('should return an error message for an invalid auth method', () => {
        expect(validateAuthMethod('invalid-method')).toBe('Invalid auth method selected.');
    });
});
//# sourceMappingURL=auth.test.js.map