static const char* op_c_source =
"/* This file is an image processing operation for GEGL                        \n"
" * GEGL is free software; you can redistribute it and/or                      \n"
" * modify it under the terms of the GNU Lesser General Public                 \n"
" * License as published by the Free Software Foundation; either               \n"
" * version 3 of the License, or (at your option) any later version.           \n"
" *                                                                            \n"
" * GEGL is distributed in the hope that it will be useful,                    \n"
" * but WITHOUT ANY WARRANTY; without even the implied warranty of             \n"
" * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          \n"
" * Lesser General Public License for more details.                            \n"
" *                                                                            \n"
" * You should have received a copy of the GNU Lesser General Public           \n"
" * License along with GEGL; if not, see <http://www.gnu.org/licenses/>.       \n"
" *                                                                            \n"
" * Copyright 2006 Dominik Ernst <dernst@gmx.de>                               \n"
" *           2014 Daniel Sabo                                                 \n"
" *                                                                            \n"
" * Recursive Gauss IIR Filter as described by Young / van Vliet               \n"
" * in \"Signal Processing 44 (1995) 139 - 151\"                               \n"
" *                                                                            \n"
" * NOTE: The IIR filter should not be used for radius < 0.5, since it         \n"
" *       becomes very inaccurate.                                             \n"
" */                                                                           \n"
"                                                                              \n"
"#include \"config.h\"                                                         \n"
"#include <glib/gi18n-lib.h>                                                   \n"
"                                                                              \n"
"#ifdef GEGL_PROPERTIES                                                        \n"
"                                                                              \n"
"enum_start (gegl_gaussian_blur_filter)                                        \n"
"  enum_value (GEGL_GAUSSIAN_BLUR_FILTER_AUTO, \"auto\", N_(\"Auto\"))         \n"
"  enum_value (GEGL_GAUSSIAN_BLUR_FILTER_FIR,  \"fir\",  N_(\"FIR\"))          \n"
"  enum_value (GEGL_GAUSSIAN_BLUR_FILTER_IIR,  \"iir\",  N_(\"IIR\"))          \n"
"enum_end (GeglGaussianBlurFilter)                                             \n"
"                                                                              \n"
"property_double (std_dev_x, _(\"Size X\"), 1.5)                               \n"
"    description (_(\"Standard deviation for the horizontal axis\"))           \n"
"    value_range (0.0, 1500.0)                                                 \n"
"    ui_range    (0.0, 100.0)                                                  \n"
"    ui_gamma    (3.0)                                                         \n"
"    ui_meta     (\"unit\", \"pixel-distance\")                                \n"
"    ui_meta     (\"axis\", \"x\")                                             \n"
"                                                                              \n"
"property_double (std_dev_y, _(\"Size Y\"), 1.5)                               \n"
"    description (_(\"Standard deviation for the vertical axis\"))             \n"
"    value_range (0.0, 1500.0)                                                 \n"
"    ui_range    (0.0, 100.0)                                                  \n"
"    ui_gamma    (3.0)                                                         \n"
"    ui_meta     (\"unit\", \"pixel-distance\")                                \n"
"    ui_meta     (\"axis\", \"y\")                                             \n"
"                                                                              \n"
"property_enum  (filter, _(\"Filter\"),                                        \n"
"    GeglGaussianBlurFilter, gegl_gaussian_blur_filter,                        \n"
"    GEGL_GAUSSIAN_BLUR_FILTER_AUTO)                                           \n"
"    description (_(\"Optional parameter to override the automatic selection of blur filter\"))\n"
"                                                                              \n"
"#else                                                                         \n"
"                                                                              \n"
"#define GEGL_OP_AREA_FILTER                                                   \n"
"#define GEGL_OP_NAME     gaussian_blur                                        \n"
"#define GEGL_OP_C_SOURCE gaussian-blur.c                                      \n"
"                                                                              \n"
"#include \"gegl-op.h\"                                                        \n"
"#include <math.h>                                                             \n"
"#include <stdio.h>                                                            \n"
"                                                                              \n"
"#define RADIUS_SCALE   4                                                      \n"
"                                                                              \n"
"static void                                                                   \n"
"iir_young_find_constants (gfloat   radius,                                    \n"
"                          gdouble *B,                                         \n"
"                          gdouble *b);                                        \n"
"                                                                              \n"
"static gint                                                                   \n"
"fir_gen_convolve_matrix (gdouble   sigma,                                     \n"
"                         gdouble **cmatrix_p);                                \n"
"                                                                              \n"
"                                                                              \n"
"static void                                                                   \n"
"iir_young_find_constants (gfloat   sigma,                                     \n"
"                          gdouble *B,                                         \n"
"                          gdouble *b)                                         \n"
"{                                                                             \n"
"  gdouble q;                                                                  \n"
"                                                                              \n"
"  if (GEGL_FLOAT_IS_ZERO (sigma))                                             \n"
"    {                                                                         \n"
"      /* to prevent unexpected ringing at tile boundaries,                    \n"
"         we define an (expensive) copy operation here */                      \n"
"      *B = 1.0;                                                               \n"
"      b[0] = 1.0;                                                             \n"
"      b[1] = b[2] = b[3] = 0.0;                                               \n"
"      return;                                                                 \n"
"    }                                                                         \n"
"                                                                              \n"
"  if(sigma >= 2.5)                                                            \n"
"    q = 0.98711*sigma - 0.96330;                                              \n"
"  else                                                                        \n"
"    q = 3.97156 - 4.14554*sqrt(1-0.26891*sigma);                              \n"
"                                                                              \n"
"  b[0] = 1.57825 + (2.44413*q) + (1.4281*q*q) + (0.422205*q*q*q);             \n"
"  b[1] = (2.44413*q) + (2.85619*q*q) + (1.26661*q*q*q);                       \n"
"  b[2] = -((1.4281*q*q) + (1.26661*q*q*q));                                   \n"
"  b[3] = 0.422205*q*q*q;                                                      \n"
"                                                                              \n"
"  *B = 1 - ( (b[1]+b[2]+b[3])/b[0] );                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static inline void                                                            \n"
"iir_young_blur_pixels_1D (gfloat  *buf,                                       \n"
"                          gint     components,                                \n"
"                          gdouble  B,                                         \n"
"                          gdouble *b,                                         \n"
"                          gfloat  *w,                                         \n"
"                          gint     w_len)                                     \n"
"{                                                                             \n"
"  gint wcount, i, c;                                                          \n"
"  gdouble tmp;                                                                \n"
"  gdouble recip = 1.0 / b[0];                                                 \n"
"  gint offset = 0;                                                            \n"
"                                                                              \n"
"  /* forward filter */                                                        \n"
"  wcount = 0;                                                                 \n"
"                                                                              \n"
"  while (wcount < w_len)                                                      \n"
"    {                                                                         \n"
"      for (c = 0; c < components; ++c)                                        \n"
"        {                                                                     \n"
"          tmp = 0;                                                            \n"
"                                                                              \n"
"          for (i = 1; i < 4; i++)                                             \n"
"            {                                                                 \n"
"              if (wcount - i >= 0)                                            \n"
"                tmp += b[i] * w[(wcount - i) * components + c];               \n"
"            }                                                                 \n"
"                                                                              \n"
"          tmp *= recip;                                                       \n"
"          tmp += B * buf[offset + c];                                         \n"
"          w[wcount * components + c] = tmp;                                   \n"
"        }                                                                     \n"
"                                                                              \n"
"      wcount++;                                                               \n"
"      offset += components;                                                   \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* backward filter */                                                       \n"
"  wcount = w_len - 1;                                                         \n"
"  offset -= components;                                                       \n"
"                                                                              \n"
"  while (wcount >= 0)                                                         \n"
"    {                                                                         \n"
"      for (c = 0; c < components; ++c)                                        \n"
"        {                                                                     \n"
"          tmp = 0;                                                            \n"
"                                                                              \n"
"          for (i = 1; i < 4; i++)                                             \n"
"            {                                                                 \n"
"              if (wcount + i < w_len)                                         \n"
"                tmp += b[i] * buf[offset + components * i + c];               \n"
"            }                                                                 \n"
"                                                                              \n"
"          tmp *= recip;                                                       \n"
"          tmp += B * w[wcount * components + c];                              \n"
"          buf[offset + c] = tmp;                                              \n"
"        }                                                                     \n"
"                                                                              \n"
"      offset -= components;                                                   \n"
"      wcount--;                                                               \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"/* expects src and dst buf to have the same height and no y-offset */         \n"
"static void                                                                   \n"
"iir_young_hor_blur (GeglBuffer          *src,                                 \n"
"                    const GeglRectangle *src_rect,                            \n"
"                    GeglBuffer          *dst,                                 \n"
"                    const GeglRectangle *dst_rect,                            \n"
"                    gdouble              B,                                   \n"
"                    gdouble             *b)                                   \n"
"{                                                                             \n"
"  gint v;                                                                     \n"
"  const Babl *format = babl_format (\"RaGaBaA float\");                       \n"
"  const int pixel_count = src_rect->width;                                    \n"
"  gfloat *buf     = gegl_malloc (pixel_count * sizeof(gfloat) * 4);           \n"
"  gfloat *scratch = gegl_malloc (pixel_count * sizeof(gfloat) * 4);           \n"
"  GeglRectangle read_rect = {src_rect->x, dst_rect->y, src_rect->width, 1};   \n"
"                                                                              \n"
"  for (v = 0; v < dst_rect->height; v++)                                      \n"
"    {                                                                         \n"
"      read_rect.y = dst_rect->y + v;                                          \n"
"      gegl_buffer_get (src, &read_rect, 1.0, format, buf, GEGL_AUTO_ROWSTRIDE, GEGL_ABYSS_NONE);\n"
"                                                                              \n"
"      iir_young_blur_pixels_1D (buf, 4, B, b, scratch, pixel_count);          \n"
"                                                                              \n"
"      gegl_buffer_set (dst, &read_rect, 0, format, buf, GEGL_AUTO_ROWSTRIDE); \n"
"    }                                                                         \n"
"                                                                              \n"
"  gegl_free (buf);                                                            \n"
"  gegl_free (scratch);                                                        \n"
"}                                                                             \n"
"                                                                              \n"
"/* expects src and dst buf to have the same width and no x-offset */          \n"
"static void                                                                   \n"
"iir_young_ver_blur (GeglBuffer          *src,                                 \n"
"                    const GeglRectangle *src_rect,                            \n"
"                    GeglBuffer          *dst,                                 \n"
"                    const GeglRectangle *dst_rect,                            \n"
"                    gdouble              B,                                   \n"
"                    gdouble             *b)                                   \n"
"{                                                                             \n"
"  gint u;                                                                     \n"
"  const Babl *format = babl_format (\"RaGaBaA float\");                       \n"
"  const int pixel_count = src_rect->height;                                   \n"
"  gfloat *buf     = gegl_malloc (pixel_count * sizeof(gfloat) * 4);           \n"
"  gfloat *scratch = gegl_malloc (pixel_count * sizeof(gfloat) * 4);           \n"
"  GeglRectangle read_rect = {dst_rect->x, src_rect->y, 1, src_rect->height};  \n"
"                                                                              \n"
"  for (u = 0; u < dst_rect->width; u++)                                       \n"
"    {                                                                         \n"
"      read_rect.x = dst_rect->x + u;                                          \n"
"      gegl_buffer_get (src, &read_rect, 1.0, format, buf, GEGL_AUTO_ROWSTRIDE, GEGL_ABYSS_NONE);\n"
"                                                                              \n"
"      iir_young_blur_pixels_1D (buf, 4, B, b, scratch, pixel_count);          \n"
"                                                                              \n"
"      gegl_buffer_set (dst, &read_rect, 0, format, buf, GEGL_AUTO_ROWSTRIDE); \n"
"    }                                                                         \n"
"                                                                              \n"
"  gegl_free (buf);                                                            \n"
"  gegl_free (scratch);                                                        \n"
"}                                                                             \n"
"                                                                              \n"
"static gint                                                                   \n"
"fir_calc_convolve_matrix_length (gdouble sigma)                               \n"
"{                                                                             \n"
"  return sigma > GEGL_FLOAT_EPSILON ? ceil (sigma) * 6 + 1 : 1;               \n"
"}                                                                             \n"
"                                                                              \n"
"static gint                                                                   \n"
"fir_gen_convolve_matrix (gdouble   sigma,                                     \n"
"                         gdouble **cmatrix_p)                                 \n"
"{                                                                             \n"
"  gint     matrix_length;                                                     \n"
"  gdouble *cmatrix;                                                           \n"
"                                                                              \n"
"  matrix_length = fir_calc_convolve_matrix_length (sigma);                    \n"
"  cmatrix = g_new (gdouble, matrix_length);                                   \n"
"  if (!cmatrix)                                                               \n"
"    return 0;                                                                 \n"
"                                                                              \n"
"  if (matrix_length == 1)                                                     \n"
"    {                                                                         \n"
"      cmatrix[0] = 1;                                                         \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      gint i,x;                                                               \n"
"      gdouble sum = 0;                                                        \n"
"                                                                              \n"
"      for (i=0; i<matrix_length/2+1; i++)                                     \n"
"        {                                                                     \n"
"          gdouble y;                                                          \n"
"                                                                              \n"
"          x = i - (matrix_length/2);                                          \n"
"          y = (1.0/(sigma*sqrt(2.0*G_PI))) * exp(-(x*x) / (2.0*sigma*sigma)); \n"
"                                                                              \n"
"          cmatrix[i] = y;                                                     \n"
"          sum += cmatrix[i];                                                  \n"
"        }                                                                     \n"
"                                                                              \n"
"      for (i=matrix_length/2 + 1; i<matrix_length; i++)                       \n"
"        {                                                                     \n"
"          cmatrix[i] = cmatrix[matrix_length-i-1];                            \n"
"          sum += cmatrix[i];                                                  \n"
"        }                                                                     \n"
"                                                                              \n"
"      for (i=0; i<matrix_length; i++)                                         \n"
"      {                                                                       \n"
"        cmatrix[i] /= sum;                                                    \n"
"      }                                                                       \n"
"    }                                                                         \n"
"                                                                              \n"
"  *cmatrix_p = cmatrix;                                                       \n"
"  return matrix_length;                                                       \n"
"}                                                                             \n"
"                                                                              \n"
"static inline void                                                            \n"
"fir_get_mean_pixel_1D (gfloat  *src,                                          \n"
"                       gfloat  *dst,                                          \n"
"                       gint     components,                                   \n"
"                       gdouble *cmatrix,                                      \n"
"                       gint     matrix_length)                                \n"
"{                                                                             \n"
"  gint    i, c;                                                               \n"
"  gint    offset;                                                             \n"
"  gdouble acc[components];                                                    \n"
"                                                                              \n"
"  for (c = 0; c < components; ++c)                                            \n"
"    acc[c] = 0;                                                               \n"
"                                                                              \n"
"  offset = 0;                                                                 \n"
"                                                                              \n"
"  for (i = 0; i < matrix_length; i++)                                         \n"
"    {                                                                         \n"
"      for (c = 0; c < components; ++c)                                        \n"
"        acc[c] += src[offset++] * cmatrix[i];                                 \n"
"    }                                                                         \n"
"                                                                              \n"
"  for (c = 0; c < components; ++c)                                            \n"
"    dst[c] = acc[c];                                                          \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"fir_hor_blur (GeglBuffer          *src,                                       \n"
"              GeglBuffer          *dst,                                       \n"
"              const GeglRectangle *dst_rect,                                  \n"
"              gdouble             *cmatrix,                                   \n"
"              gint                 matrix_length)                             \n"
"{                                                                             \n"
"  gint        u, v;                                                           \n"
"  const gint  radius = matrix_length / 2;                                     \n"
"  const Babl *format = babl_format (\"RaGaBaA float\");                       \n"
"                                                                              \n"
"  GeglRectangle write_rect = {dst_rect->x, dst_rect->y, dst_rect->width, 1};  \n"
"  gfloat *dst_buf     = gegl_malloc (write_rect.width * sizeof(gfloat) * 4);  \n"
"                                                                              \n"
"  GeglRectangle read_rect = {dst_rect->x - radius, dst_rect->y, dst_rect->width + 2 * radius, 1};\n"
"  gfloat *src_buf    = gegl_malloc (read_rect.width * sizeof(gfloat) * 4);    \n"
"                                                                              \n"
"  for (v = 0; v < dst_rect->height; v++)                                      \n"
"    {                                                                         \n"
"      gint offset     = 0;                                                    \n"
"      read_rect.y     = dst_rect->y + v;                                      \n"
"      write_rect.y    = dst_rect->y + v;                                      \n"
"      gegl_buffer_get (src, &read_rect, 1.0, format, src_buf, GEGL_AUTO_ROWSTRIDE, GEGL_ABYSS_NONE);\n"
"                                                                              \n"
"      for (u = 0; u < dst_rect->width; u++)                                   \n"
"        {                                                                     \n"
"          fir_get_mean_pixel_1D (src_buf + offset,                            \n"
"                                 dst_buf + offset,                            \n"
"                                 4,                                           \n"
"                                 cmatrix,                                     \n"
"                                 matrix_length);                              \n"
"          offset += 4;                                                        \n"
"        }                                                                     \n"
"                                                                              \n"
"      gegl_buffer_set (dst, &write_rect, 0, format, dst_buf, GEGL_AUTO_ROWSTRIDE);\n"
"    }                                                                         \n"
"                                                                              \n"
"  gegl_free (src_buf);                                                        \n"
"  gegl_free (dst_buf);                                                        \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"fir_ver_blur (GeglBuffer          *src,                                       \n"
"              GeglBuffer          *dst,                                       \n"
"              const GeglRectangle *dst_rect,                                  \n"
"              gdouble             *cmatrix,                                   \n"
"              gint                 matrix_length)                             \n"
"{                                                                             \n"
"  gint        u,v;                                                            \n"
"  const gint  radius = matrix_length / 2;                                     \n"
"  const Babl *format = babl_format (\"RaGaBaA float\");                       \n"
"                                                                              \n"
"  GeglRectangle write_rect = {dst_rect->x, dst_rect->y, 1, dst_rect->height}; \n"
"  gfloat *dst_buf    = gegl_malloc (write_rect.height * sizeof(gfloat) * 4);  \n"
"                                                                              \n"
"  GeglRectangle read_rect  = {dst_rect->x, dst_rect->y - radius, 1, dst_rect->height + 2 * radius};\n"
"  gfloat *src_buf    = gegl_malloc (read_rect.height * sizeof(gfloat) * 4);   \n"
"                                                                              \n"
"  for (u = 0; u < dst_rect->width; u++)                                       \n"
"    {                                                                         \n"
"      gint offset     = 0;                                                    \n"
"      read_rect.x     = dst_rect->x + u;                                      \n"
"      write_rect.x    = dst_rect->x + u;                                      \n"
"      gegl_buffer_get (src, &read_rect, 1.0, format, src_buf, GEGL_AUTO_ROWSTRIDE, GEGL_ABYSS_NONE);\n"
"                                                                              \n"
"      for (v = 0; v < dst_rect->height; v++)                                  \n"
"        {                                                                     \n"
"          fir_get_mean_pixel_1D (src_buf + offset,                            \n"
"                                 dst_buf + offset,                            \n"
"                                 4,                                           \n"
"                                 cmatrix,                                     \n"
"                                 matrix_length);                              \n"
"          offset += 4;                                                        \n"
"        }                                                                     \n"
"                                                                              \n"
"      gegl_buffer_set (dst, &write_rect, 0, format, dst_buf, GEGL_AUTO_ROWSTRIDE);\n"
"    }                                                                         \n"
"                                                                              \n"
"  gegl_free (src_buf);                                                        \n"
"  gegl_free (dst_buf);                                                        \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"prepare (GeglOperation *operation)                                            \n"
"{                                                                             \n"
"  GeglOperationAreaFilter *area = GEGL_OPERATION_AREA_FILTER (operation);     \n"
"  GeglProperties          *o    = GEGL_PROPERTIES (operation);                \n"
"                                                                              \n"
"  gfloat fir_radius_x = fir_calc_convolve_matrix_length (o->std_dev_x) / 2;   \n"
"  gfloat fir_radius_y = fir_calc_convolve_matrix_length (o->std_dev_y) / 2;   \n"
"  gfloat iir_radius_x = o->std_dev_x * RADIUS_SCALE;                          \n"
"  gfloat iir_radius_y = o->std_dev_y * RADIUS_SCALE;                          \n"
"                                                                              \n"
"  /* XXX: these should be calculated exactly considering o->filter, but we just\n"
"   * make sure there is enough space */                                       \n"
"  area->left = area->right = ceil (MAX (fir_radius_x, iir_radius_x));         \n"
"  area->top = area->bottom = ceil (MAX (fir_radius_y, iir_radius_y));         \n"
"                                                                              \n"
"  gegl_operation_set_format (operation, \"input\",                            \n"
"                             babl_format (\"RaGaBaA float\"));                \n"
"  gegl_operation_set_format (operation, \"output\",                           \n"
"                             babl_format (\"RaGaBaA float\"));                \n"
"}                                                                             \n"
"                                                                              \n"
"#include \"opencl/gegl-cl.h\"                                                 \n"
"#include \"gegl-buffer-cl-iterator.h\"                                        \n"
"                                                                              \n"
"#include \"opencl/gaussian-blur.cl.h\"                                        \n"
"                                                                              \n"
"static GeglClRunData *cl_data = NULL;                                         \n"
"                                                                              \n"
"static gboolean                                                               \n"
"cl_gaussian_blur (cl_mem                in_tex,                               \n"
"                  cl_mem                out_tex,                              \n"
"                  cl_mem                aux_tex,                              \n"
"                  size_t                global_worksize,                      \n"
"                  const GeglRectangle  *roi,                                  \n"
"                  const GeglRectangle  *src_rect,                             \n"
"                  const GeglRectangle  *aux_rect,                             \n"
"                  gfloat               *dmatrix_x,                            \n"
"                  gint                  matrix_length_x,                      \n"
"                  gint                  xoff,                                 \n"
"                  gfloat               *dmatrix_y,                            \n"
"                  gint                  matrix_length_y,                      \n"
"                  gint                  yoff)                                 \n"
"{                                                                             \n"
"  cl_int cl_err = 0;                                                          \n"
"                                                                              \n"
"  size_t global_ws[2];                                                        \n"
"                                                                              \n"
"  cl_mem cl_matrix_x = NULL;                                                  \n"
"  cl_mem cl_matrix_y = NULL;                                                  \n"
"                                                                              \n"
"  if (!cl_data)                                                               \n"
"    {                                                                         \n"
"      const char *kernel_name[] = {\"fir_ver_blur\", \"fir_hor_blur\", NULL}; \n"
"      cl_data = gegl_cl_compile_and_build (gaussian_blur_cl_source, kernel_name);\n"
"    }                                                                         \n"
"                                                                              \n"
"  if (!cl_data)                                                               \n"
"    return TRUE;                                                              \n"
"                                                                              \n"
"  cl_matrix_x = gegl_clCreateBuffer (gegl_cl_get_context(),                   \n"
"                                     CL_MEM_COPY_HOST_PTR | CL_MEM_READ_ONLY, \n"
"                                     matrix_length_x * sizeof(cl_float), dmatrix_x, &cl_err);\n"
"  CL_CHECK;                                                                   \n"
"                                                                              \n"
"  cl_matrix_y = gegl_clCreateBuffer (gegl_cl_get_context(),                   \n"
"                                     CL_MEM_COPY_HOST_PTR | CL_MEM_READ_ONLY, \n"
"                                     matrix_length_y * sizeof(cl_float), dmatrix_y, &cl_err);\n"
"  CL_CHECK;                                                                   \n"
"                                                                              \n"
"  global_ws[0] = aux_rect->width;                                             \n"
"  global_ws[1] = aux_rect->height;                                            \n"
"                                                                              \n"
"  cl_err = gegl_cl_set_kernel_args (cl_data->kernel[1],                       \n"
"                                    sizeof(cl_mem), (void*)&in_tex,           \n"
"                                    sizeof(cl_int), (void*)&src_rect->width,  \n"
"                                    sizeof(cl_mem), (void*)&aux_tex,          \n"
"                                    sizeof(cl_mem), (void*)&cl_matrix_x,      \n"
"                                    sizeof(cl_int), (void*)&matrix_length_x,  \n"
"                                    sizeof(cl_int), (void*)&xoff,             \n"
"                                    NULL);                                    \n"
"  CL_CHECK;                                                                   \n"
"                                                                              \n"
"  cl_err = gegl_clEnqueueNDRangeKernel (gegl_cl_get_command_queue (),         \n"
"                                        cl_data->kernel[1], 2,                \n"
"                                        NULL, global_ws, NULL,                \n"
"                                        0, NULL, NULL);                       \n"
"  CL_CHECK;                                                                   \n"
"                                                                              \n"
"                                                                              \n"
"  global_ws[0] = roi->width;                                                  \n"
"  global_ws[1] = roi->height;                                                 \n"
"                                                                              \n"
"  cl_err = gegl_cl_set_kernel_args (cl_data->kernel[0],                       \n"
"                                    sizeof(cl_mem), (void*)&aux_tex,          \n"
"                                    sizeof(cl_int), (void*)&aux_rect->width,  \n"
"                                    sizeof(cl_mem), (void*)&out_tex,          \n"
"                                    sizeof(cl_mem), (void*)&cl_matrix_y,      \n"
"                                    sizeof(cl_int), (void*)&matrix_length_y,  \n"
"                                    sizeof(cl_int), (void*)&yoff,             \n"
"                                    NULL);                                    \n"
"  CL_CHECK;                                                                   \n"
"                                                                              \n"
"  cl_err = gegl_clEnqueueNDRangeKernel (gegl_cl_get_command_queue (),         \n"
"                                        cl_data->kernel[0], 2,                \n"
"                                        NULL, global_ws, NULL,                \n"
"                                        0, NULL, NULL);                       \n"
"  CL_CHECK;                                                                   \n"
"                                                                              \n"
"  cl_err = gegl_clFinish (gegl_cl_get_command_queue ());                      \n"
"  CL_CHECK;                                                                   \n"
"                                                                              \n"
"  cl_err = gegl_clReleaseMemObject (cl_matrix_x);                             \n"
"  CL_CHECK_ONLY (cl_err);                                                     \n"
"  cl_err = gegl_clReleaseMemObject (cl_matrix_y);                             \n"
"  CL_CHECK_ONLY (cl_err);                                                     \n"
"                                                                              \n"
"  return FALSE;                                                               \n"
"                                                                              \n"
"error:                                                                        \n"
"  if (cl_matrix_x)                                                            \n"
"    gegl_clReleaseMemObject (cl_matrix_x);                                    \n"
"  if (cl_matrix_y)                                                            \n"
"    gegl_clReleaseMemObject (cl_matrix_y);                                    \n"
"                                                                              \n"
"  return TRUE;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"static gboolean                                                               \n"
"cl_process (GeglOperation       *operation,                                   \n"
"            GeglBuffer          *input,                                       \n"
"            GeglBuffer          *output,                                      \n"
"            const GeglRectangle *result)                                      \n"
"{                                                                             \n"
"  const Babl *in_format  = gegl_operation_get_format (operation, \"input\");  \n"
"  const Babl *out_format = gegl_operation_get_format (operation, \"output\"); \n"
"  gint err = 0;                                                               \n"
"  gint j;                                                                     \n"
"                                                                              \n"
"  GeglOperationAreaFilter *op_area = GEGL_OPERATION_AREA_FILTER (operation);  \n"
"  GeglProperties          *o       = GEGL_PROPERTIES (operation);             \n"
"                                                                              \n"
"  gdouble *cmatrix_x, *cmatrix_y;                                             \n"
"  gint cmatrix_len_x, cmatrix_len_y;                                          \n"
"                                                                              \n"
"  gfloat *fmatrix_x, *fmatrix_y;                                              \n"
"                                                                              \n"
"  cmatrix_len_x = fir_gen_convolve_matrix (o->std_dev_x, &cmatrix_x);         \n"
"  cmatrix_len_y = fir_gen_convolve_matrix (o->std_dev_y, &cmatrix_y);         \n"
"                                                                              \n"
"  fmatrix_x = g_new (gfloat, cmatrix_len_x);                                  \n"
"  fmatrix_y = g_new (gfloat, cmatrix_len_y);                                  \n"
"                                                                              \n"
"  for(j=0; j<cmatrix_len_x; j++)                                              \n"
"    fmatrix_x[j] = (gfloat) cmatrix_x[j];                                     \n"
"                                                                              \n"
"  for(j=0; j<cmatrix_len_y; j++)                                              \n"
"    fmatrix_y[j] = (gfloat) cmatrix_y[j];                                     \n"
"                                                                              \n"
"  {                                                                           \n"
"  GeglBufferClIterator *i = gegl_buffer_cl_iterator_new (output,              \n"
"                                                         result,              \n"
"                                                         out_format,          \n"
"                                                         GEGL_CL_BUFFER_WRITE);\n"
"                                                                              \n"
"  gint read = gegl_buffer_cl_iterator_add_2 (i,                               \n"
"                                             input,                           \n"
"                                             result,                          \n"
"                                             in_format,                       \n"
"                                             GEGL_CL_BUFFER_READ,             \n"
"                                             op_area->left,                   \n"
"                                             op_area->right,                  \n"
"                                             op_area->top,                    \n"
"                                             op_area->bottom,                 \n"
"                                             GEGL_ABYSS_NONE);                \n"
"                                                                              \n"
"  gint aux = gegl_buffer_cl_iterator_add_aux (i,                              \n"
"                                              result,                         \n"
"                                              in_format,                      \n"
"                                              0,                              \n"
"                                              0,                              \n"
"                                              op_area->top,                   \n"
"                                              op_area->bottom);               \n"
"                                                                              \n"
"  while (gegl_buffer_cl_iterator_next (i, &err) && !err)                      \n"
"    {                                                                         \n"
"      err = cl_gaussian_blur(i->tex[read],                                    \n"
"                             i->tex[0],                                       \n"
"                             i->tex[aux],                                     \n"
"                             i->size[0],                                      \n"
"                             &i->roi[0],                                      \n"
"                             &i->roi[read],                                   \n"
"                             &i->roi[aux],                                    \n"
"                             fmatrix_x,                                       \n"
"                             cmatrix_len_x,                                   \n"
"                             op_area->left,                                   \n"
"                             fmatrix_y,                                       \n"
"                             cmatrix_len_y,                                   \n"
"                             op_area->top);                                   \n"
"                                                                              \n"
"      if (err)                                                                \n"
"        {                                                                     \n"
"          gegl_buffer_cl_iterator_stop (i);                                   \n"
"          break;                                                              \n"
"        }                                                                     \n"
"    }                                                                         \n"
"  }                                                                           \n"
"                                                                              \n"
"  g_free (fmatrix_x);                                                         \n"
"  g_free (fmatrix_y);                                                         \n"
"                                                                              \n"
"  g_free (cmatrix_x);                                                         \n"
"  g_free (cmatrix_y);                                                         \n"
"  return !err;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"static gboolean                                                               \n"
"process (GeglOperation       *operation,                                      \n"
"         GeglBuffer          *input,                                          \n"
"         GeglBuffer          *output,                                         \n"
"         const GeglRectangle *result,                                         \n"
"         gint                 level)                                          \n"
"{                                                                             \n"
"  GeglRectangle rect;                                                         \n"
"  GeglBuffer *temp;                                                           \n"
"  GeglOperationAreaFilter *op_area = GEGL_OPERATION_AREA_FILTER (operation);  \n"
"  GeglProperties          *o       = GEGL_PROPERTIES (operation);             \n"
"                                                                              \n"
"  GeglRectangle temp_extend;                                                  \n"
"  gdouble       B, b[4];                                                      \n"
"  gdouble      *cmatrix;                                                      \n"
"  gint          cmatrix_len;                                                  \n"
"  gboolean      horizontal_irr;                                               \n"
"  gboolean      vertical_irr;                                                 \n"
"                                                                              \n"
"  rect.x      = result->x - op_area->left;                                    \n"
"  rect.width  = result->width + op_area->left + op_area->right;               \n"
"  rect.y      = result->y - op_area->top;                                     \n"
"  rect.height = result->height + op_area->top + op_area->bottom;              \n"
"                                                                              \n"
"  if (o->filter == GEGL_GAUSSIAN_BLUR_FILTER_IIR)                             \n"
"    {                                                                         \n"
"      horizontal_irr = TRUE;                                                  \n"
"      vertical_irr   = TRUE;                                                  \n"
"    }                                                                         \n"
"  else if (o->filter == GEGL_GAUSSIAN_BLUR_FILTER_FIR)                        \n"
"    {                                                                         \n"
"      horizontal_irr = FALSE;                                                 \n"
"      vertical_irr   = FALSE;                                                 \n"
"    }                                                                         \n"
"  else /* GEGL_GAUSSIAN_BLUR_FILTER_AUTO */                                   \n"
"    {                                                                         \n"
"      horizontal_irr = o->std_dev_x > 1.0;                                    \n"
"      vertical_irr   = o->std_dev_y > 1.0;                                    \n"
"    }                                                                         \n"
"                                                                              \n"
"  if (gegl_operation_use_opencl (operation) && !(horizontal_irr | vertical_irr))\n"
"    if (cl_process(operation, input, output, result))                         \n"
"      return TRUE;                                                            \n"
"                                                                              \n"
"  gegl_rectangle_intersect (&temp_extend, &rect, gegl_buffer_get_extent (input));\n"
"  temp_extend.x      = result->x;                                             \n"
"  temp_extend.width  = result->width;                                         \n"
"  temp = gegl_buffer_new (&temp_extend, babl_format (\"RaGaBaA float\"));     \n"
"                                                                              \n"
"  if (horizontal_irr)                                                         \n"
"    {                                                                         \n"
"      iir_young_find_constants (o->std_dev_x, &B, b);                         \n"
"      iir_young_hor_blur (input, &rect, temp, &temp_extend, B, b);            \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      cmatrix_len = fir_gen_convolve_matrix (o->std_dev_x, &cmatrix);         \n"
"      fir_hor_blur (input, temp, &temp_extend, cmatrix, cmatrix_len);         \n"
"      g_free (cmatrix);                                                       \n"
"    }                                                                         \n"
"                                                                              \n"
"  if (vertical_irr)                                                           \n"
"    {                                                                         \n"
"      iir_young_find_constants (o->std_dev_y, &B, b);                         \n"
"      iir_young_ver_blur (temp, &rect, output, result, B, b);                 \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      cmatrix_len = fir_gen_convolve_matrix (o->std_dev_y, &cmatrix);         \n"
"      fir_ver_blur (temp, output, result, cmatrix, cmatrix_len);              \n"
"      g_free (cmatrix);                                                       \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_object_unref (temp);                                                      \n"
"  return  TRUE;                                                               \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static void                                                                   \n"
"gegl_op_class_init (GeglOpClass *klass)                                       \n"
"{                                                                             \n"
"  GeglOperationClass       *operation_class;                                  \n"
"  GeglOperationFilterClass *filter_class;                                     \n"
"                                                                              \n"
"  operation_class = GEGL_OPERATION_CLASS (klass);                             \n"
"  filter_class    = GEGL_OPERATION_FILTER_CLASS (klass);                      \n"
"                                                                              \n"
"  operation_class->prepare        = prepare;                                  \n"
"  operation_class->opencl_support = TRUE;                                     \n"
"                                                                              \n"
"  filter_class->process           = process;                                  \n"
"                                                                              \n"
"  gegl_operation_class_set_keys (operation_class,                             \n"
"    \"name\",        \"gegl:gaussian-blur-old\",                              \n"
"    \"title\",       _(\"Gaussian Blur\"),                                    \n"
"    \"categories\",  \"blur\",                                                \n"
"    \"description\", _(\"Each result pixel is the average of the neighbouring pixels weighted by a normal distribution with specified standard deviation.\"),\n"
"    NULL);                                                                    \n"
"}                                                                             \n"
"                                                                              \n"
"#endif                                                                        \n"
;
