static const char* op_c_source =
"/* This file is an image processing operation for GEGL                        \n"
" *                                                                            \n"
" * GEGL is free software; you can redistribute it and/or                      \n"
" * modify it under the terms of the GNU Lesser General Public                 \n"
" * License as published by the Free Software Foundation; either               \n"
" * version 3 of the License, or (at your option) any later version.           \n"
" *                                                                            \n"
" * GEGL is distributed in the hope that it will be useful,                    \n"
" * but WITHOUT ANY WARRANTY; without even the implied warranty of             \n"
" * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          \n"
" * Lesser General Public License for more details.                            \n"
" *                                                                            \n"
" * You should have received a copy of the GNU Lesser General Public           \n"
" * License along with GEGL; if not, see <http://www.gnu.org/licenses/>.       \n"
" *                                                                            \n"
" * Copyright        2010 Danny Robson      <danny@blubinc.net>                \n"
" * (pfscalibration) 2006 Grzegorz Krawczyk <gkrawczyk@users.sourceforge.net>  \n"
" */                                                                           \n"
"                                                                              \n"
"#include \"config.h\"                                                         \n"
"#include <glib/gi18n-lib.h>                                                   \n"
"                                                                              \n"
"#ifdef GEGL_PROPERTIES                                                        \n"
"                                                                              \n"
"property_string(exposures, _(\"Exposure values\"), \"\")                      \n"
"    description (_(\"Relative brightness of each exposure in EV\"))           \n"
"                                                                              \n"
"property_int (steps, _(\"Discretization bits\"), 13)                          \n"
"    description (_(\"Log2 of source's discretization steps\"))                \n"
"    value_range (8, 32)                                                       \n"
"                                                                              \n"
"property_double (sigma, _(\"Weight sigma\"), 8.0f)                            \n"
"    description (_(\"Weight distribution sigma controlling response contributions\"))\n"
"    value_range (0.0f, 32.0f)                                                 \n"
"                                                                              \n"
"#else                                                                         \n"
"                                                                              \n"
"/*#define DEBUG_LINEARIZE*/                                                   \n"
"/*#define DEBUG_SAVE_CURVES*/                                                 \n"
"                                                                              \n"
"#include <gegl-plugin.h>                                                      \n"
"struct _GeglOp                                                                \n"
"{                                                                             \n"
"  GeglOperationFilter parent_instance;                                        \n"
"  gpointer            properties;                                             \n"
"};                                                                            \n"
"                                                                              \n"
"typedef struct                                                                \n"
"{                                                                             \n"
"  GeglOperationFilterClass parent_class;                                      \n"
"} GeglOpClass;                                                                \n"
"                                                                              \n"
"                                                                              \n"
"#define GEGL_OP_C_SOURCE exp-combine.c                                        \n"
"#define GEGL_OP_NAME     exp_combine                                          \n"
"#include \"gegl-op.h\"                                                        \n"
"GEGL_DEFINE_DYNAMIC_OPERATION(GEGL_TYPE_OPERATION_FILTER)                     \n"
"                                                                              \n"
"#include <errno.h>                                                            \n"
"#include <math.h>                                                             \n"
"#include <stdio.h>                                                            \n"
"                                                                              \n"
"#include \"gegl-debug.h\"                                                     \n"
"#include \"graph/gegl-node-private.h\"                                        \n"
"#include \"graph/gegl-pad.h\"                                                 \n"
"                                                                              \n"
"static const gchar *PAD_FORMAT = \"R'G'B' float\";                            \n"
"static const gchar *EXP_PREFIX = \"exposure-\";                               \n"
"                                                                              \n"
"/* maximum iterations after algorithm accepts local minima */                 \n"
"static const guint MAXIT = 500;                                               \n"
"                                                                              \n"
"/* maximum accepted error */                                                  \n"
"static const gfloat MAX_DELTA  = 1e-5f;                                       \n"
"static const gfloat MIN_WEIGHT = 1e-3f;                                       \n"
"                                                                              \n"
"/* number of pixels used if downscaling for curve estimation */               \n"
"static const guint  MAX_SCALED_PIXELS = 1000 * 1000;                          \n"
"                                                                              \n"
"typedef enum                                                                  \n"
"{                                                                             \n"
"  PIXELS_ACTIVE,    /* Must be lowest valid ID, zero */                       \n"
"  PIXELS_FULL,                                                                \n"
"  PIXELS_SCALED,                                                              \n"
"                                                                              \n"
"  NUM_PIXEL_BUCKETS                                                           \n"
"} pixel_bucket;                                                               \n"
"                                                                              \n"
"                                                                              \n"
"typedef struct _exposure                                                      \n"
"{                                                                             \n"
"  /* Immediately higher and lower exposures */                                \n"
"  struct _exposure *hi;                                                       \n"
"  struct _exposure *lo;                                                       \n"
"                                                                              \n"
"  gfloat *pixels[NUM_PIXEL_BUCKETS];                                          \n"
"                                                                              \n"
"  /* 'Duration' of exposure. May only be relatively correct against the       \n"
"   * other exposures in sequence.                                             \n"
"   */                                                                         \n"
"  gfloat  ti;                                                                 \n"
"} exposure;                                                                   \n"
"                                                                              \n"
"                                                                              \n"
"static void                                                                   \n"
"gegl_expcombine_exposures_set_active (GSList       *exposures,                \n"
"                                      pixel_bucket  bucket)                   \n"
"{                                                                             \n"
"  g_return_if_fail (bucket > PIXELS_ACTIVE && bucket < NUM_PIXEL_BUCKETS);    \n"
"                                                                              \n"
"  for (; exposures; exposures = exposures->next)                              \n"
"    {                                                                         \n"
"      exposure *e              = exposures->data;                             \n"
"      e->pixels[PIXELS_ACTIVE] = e->pixels[bucket];                           \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* Comparator for struct exposure: order by ti */                             \n"
"static int                                                                    \n"
"gegl_expcombine_exposure_cmp (gconstpointer _a,                               \n"
"                              gconstpointer _b)                               \n"
"{                                                                             \n"
"  const exposure *a = _a,                                                     \n"
"                 *b = _b;                                                     \n"
"                                                                              \n"
"  if (a->ti > b->ti)                                                          \n"
"    return  1;                                                                \n"
"  else if (a->ti < b->ti)                                                     \n"
"    return -1;                                                                \n"
"  else                                                                        \n"
"    return  0;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static exposure*                                                              \n"
"gegl_expcombine_new_exposure (void)                                           \n"
"{                                                                             \n"
"  exposure *e = g_new (exposure, 1);                                          \n"
"  e->hi = e->lo = e;                                                          \n"
"                                                                              \n"
"  memset (e->pixels, 0, sizeof (e->pixels));                                  \n"
"  e->ti = NAN;                                                                \n"
"                                                                              \n"
"  return e;                                                                   \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static void                                                                   \n"
"gegl_expcombine_destroy_exposure (exposure *e)                                \n"
"{                                                                             \n"
"  guint i, j;                                                                 \n"
"                                                                              \n"
"  /* Unlink ourselves from the next hi and lo exposures */                    \n"
"  g_return_if_fail (e->lo);                                                   \n"
"  g_return_if_fail (e->hi);                                                   \n"
"                                                                              \n"
"  e->lo->hi = (e->hi == e) ? e->lo : e->hi;                                   \n"
"  e->hi->lo = (e->lo == e) ? e->hi : e->lo;                                   \n"
"                                                                              \n"
"  /* Free each pixel bucket. Each time a non-null is found, free it then null \n"
"   * any references to the same pixels in later buckets.                      \n"
"   */                                                                         \n"
"  for (i = PIXELS_ACTIVE + 1; i < NUM_PIXEL_BUCKETS; ++i)                     \n"
"    {                                                                         \n"
"      if (e->pixels[i])                                                       \n"
"        {                                                                     \n"
"          g_free (e->pixels[i]);                                              \n"
"                                                                              \n"
"          for (j = i + 1; j < NUM_PIXEL_BUCKETS; ++j)                         \n"
"              if (e->pixels[j] == e->pixels[i])                               \n"
"                  e->pixels[j] = NULL;                                        \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_free (e);                                                                 \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* Initialise a response curve to linear. The absolute values are probably    \n"
" * irrelevant as the curve should be normalised later.                        \n"
" */                                                                           \n"
"static void                                                                   \n"
"gegl_expcombine_response_linear (gfloat *response,                            \n"
"                                 guint   steps)                               \n"
"{                                                                             \n"
"  guint i;                                                                    \n"
"  for (i = 0; i < steps; ++i)                                                 \n"
"    response[i] = i / 255.0f;                                                 \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* Generate a weighting curve for the range of pixel values. Weights pixels in\n"
" * the centre of the range more highly than the extremes.                     \n"
" *                                                                            \n"
" * -sigma              weights the exponential arguments                      \n"
" * -step_min,step_max  describes the region we provide weighting for          \n"
" */                                                                           \n"
"static void                                                                   \n"
"gegl_expcombine_weights_gauss (gfloat *weights,                               \n"
"                               guint   steps,                                 \n"
"                               guint   step_min,                              \n"
"                               guint   step_max,                              \n"
"                               gfloat  sigma)                                 \n"
"{                                                                             \n"
"  gfloat step_mid1, step_mid2;                                                \n"
"  gfloat weight;                                                              \n"
"  guint i;                                                                    \n"
"                                                                              \n"
"  g_return_if_fail (weights);                                                 \n"
"  g_return_if_fail (step_min <= step_max);                                    \n"
"  g_return_if_fail (step_min <  steps);                                       \n"
"  g_return_if_fail (step_max <  steps);                                       \n"
"                                                                              \n"
"  step_mid1 = step_min + (step_max - step_min) / 2.0f - 0.5f;                 \n"
"  step_mid2 = (step_mid1 - step_min) * (step_mid1 - step_min);                \n"
"                                                                              \n"
"  for (i = 0; i < steps; ++i)                                                 \n"
"    {                                                                         \n"
"      if (i < step_min || i > step_max)                                       \n"
"        {                                                                     \n"
"          weights[i] = 0.0f;                                                  \n"
"          continue;                                                           \n"
"        }                                                                     \n"
"                                                                              \n"
"      /* gkrawczyk: that's not really a Gaussian, but equation is             \n"
"       * taken from Robertson02 paper.                                        \n"
"       */                                                                     \n"
"      weight = exp ( -sigma                 *                                 \n"
"                    ((gfloat)i - step_mid1) *                                 \n"
"                    ((gfloat)i - step_mid1) /                                 \n"
"                     step_mid2);                                              \n"
"                                                                              \n"
"      /* ignore very low weights. we rely on these weights being set to zero  \n"
"       * in the response estimation code, thus this should not be removed.    \n"
"       */                                                                     \n"
"      if (weight < MIN_WEIGHT)                                                \n"
"        weights[i] = 0.0f;                                                    \n"
"      else                                                                    \n"
"        weights[i] = weight;                                                  \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* Normalise the response curve, in place, using the midpoint of the non-zero \n"
" * region of the response curve. Returns the mid-point value of the curve.    \n"
" */                                                                           \n"
"static gfloat                                                                 \n"
"gegl_expcombine_normalize (gfloat *response,                                  \n"
"                           guint   steps)                                     \n"
"{                                                                             \n"
"  guint  step_min, step_max, step_mid;                                        \n"
"  guint  i;                                                                   \n"
"  gfloat val_mid;                                                             \n"
"                                                                              \n"
"  g_return_val_if_fail (response, NAN);                                       \n"
"  g_return_val_if_fail (steps > 0, NAN);                                      \n"
"                                                                              \n"
"  /* Find the first and last non-zero values in response curve */             \n"
"  for (step_min = 0;                                                          \n"
"       step_min < steps && response[step_min] == 0;                           \n"
"       ++step_min)                                                            \n"
"    ;                                                                         \n"
"  for (step_max = steps - 1;                                                  \n"
"       step_max > 0 && response[step_max] == 0;                               \n"
"       --step_max)                                                            \n"
"    ;                                                                         \n"
"                                                                              \n"
"  g_return_val_if_fail (step_max >= step_min, NAN);                           \n"
"  step_mid = step_min + (step_max - step_min) / 2;                            \n"
"                                                                              \n"
"  /* Find the non-zero mid-value of the response curve */                     \n"
"  val_mid = response[step_mid];                                               \n"
"  if (val_mid == 0.0f)                                                        \n"
"  {                                                                           \n"
"    /* find first non-zero middle response */                                 \n"
"    while (step_mid < step_max && response[step_mid] == 0.0f)                 \n"
"        ++step_mid;                                                           \n"
"    val_mid = response[step_mid];                                             \n"
"  }                                                                           \n"
"                                                                              \n"
"  /* Normalize response curve values via the mid-value */                     \n"
"  g_return_val_if_fail (val_mid != 0.0f, 0.0f);                               \n"
"                                                                              \n"
"  for (i = 0; i < steps; ++i)                                                 \n"
"      response[i] /= val_mid;                                                 \n"
"                                                                              \n"
"  return val_mid;                                                             \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* Apply debevec model for response curve application. Does not suffer from   \n"
" * apparent discolouration in some areas of image reconstruction when         \n"
" * compared with robertson.                                                   \n"
" */                                                                           \n"
"static int                                                                    \n"
"gegl_expcombine_apply_debevec  (gfloat              *hdr,                     \n"
"                                const guint          components,              \n"
"                                GSList              *imgs,                    \n"
"                                gfloat             **response,                \n"
"                                const gfloat        *weighting,               \n"
"                                const guint          steps,                   \n"
"                                const GeglRectangle *extent)                  \n"
"{                                                                             \n"
"  guint  step_min, step_max;                                                  \n"
"  guint  num_imgs    = g_slist_length (imgs);                                 \n"
"  guint  saturated   = 0;                                                     \n"
"  guint  pixel_count = (guint)(extent->width * extent->height);               \n"
"  guint  i, j;                                                                \n"
"                                                                              \n"
"  g_return_val_if_fail (hdr,                        G_MAXINT);                \n"
"  g_return_val_if_fail (g_slist_length (imgs) > 0,  G_MAXINT);                \n"
"  g_return_val_if_fail (response,                   G_MAXINT);                \n"
"  g_return_val_if_fail (weighting,                  G_MAXINT);                \n"
"  g_return_val_if_fail (steps > 0,                  G_MAXINT);                \n"
"  g_return_val_if_fail (extent,                     G_MAXINT);                \n"
"  g_return_val_if_fail (extent->width  > 0,         G_MAXINT);                \n"
"  g_return_val_if_fail (extent->height > 0,         G_MAXINT);                \n"
"                                                                              \n"
"  /* anti saturation: calculate trusted camera output range */                \n"
"  for (step_min = 0, i = step_min; i <    steps; ++i)                         \n"
"    if (weighting[i] > 0)                                                     \n"
"      {                                                                       \n"
"        step_min = i;                                                         \n"
"        break;                                                                \n"
"      }                                                                       \n"
"  for (step_max = steps - 1, i = step_max; i > step_min; --i)                 \n"
"    if (weighting[i] > 0)                                                     \n"
"      {                                                                       \n"
"        step_max = i;                                                         \n"
"        break;                                                                \n"
"      }                                                                       \n"
"  g_return_val_if_fail (step_max >= step_min, G_MAXINT);                      \n"
"                                                                              \n"
"  for (j = 0; j < pixel_count; ++j)                                           \n"
"    {                                                                         \n"
"      gfloat  sum[3]  = { 0.0f, 0.0f, 0.0f },                                 \n"
"              div     = 0.0f;                                                 \n"
"      gfloat  ti_max  = G_MINFLOAT,                                           \n"
"              ti_min  = G_MAXFLOAT;                                           \n"
"      gfloat  average;                                                        \n"
"      guint   white_step[3] = {0,0,0}, black_step[3] = {0,0,0};               \n"
"                                                                              \n"
"      /* all exposures for each pixel */                                      \n"
"      for (i = 0; i < num_imgs; ++i)                                          \n"
"        {                                                                     \n"
"          exposure *exp_i;                                                    \n"
"          guint     step[3], step_hi[3], step_lo[3];                          \n"
"                                                                              \n"
"          exp_i    = g_slist_nth_data (imgs, i);                              \n"
"          step[0]  = exp_i->pixels[PIXELS_ACTIVE][0 + j * components];        \n"
"          step[1]  = exp_i->pixels[PIXELS_ACTIVE][1 + j * components];        \n"
"          step[2]  = exp_i->pixels[PIXELS_ACTIVE][2 + j * components];        \n"
"          g_return_val_if_fail (step[0] < steps && step[1] < steps && step[2] < steps, G_MAXINT);\n"
"                                                                              \n"
"          /* anti saturation: observe minimum exposure time at which saturated\n"
"           * value is present, and maximum exp time at which black value is   \n"
"           * present                                                          \n"
"           */                                                                 \n"
"          if ((step[0] > step_max ||                                          \n"
"               step[1] > step_max ||                                          \n"
"               step[2] > step_max) && exp_i->ti < ti_min)                     \n"
"            {                                                                 \n"
"              white_step[0] = MIN (step[0], steps);                           \n"
"              white_step[1] = MIN (step[1], steps);                           \n"
"              white_step[2] = MIN (step[2], steps);                           \n"
"              ti_min        = exp_i->ti;                                      \n"
"            }                                                                 \n"
"                                                                              \n"
"          if ((step[0] < step_min ||                                          \n"
"               step[1] < step_min ||                                          \n"
"               step[2] < step_min) && exp_i->ti > ti_max)                     \n"
"            {                                                                 \n"
"              /* This could be protected by a max (0, step), but we're using  \n"
"               * unsigned ints so it would be worthless currently.            \n"
"               */                                                             \n"
"              black_step[0] = step[0];                                        \n"
"              black_step[1] = step[1];                                        \n"
"              black_step[2] = step[2];                                        \n"
"              ti_max        = exp_i->ti;                                      \n"
"            }                                                                 \n"
"                                                                              \n"
"          /* anti ghosting: monotonous increase in time should result in      \n"
"           * monotonous increase in intensity; make forward and backward check,\n"
"           * ignore value if condition not satisfied                          \n"
"           */                                                                 \n"
"          step_lo[0] = exp_i->lo->pixels[PIXELS_ACTIVE][0 + j * components];  \n"
"          step_lo[1] = exp_i->lo->pixels[PIXELS_ACTIVE][1 + j * components];  \n"
"          step_lo[2] = exp_i->lo->pixels[PIXELS_ACTIVE][2 + j * components];  \n"
"          g_return_val_if_fail (step_lo[0] < steps &&                         \n"
"                                step_lo[1] < steps &&                         \n"
"                                step_lo[2] < steps, G_MAXINT);                \n"
"                                                                              \n"
"          step_hi[0] = exp_i->hi->pixels[PIXELS_ACTIVE][0 + j * components];  \n"
"          step_hi[1] = exp_i->hi->pixels[PIXELS_ACTIVE][1 + j * components];  \n"
"          step_hi[2] = exp_i->hi->pixels[PIXELS_ACTIVE][2 + j * components];  \n"
"          g_return_val_if_fail (step_hi[0] < steps &&                         \n"
"                                step_hi[1] < steps &&                         \n"
"                                step_hi[2] < steps, G_MAXINT);                \n"
"                                                                              \n"
"          if (step_lo[0] > step[0] ||                                         \n"
"              step_lo[1] > step[1] ||                                         \n"
"              step_lo[2] > step[2])                                           \n"
"            {                                                                 \n"
"              white_step[0] = step[0];                                        \n"
"              white_step[1] = step[1];                                        \n"
"              white_step[2] = step[2];                                        \n"
"              /* TODO: This is not an fminf in luminance. Should it be? */    \n"
"              /* ti_min        = fminf (exp_i->ti, ti_min); */                \n"
"              continue;                                                       \n"
"            }                                                                 \n"
"                                                                              \n"
"          if (step_hi[0] < step[0] ||                                         \n"
"              step_hi[1] < step[1] ||                                         \n"
"              step_hi[2] < step[2])                                           \n"
"            {                                                                 \n"
"              black_step[0] = step[0];                                        \n"
"              black_step[1] = step[1];                                        \n"
"              black_step[2] = step[2];                                        \n"
"              /* TODO: This is not an fminf in luminance. Should it be? */    \n"
"              /* ti_max        = fmaxf (exp_i->ti, ti_max); */                \n"
"              continue;                                                       \n"
"            }                                                                 \n"
"                                                                              \n"
"          /* Weight the addition into each channel by the average weight of   \n"
"           * the channels and the exposure duration. This is the key          \n"
"           * difference to robertson.                                         \n"
"           */                                                                 \n"
"          average  = weighting [step[0]] +                                    \n"
"                     weighting [step[1]] +                                    \n"
"                     weighting [step[2]];                                     \n"
"          average /= 3.0f;                                                    \n"
"                                                                              \n"
"          sum[0] += response[0][step[0]] * average / exp_i->ti;               \n"
"          sum[1] += response[1][step[1]] * average / exp_i->ti;               \n"
"          sum[2] += response[2][step[2]] * average / exp_i->ti;               \n"
"                                                                              \n"
"          div += average;                                                     \n"
"        }                                                                     \n"
"                                                                              \n"
"      g_return_val_if_fail (sum[0] >= 0.0f, G_MAXINT);                        \n"
"      g_return_val_if_fail (sum[1] >= 0.0f, G_MAXINT);                        \n"
"      g_return_val_if_fail (sum[2] >= 0.0f, G_MAXINT);                        \n"
"      g_return_val_if_fail (   div >= 0.0f, G_MAXINT);                        \n"
"      g_return_val_if_fail (ti_max <= ti_min, G_MAXINT);                      \n"
"                                                                              \n"
"      /* anti saturation: if a meaningful representation of pixel was not     \n"
"       * found, replace it with information from observed data                \n"
"       */                                                                     \n"
"      if (div == 0.0f)                                                        \n"
"          ++saturated;                                                        \n"
"                                                                              \n"
"      if (div == 0.0f && ti_max != G_MINFLOAT)                                \n"
"        {                                                                     \n"
"          sum[0] = response[0][black_step[0]] / ti_max;                       \n"
"          sum[1] = response[1][black_step[1]] / ti_max;                       \n"
"          sum[2] = response[2][black_step[2]] / ti_max;                       \n"
"          div = 1.0f;                                                         \n"
"        }                                                                     \n"
"      else if (div == 0.0f && ti_min != G_MAXFLOAT)                           \n"
"        {                                                                     \n"
"          sum[0] = response[0][white_step[0]] / ti_min;                       \n"
"          sum[1] = response[1][white_step[1]] / ti_min;                       \n"
"          sum[2] = response[2][white_step[2]] / ti_min;                       \n"
"          div = 1.0f;                                                         \n"
"        }                                                                     \n"
"                                                                              \n"
"      if (div == 0.0f)                                                        \n"
"        {                                                                     \n"
"          hdr[0 + j * components] = 0.0f;                                     \n"
"          hdr[1 + j * components] = 0.0f;                                     \n"
"          hdr[2 + j * components] = 0.0f;                                     \n"
"        }                                                                     \n"
"      else                                                                    \n"
"        {                                                                     \n"
"          hdr[0 + j * components] = sum[0] / div;                             \n"
"          hdr[1 + j * components] = sum[1] / div;                             \n"
"          hdr[2 + j * components] = sum[2] / div;                             \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  return saturated;                                                           \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/**                                                                           \n"
" * @brief Create HDR image by applying response curve to given images         \n"
" * taken with different exposures                                             \n"
" *                                                                            \n"
" * @param hdr [out] HDR image                                                 \n"
" * @param imgs      list of scene exposures                                   \n"
" * @param response  camera response function (array size of `steps')          \n"
" * @param weighting weighting function for camera output values (array size of `steps')\n"
" * @param steps     number of camera output levels                            \n"
" * @return          number of saturated pixels in the HDR image (0: all OK)   \n"
" */                                                                           \n"
"static int                                                                    \n"
"gegl_expcombine_apply_response (gfloat              *hdr,                     \n"
"                                const guint          offset,                  \n"
"                                const guint          components,              \n"
"                                GSList              *imgs,                    \n"
"                                const gfloat        *response,                \n"
"                                const gfloat        *weighting,               \n"
"                                const guint          steps,                   \n"
"                                const GeglRectangle *extent)                  \n"
"{                                                                             \n"
"  guint  step_min, step_max;                                                  \n"
"  guint  num_imgs  = g_slist_length (imgs);                                   \n"
"  guint  saturated = 0;                                                       \n"
"  guint  pixel_count = (guint)(extent->width * extent->height);               \n"
"  guint  i, j;                                                                \n"
"                                                                              \n"
"  g_return_val_if_fail (hdr,                        G_MAXINT);                \n"
"  g_return_val_if_fail (g_slist_length (imgs) > 0,  G_MAXINT);                \n"
"  g_return_val_if_fail (response,                   G_MAXINT);                \n"
"  g_return_val_if_fail (weighting,                  G_MAXINT);                \n"
"  g_return_val_if_fail (steps > 0,                  G_MAXINT);                \n"
"  g_return_val_if_fail (extent,                     G_MAXINT);                \n"
"  g_return_val_if_fail (extent->width  > 0,         G_MAXINT);                \n"
"  g_return_val_if_fail (extent->height > 0,         G_MAXINT);                \n"
"                                                                              \n"
"  /* anti saturation: calculate trusted camera output range */                \n"
"  for (step_min = 0, i = step_min; i < steps; ++i)                            \n"
"    {                                                                         \n"
"      if (weighting[i] > 0)                                                   \n"
"        {                                                                     \n"
"          step_min = i;                                                       \n"
"          break;                                                              \n"
"        }                                                                     \n"
"    }                                                                         \n"
"  for (step_max = steps - 1, i = step_max; i > step_min; --i)                 \n"
"    {                                                                         \n"
"      if (weighting[i] > 0)                                                   \n"
"        {                                                                     \n"
"          step_max = i;                                                       \n"
"          break;                                                              \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_return_val_if_fail (step_max >= step_min, G_MAXINT);                      \n"
"                                                                              \n"
"  for (j = 0; j < pixel_count; ++j)                                           \n"
"    {                                                                         \n"
"      gfloat  sum    = 0.0f,                                                  \n"
"              div    = 0.0f;                                                  \n"
"      gfloat  ti_max = G_MINFLOAT,                                            \n"
"              ti_min = G_MAXFLOAT;                                            \n"
"                                                                              \n"
"      /* all exposures for each pixel */                                      \n"
"      for (i = 0; i < num_imgs; ++i)                                          \n"
"        {                                                                     \n"
"          exposure *exp_i;                                                    \n"
"          guint     step, step_hi, step_lo;                                   \n"
"                                                                              \n"
"          exp_i = g_slist_nth_data (imgs, i);                                 \n"
"          step  = exp_i->pixels[PIXELS_ACTIVE][offset + j * components];      \n"
"          g_return_val_if_fail (step < steps, G_MAXINT);                      \n"
"                                                                              \n"
"          /* anti saturation: observe minimum exposure time at which saturated\n"
"           * value is present, and maximum exp time at which black value is   \n"
"           * present                                                          \n"
"           */                                                                 \n"
"          if (step > step_max)                                                \n"
"              ti_min = fminf (ti_min, exp_i->ti);                             \n"
"          if (step < step_min)                                                \n"
"              ti_max = fmaxf (ti_max, exp_i->ti);                             \n"
"                                                                              \n"
"          /* anti ghosting: monotonous increase in time should result in      \n"
"           * monotonous increase in intensity; make forward and backward check,\n"
"           * ignore value if condition not satisfied                          \n"
"           */                                                                 \n"
"          step_lo = exp_i->lo->pixels[PIXELS_ACTIVE][offset + j * components];\n"
"          step_hi = exp_i->hi->pixels[PIXELS_ACTIVE][offset + j * components];\n"
"                                                                              \n"
"          if (step_lo > step || step_hi < step)                               \n"
"               continue;                                                      \n"
"                                                                              \n"
"          sum += weighting[step] * exp_i->ti * response[step];                \n"
"          div += weighting[step] * exp_i->ti * exp_i->ti;                     \n"
"        }                                                                     \n"
"                                                                              \n"
"      g_return_val_if_fail (sum >= 0.0f, G_MAXINT);                           \n"
"      g_return_val_if_fail (div >= 0.0f, G_MAXINT);                           \n"
"      g_return_val_if_fail (ti_max <= ti_min, G_MAXINT);                      \n"
"                                                                              \n"
"      /* anti saturation: if a meaningful representation of pixel was not     \n"
"       * found, replace it with information from observed data                \n"
"       */                                                                     \n"
"      if (div == 0.0f)                                                        \n"
"          ++saturated;                                                        \n"
"                                                                              \n"
"      if (div == 0.0f && ti_max != G_MINFLOAT)                                \n"
"        {                                                                     \n"
"          sum = response[step_min];                                           \n"
"          div = ti_max;                                                       \n"
"        }                                                                     \n"
"                                                                              \n"
"      if (div == 0.0f && ti_min != G_MAXFLOAT)                                \n"
"        {                                                                     \n"
"          sum = response[step_max];                                           \n"
"          div = ti_min;                                                       \n"
"        }                                                                     \n"
"                                                                              \n"
"      if (div != 0.0f)                                                        \n"
"          hdr[offset + j * components] = sum / div;                           \n"
"      else                                                                    \n"
"          hdr[offset + j * components] = 0.0f;                                \n"
"    }                                                                         \n"
"                                                                              \n"
"  return saturated;                                                           \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/**                                                                           \n"
" * @brief Calculate camera response using Robertson02 algorithm               \n"
" *                                                                            \n"
" * @param luminance [out] estimated luminance values                          \n"
" * @param imgs            list of scene exposures                             \n"
" * @param response  [out] array to put response function                      \n"
" * @param weighting       weights                                             \n"
" * @param steps           max camera output (no of discrete steps)            \n"
" * @return                number of saturated pixels in the HDR image (0: all OK)\n"
" */                                                                           \n"
"                                                                              \n"
"static int                                                                    \n"
"gegl_expcombine_get_response (gfloat              *hdr,                       \n"
"                              const guint          offset,                    \n"
"                              const guint          components,                \n"
"                              GSList              *imgs,                      \n"
"                              gfloat              *response,                  \n"
"                              const gfloat        *weighting,                 \n"
"                              guint                steps,                     \n"
"                              const GeglRectangle *extent)                    \n"
"{                                                                             \n"
"  gfloat  *response_old;                                                      \n"
"  gfloat   delta, delta_old;                                                  \n"
"                                                                              \n"
"  gulong  *card;                                                              \n"
"  gfloat  *sum;                                                               \n"
"                                                                              \n"
"  guint    i, j, hits;                                                        \n"
"  guint    iterations;                                                        \n"
"  guint    pixel_count = (guint)(extent->height * extent->width);             \n"
"  gulong   saturated  = 0;                                                    \n"
"  gboolean converged;                                                         \n"
"                                                                              \n"
"  g_return_val_if_fail (hdr,                        G_MAXINT);                \n"
"  g_return_val_if_fail (g_slist_length (imgs) > 1,  G_MAXINT);                \n"
"  g_return_val_if_fail (response,                   G_MAXINT);                \n"
"  g_return_val_if_fail (weighting,                  G_MAXINT);                \n"
"  g_return_val_if_fail (steps > 0,                  G_MAXINT);                \n"
"  g_return_val_if_fail (extent,                     G_MAXINT);                \n"
"  g_return_val_if_fail (extent->width  > 0,         G_MAXINT);                \n"
"  g_return_val_if_fail (extent->height > 0,         G_MAXINT);                \n"
"                                                                              \n"
"  response_old = g_new (gfloat, steps);                                       \n"
"                                                                              \n"
"  /* 0. Initialization */                                                     \n"
"  gegl_expcombine_normalize (response, steps);                                \n"
"  for (i = 0; i < steps; ++i)                                                 \n"
"      response_old[i] = response[i];                                          \n"
"                                                                              \n"
"  saturated = gegl_expcombine_apply_response (hdr, offset, components, imgs,  \n"
"                                              response, weighting, steps,     \n"
"                                              extent);                        \n"
"                                                                              \n"
"  converged  = FALSE;                                                         \n"
"  card       = g_new (gulong, steps);                                         \n"
"  sum        = g_new (gfloat, steps);                                         \n"
"  iterations = 0;                                                             \n"
"  delta_old  = 0.0f;                                                          \n"
"                                                                              \n"
"  /* Optimization process */                                                  \n"
"  while (!converged)                                                          \n"
"  {                                                                           \n"
"    GSList *cursor = imgs;                                                    \n"
"    /* 1. Minimize with respect to response */                                \n"
"    for (i = 0; i < steps; ++i)                                               \n"
"      {                                                                       \n"
"        card[i] = 0;                                                          \n"
"        sum [i] = 0.0f;                                                       \n"
"      }                                                                       \n"
"                                                                              \n"
"    for (cursor = imgs; cursor; cursor = cursor->next)                        \n"
"      {                                                                       \n"
"        exposure *e = cursor->data;                                           \n"
"                                                                              \n"
"        for (j = 0; j < pixel_count; ++j)                                     \n"
"          {                                                                   \n"
"            guint step = e->pixels[PIXELS_ACTIVE][offset + j * components];   \n"
"            if (step < steps)                                                 \n"
"              {                                                               \n"
"                sum[step] += e->ti * hdr[offset + j * components];            \n"
"                ++card[step];                                                 \n"
"              }                                                               \n"
"            else                                                              \n"
"              g_warning (\"robertson02: m out of range: %u\", step);          \n"
"          }                                                                   \n"
"      }                                                                       \n"
"                                                                              \n"
"    for (i = 0; i < steps; ++i)                                               \n"
"      {                                                                       \n"
"        if (card[i] != 0)                                                     \n"
"          response[i] = sum[i] / card[i];                                     \n"
"        else                                                                  \n"
"          response[i] = 0.0f;                                                 \n"
"      }                                                                       \n"
"                                                                              \n"
"    /* 2. Apply new response */                                               \n"
"    gegl_expcombine_normalize (response, steps);                              \n"
"    saturated = gegl_expcombine_apply_response (hdr, offset, components, imgs,\n"
"                                                response, weighting, steps,   \n"
"                                                extent);                      \n"
"                                                                              \n"
"    /* 3. Check stopping condition */                                         \n"
"    delta = 0.0f;                                                             \n"
"    hits  = 0;                                                                \n"
"                                                                              \n"
"    for (i = 0; i < steps; ++i)                                               \n"
"      {                                                                       \n"
"        g_return_val_if_fail (response[i] >= 0, G_MAXINT);                    \n"
"                                                                              \n"
"        if (response[i] != 0.0f)                                              \n"
"          {                                                                   \n"
"            gdouble diff     = response[i] - response_old[i];                 \n"
"            delta           += diff * diff;                                   \n"
"            response_old[i]  = response[i];                                   \n"
"            ++hits;                                                           \n"
"          }                                                                   \n"
"      }                                                                       \n"
"    delta /= hits;                                                            \n"
"                                                                              \n"
"                                                                              \n"
"    GEGL_NOTE (GEGL_DEBUG_PROCESS,                                            \n"
"               \"exp-combine convergence, #%u delta=%f (coverage: %u%%)\",    \n"
"               iterations,                                                    \n"
"               delta,                                                         \n"
"               (guint)(100.0f * (gfloat)hits / steps));                       \n"
"    if (delta < MAX_DELTA)                                                    \n"
"      converged = TRUE;                                                       \n"
"    else if (isnan (delta) || (iterations > MAXIT && delta_old < delta))      \n"
"      {                                                                       \n"
"        g_warning (\"exp-combine failed to converge. too noisy data in range.\");\n"
"        break;                                                                \n"
"      }                                                                       \n"
"                                                                              \n"
"    delta_old = delta;                                                        \n"
"    ++iterations;                                                             \n"
"  }                                                                           \n"
"                                                                              \n"
"  g_free (response_old);                                                      \n"
"  g_free (card);                                                              \n"
"  g_free (sum);                                                               \n"
"                                                                              \n"
"  return saturated;                                                           \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* All exposure pads have the common prefix, followed by a numeric            \n"
" * identifier.                                                                \n"
" */                                                                           \n"
"static inline gboolean                                                        \n"
"gegl_expcombine_is_exposure_padname (const gchar *p)                          \n"
"{                                                                             \n"
"  return g_str_has_prefix (p, EXP_PREFIX);                                    \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static inline gboolean                                                        \n"
"gegl_expcombine_is_exposure_pad (GeglPad *p)                                  \n"
"{                                                                             \n"
"  return gegl_expcombine_is_exposure_padname (gegl_pad_get_name (p));         \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* XXX: We create a large quantity of pads for all the exposures, hoping we   \n"
" * create sufficient numbers and they don't utilise too many resources. This  \n"
" * is a bad hack, but works for the time being...                             \n"
" */                                                                           \n"
"static void                                                                   \n"
"gegl_expcombine_attach (GeglOperation *operation)                             \n"
"{                                                                             \n"
"  GParamSpec  *pspec;                                                         \n"
"  gchar       padname[16];                                                    \n"
"  guint       i;                                                              \n"
"                                                                              \n"
"  pspec = g_param_spec_object (\"output\",                                    \n"
"                               \"output\",                                    \n"
"                               \"Output buffer\",                             \n"
"                               GEGL_TYPE_BUFFER,                              \n"
"                               G_PARAM_READWRITE |                            \n"
"                               GEGL_PARAM_PAD_OUTPUT);                        \n"
"                                                                              \n"
"  gegl_operation_create_pad (operation, pspec);                               \n"
"  g_param_spec_sink (pspec);                                                  \n"
"                                                                              \n"
"  for (i = 0; i <= 99; ++i)                                                   \n"
"    {                                                                         \n"
"      snprintf (padname, G_N_ELEMENTS (padname), \"exposure_%u\", i);         \n"
"                                                                              \n"
"      pspec = g_param_spec_object (padname,                                   \n"
"                                   padname,                                   \n"
"                                   \"Exposure input.\",                       \n"
"                                   GEGL_TYPE_BUFFER,                          \n"
"                                   G_PARAM_READWRITE |                        \n"
"                                   GEGL_PARAM_PAD_INPUT);                     \n"
"                                                                              \n"
"      gegl_operation_create_pad (operation, pspec);                           \n"
"      g_param_spec_sink (pspec);                                              \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"                                                                              \n"
"static void                                                                   \n"
"gegl_expcombine_prepare (GeglOperation *operation)                            \n"
"{                                                                             \n"
"  GSList *inputs = gegl_node_get_input_pads (operation->node);                \n"
"                                                                              \n"
"  /* Set all the pads output formats */                                       \n"
"  for (; inputs; inputs = inputs->next)                                       \n"
"    {                                                                         \n"
"      GeglPad *pad = inputs->data;                                            \n"
"                                                                              \n"
"      gegl_pad_set_format (pad, babl_format (PAD_FORMAT));                    \n"
"    }                                                                         \n"
"                                                                              \n"
"  gegl_operation_set_format (operation, \"output\", babl_format (PAD_FORMAT));\n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* Order pads beginning with EXP_PREFIX by their trailing exposure value.     \n"
" * Non-exposure pads come first, then exposure pads without an EV, then       \n"
" * ordered exposure pads.                                                     \n"
" */                                                                           \n"
"static int                                                                    \n"
"gegl_expcombine_pad_cmp (gconstpointer _a, gconstpointer _b)                  \n"
"{                                                                             \n"
"  const gchar *a = gegl_pad_get_name (GEGL_PAD (_a)),                         \n"
"              *b = gegl_pad_get_name (GEGL_PAD (_b));                         \n"
"  guint64      x, y;                                                          \n"
"                                                                              \n"
"  if (!g_str_has_prefix (b, EXP_PREFIX)) return  1;                           \n"
"  if (!g_str_has_prefix (a, EXP_PREFIX)) return -1;                           \n"
"                                                                              \n"
"  a = strrchr (a, '-');                                                       \n"
"  b = strrchr (b, '-');                                                       \n"
"                                                                              \n"
"  g_return_val_if_fail (b, -1);                                               \n"
"  g_return_val_if_fail (a, -1);                                               \n"
"                                                                              \n"
"  y = g_ascii_strtoll (b + 1, NULL, 10);                                      \n"
"  if (errno) return  1;                                                       \n"
"  x = g_ascii_strtoll (a + 1, NULL, 10);                                      \n"
"  if (errno) return -1;                                                       \n"
"                                                                              \n"
"  if (x < y) return -1;                                                       \n"
"  if (x > y) return  1;                                                       \n"
"  return 0;                                                                   \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* Extract a list of exposure pixel and metadata from the operation context.  \n"
" * The data is not guaranteed to be ordered within the list. May return NULL  \n"
" * for error.                                                                 \n"
" */                                                                           \n"
"static GSList *                                                               \n"
"gegl_expcombine_get_exposures (GeglOperation        *operation,               \n"
"                               GeglOperationContext *context,                 \n"
"                               const GeglRectangle  *full_roi,                \n"
"                               GeglRectangle        *scaled_roi)              \n"
"{                                                                             \n"
"  GeglProperties    *o          = GEGL_PROPERTIES (operation);                \n"
"  gchar         *ev_cursor  = o->exposures;                                   \n"
"  GSList        *exposures  = NULL,                                           \n"
"                *inputs,                                                      \n"
"                *cursor;                                                      \n"
"  guint          components = babl_format_get_n_components (babl_format (PAD_FORMAT));\n"
"  gfloat         scale;                                                       \n"
"                                                                              \n"
"  g_return_val_if_fail (operation, NULL);                                     \n"
"  g_return_val_if_fail (context, NULL);                                       \n"
"  g_return_val_if_fail (full_roi, NULL);                                      \n"
"  g_return_val_if_fail (!gegl_rectangle_is_empty (full_roi), NULL);           \n"
"  g_return_val_if_fail (scaled_roi, NULL);                                    \n"
"                                                                              \n"
"  /* Calculate the target dimensions for the downscaled buffer used in the    \n"
"   * curve recovery. Does not enlarge the buffer.                             \n"
"   * sqrt does not cause issues with rectangular buffers as it's merely a     \n"
"   * scaling factor for later.                                                \n"
"   */                                                                         \n"
"  {                                                                           \n"
"    scale       = MAX_SCALED_PIXELS /                                         \n"
"                  (gfloat)(full_roi->width * full_roi->height);               \n"
"    scale       = sqrtf (scale);                                              \n"
"    *scaled_roi = *full_roi;                                                  \n"
"                                                                              \n"
"    if (scale > 1.0f)                                                         \n"
"      {                                                                       \n"
"        scale = 1.0f;                                                         \n"
"      }                                                                       \n"
"    else                                                                      \n"
"      {                                                                       \n"
"        scaled_roi->width  = full_roi->width  * scale;                        \n"
"        scaled_roi->height = full_roi->height * scale;                        \n"
"      }                                                                       \n"
"  }                                                                           \n"
"                                                                              \n"
"  inputs = g_slist_copy (gegl_node_get_input_pads (operation->node));         \n"
"  inputs = g_slist_sort (inputs, gegl_expcombine_pad_cmp);                    \n"
"                                                                              \n"
"  /* Read in each of the exposure buffers */                                  \n"
"  for (cursor = inputs; cursor; cursor = cursor->next)                        \n"
"    {                                                                         \n"
"      GeglBuffer *buffer;                                                     \n"
"      GeglPad    *pad = cursor->data;                                         \n"
"      exposure   *e;                                                          \n"
"                                                                              \n"
"      /* Weed out non-exposure pads */                                        \n"
"      if (!gegl_expcombine_is_exposure_pad (pad))                             \n"
"        {                                                                     \n"
"          if (strcmp (gegl_pad_get_name (pad), \"aux\"))                      \n"
"              g_warning (\"Unexpected pad name '%s' in exp-combine\",         \n"
"                         gegl_pad_get_name (pad));                            \n"
"          continue;                                                           \n"
"        }                                                                     \n"
"                                                                              \n"
"      /* Add exposure to list */                                              \n"
"      buffer  = gegl_operation_context_get_source (context,                   \n"
"                                                   gegl_pad_get_name (pad));  \n"
"      if (!buffer)                                                            \n"
"          continue;                                                           \n"
"                                                                              \n"
"      e                = gegl_expcombine_new_exposure ();                     \n"
"      e->pixels[PIXELS_FULL]   = g_new (gfloat, full_roi->width  *            \n"
"                                                full_roi->height *            \n"
"                                                components);                  \n"
"      gegl_buffer_get (buffer, full_roi, 1.0, babl_format (PAD_FORMAT),       \n"
"                       e->pixels[PIXELS_FULL], GEGL_AUTO_ROWSTRIDE,           \n"
"                       GEGL_ABYSS_NONE);                                      \n"
"                                                                              \n"
"      g_return_val_if_fail (scale <= 1.0f, NULL);                             \n"
"      if (scale == 1.0f)                                                      \n"
"          e->pixels[PIXELS_SCALED] = e->pixels[PIXELS_FULL];                  \n"
"      else                                                                    \n"
"        {                                                                     \n"
"          e->pixels[PIXELS_SCALED] = g_new (gfloat,                           \n"
"                                            (scaled_roi->width  *             \n"
"                                             scaled_roi->height *             \n"
"                                             components));                    \n"
"          gegl_buffer_get (buffer, scaled_roi, scale, babl_format (PAD_FORMAT),\n"
"                           e->pixels[PIXELS_SCALED], GEGL_AUTO_ROWSTRIDE,     \n"
"                           GEGL_ABYSS_NONE);                                  \n"
"        }                                                                     \n"
"                                                                              \n"
"      e->pixels[PIXELS_ACTIVE] = e->pixels[PIXELS_FULL];                      \n"
"                                                                              \n"
"      /* Read the exposure time: relate APEX brightness value only as a       \n"
"       * function of exposure time that is assume aperture = 1 and            \n"
"       * sensitivity = 1                                                      \n"
"       */                                                                     \n"
"      e->ti = g_ascii_strtod (ev_cursor, &ev_cursor);                         \n"
"      e->ti = 1.0f / powf (2.0f, e->ti);                                      \n"
"                                                                              \n"
"      /* Krawczyk: absolute calibration: this magic multiplier is a result    \n"
"       * of my research in internet plus a bit of tweaking with a luminance   \n"
"       * meter. tested with canon 350d, so might be a bit off for other       \n"
"       * cameras.                                                             \n"
"       *   e->ti /= 1.0592f * 11.4f / 3.125f;                                 \n"
"       */                                                                     \n"
"                                                                              \n"
"      /* For the sake of sanity, we scale the times to produce an image which \n"
"       * is much closer to the 'normal' 0-1 range. This prevents components   \n"
"       * from going massively outside of our gamut.                           \n"
"       *                                                                      \n"
"       * In particular this aids visual inspection and use of standard pixel  \n"
"       * value comparison functions in testing.                               \n"
"       */                                                                     \n"
"      e->ti *= 5e4f;                                                          \n"
"                                                                              \n"
"      if (errno)                                                              \n"
"        {                                                                     \n"
"          g_warning (\"Invalid exposure values specified for exp-combine\");  \n"
"          g_slist_foreach (exposures,                                         \n"
"                           (GFunc)gegl_expcombine_destroy_exposure,           \n"
"                           NULL);                                             \n"
"          g_slist_free    (exposures);                                        \n"
"                                                                              \n"
"          return NULL;                                                        \n"
"        }                                                                     \n"
"                                                                              \n"
"      exposures = g_slist_prepend (exposures, e);                             \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* Link each exposure's high and low sibling pointers. Simplifies           \n"
"   * anti-ghosting computations later.                                        \n"
"   */                                                                         \n"
"  exposures = g_slist_sort (exposures, gegl_expcombine_exposure_cmp);         \n"
"  {                                                                           \n"
"    exposure *prev = exposures->data;                                         \n"
"    for (cursor = exposures; cursor; cursor = cursor->next)                   \n"
"      {                                                                       \n"
"        exposure *e = cursor->data;                                           \n"
"                                                                              \n"
"        prev->hi = e;                                                         \n"
"        e->lo    = prev;                                                      \n"
"        prev     = e;                                                         \n"
"      }                                                                       \n"
"  }                                                                           \n"
"                                                                              \n"
"  if (exposures == NULL)                                                      \n"
"      g_warning (\"exp-combine did not find any suitable input pads\");       \n"
"  if (*ev_cursor != '\\0')                                                    \n"
"      g_warning (\"too many supplied EVs for input pads\");                   \n"
"                                                                              \n"
"  g_slist_free (inputs);                                                      \n"
"  return exposures;                                                           \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"#ifdef DEBUG_SAVE_CURVES                                                      \n"
"static void                                                                   \n"
"gegl_expcombine_save_curves (const gfloat *response,                          \n"
"                             guint         steps,                             \n"
"                             const gchar  *path)                              \n"
"{                                                                             \n"
"  guint i;                                                                    \n"
"  FILE *fd = fopen (path, \"w\");                                             \n"
"                                                                              \n"
"  for (i = 0; i < steps; ++i)                                                 \n"
"    fprintf (fd, \"%u %f\\n\", i, response[i]);                               \n"
"  fclose (fd);                                                                \n"
"                                                                              \n"
"  return;                                                                     \n"
"}                                                                             \n"
"#endif                                                                        \n"
"                                                                              \n"
"                                                                              \n"
"static gboolean                                                               \n"
"gegl_expcombine_process (GeglOperation        *operation,                     \n"
"                         GeglOperationContext *context,                       \n"
"                         const gchar          *output_pad,                    \n"
"                         const GeglRectangle  *full_roi,                      \n"
"                         gint                  level)                         \n"
"{                                                                             \n"
"  GeglProperties *o           = GEGL_PROPERTIES (operation);                  \n"
"  GeglBuffer *output      = gegl_operation_context_get_target (context,       \n"
"                                                               output_pad);   \n"
"                                                                              \n"
"  GeglRectangle   scaled_roi;                                                 \n"
"  GSList         *cursor;                                                     \n"
"  GSList         *exposures   = gegl_expcombine_get_exposures (operation,     \n"
"                                                               context,       \n"
"                                                               full_roi,      \n"
"                                                               &scaled_roi);  \n"
"                                                                              \n"
"  guint   i;                                                                  \n"
"  guint   steps      = 1 << o->steps,                                         \n"
"          step_max   = 0,                                                     \n"
"          step_min   = steps - 1;                                             \n"
"                                                                              \n"
"  guint   components = babl_format_get_n_components (babl_format (PAD_FORMAT));\n"
"  gfloat *hdr        = g_new (gfloat, full_roi->width  *                      \n"
"                                      full_roi->height *                      \n"
"                                      components);                            \n"
"  gfloat *weights     =   g_new (gfloat, steps);                              \n"
"  gfloat *response[3] = { g_new (gfloat, steps),                              \n"
"                          g_new (gfloat, steps),                              \n"
"                          g_new (gfloat, steps) };                            \n"
"                                                                              \n"
"  guint   saturated  = 0;                                                     \n"
"  gfloat  over       = 0.0f,                                                  \n"
"          under      = 0.0f;                                                  \n"
"                                                                              \n"
"  g_return_val_if_fail (output,    FALSE);                                    \n"
"  g_return_val_if_fail (exposures, FALSE);                                    \n"
"                                                                              \n"
"  g_return_val_if_fail (steps      > 0, FALSE);                               \n"
"  g_return_val_if_fail (components > 0, FALSE);                               \n"
"                                                                              \n"
"  /* Find the highest and lowest valid steps in the output. Remap from the    \n"
"   * 'normal' 0.0-1.0 floats to the range of integer steps.                   \n"
"   */                                                                         \n"
"  for (cursor = exposures; cursor; cursor = cursor->next)                     \n"
"    {                                                                         \n"
"      exposure *e = cursor->data;                                             \n"
"                                                                              \n"
"      for (i = 0; i < full_roi->width * full_roi->height * components; ++i)   \n"
"        {                                                                     \n"
"          /* Clamp the values we receive to [0.0, 1.0) and record the         \n"
"           * magnitude of this over/underflow.                                \n"
"           */                                                                 \n"
"          if (e->pixels[PIXELS_FULL][i] <= 0.0f)                              \n"
"            {                                                                 \n"
"              under += fabs (e->pixels[PIXELS_FULL][i]);                      \n"
"              e->pixels[PIXELS_FULL][i] = 0.0f;                               \n"
"            }                                                                 \n"
"          else if (e->pixels[PIXELS_FULL][i] > 1.0f)                          \n"
"            {                                                                 \n"
"              over  += e->pixels[PIXELS_FULL][i] - 1.0f;                      \n"
"              e->pixels[PIXELS_FULL][i] = 1.0f;                               \n"
"            }                                                                 \n"
"                                                                              \n"
"          e->pixels[PIXELS_FULL][i] *= (steps - 1);                           \n"
"          if (e->pixels[PIXELS_FULL][i] > 0.0f)                               \n"
"            {                                                                 \n"
"              step_max = MAX (step_max, e->pixels[PIXELS_FULL][i]);           \n"
"              step_min = MIN (step_min, e->pixels[PIXELS_FULL][i]);           \n"
"            }                                                                 \n"
"        }                                                                     \n"
"                                                                              \n"
"      if (e->pixels[PIXELS_SCALED] == e->pixels[PIXELS_FULL])                 \n"
"          continue;                                                           \n"
"                                                                              \n"
"      for (i = 0; i < scaled_roi.width * scaled_roi.height * components; ++i) \n"
"        {                                                                     \n"
"          e->pixels[PIXELS_SCALED][i]  = CLAMP (e->pixels[PIXELS_SCALED][i], 0.0f, 1.0f);\n"
"          e->pixels[PIXELS_SCALED][i] *= (steps - 1);                         \n"
"        }                                                                     \n"
"                                                                              \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_return_val_if_fail (step_max >= step_min, FALSE);                         \n"
"  if (under || over)                                                          \n"
"      g_warning (\"Unexpected pixel bounds. \"                                \n"
"                 \"Overflow error: %f, underflow error: %f\",                 \n"
"                 over  / components,                                          \n"
"                 under / components);                                         \n"
"                                                                              \n"
"  /* Initialise response curves and weights for estimation. Estimate and      \n"
"   * retrieve the converted output. Apply the response curves to the input    \n"
"   * using the debevec model rather than using the robertson output.          \n"
"   */                                                                         \n"
"  gegl_expcombine_weights_gauss (weights, steps, step_min, step_max, o->sigma);\n"
"  gegl_expcombine_exposures_set_active (exposures, PIXELS_SCALED);            \n"
"                                                                              \n"
"  for (i = 0; i < components; ++i)                                            \n"
"    {                                                                         \n"
"      gegl_expcombine_response_linear (response[i], steps);                   \n"
"      saturated += gegl_expcombine_get_response (hdr, i, components, exposures,\n"
"                                                 response[i], weights, steps, \n"
"                                                 &scaled_roi);                \n"
"    }                                                                         \n"
"                                                                              \n"
"#ifdef DEBUG_SAVE_CURVES                                                      \n"
"  gegl_expcombine_save_curves (response[0], steps, \"response_r\");           \n"
"  gegl_expcombine_save_curves (response[1], steps, \"response_g\");           \n"
"  gegl_expcombine_save_curves (response[2], steps, \"response_b\");           \n"
"#endif                                                                        \n"
"                                                                              \n"
"  gegl_expcombine_exposures_set_active (exposures, PIXELS_FULL);              \n"
"  gegl_expcombine_apply_debevec (hdr, components, exposures, response, weights,\n"
"                            steps, full_roi);                                 \n"
"                                                                              \n"
"  g_return_val_if_fail (G_N_ELEMENTS (response) == 3, FALSE);                 \n"
"  g_free (response[0]);                                                       \n"
"  g_free (response[1]);                                                       \n"
"  g_free (response[2]);                                                       \n"
"  g_free (weights);                                                           \n"
"                                                                              \n"
"  if (saturated)                                                              \n"
"    {                                                                         \n"
"      GEGL_NOTE (GEGL_DEBUG_PROCESS,                                          \n"
"                 \"exp-combine discovered %u saturated pixels (%f%%), \"      \n"
"                 \"results may be suboptimal\", saturated,                    \n"
"                 100.0f * saturated / full_roi->width / full_roi->height);    \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* Rescale the output so we can more easily visually inspect the output     \n"
"   * using LDR outputs.                                                       \n"
"   */                                                                         \n"
"#ifdef DEBUG_LINEARIZE                                                        \n"
"  {                                                                           \n"
"    gfloat max = G_MINFLOAT;                                                  \n"
"    guint  i;                                                                 \n"
"    for (i = 0; i < full_roi->width * full_roi->height * components; ++i)     \n"
"        max = MAX (max, hdr[i]);                                              \n"
"                                                                              \n"
"    for (i = 0; i < full_roi->width * full_roi->height * components; ++i)     \n"
"        hdr[i] /= max;                                                        \n"
"  }                                                                           \n"
"#endif                                                                        \n"
"                                                                              \n"
"  /* Save the HDR components to the output buffer. */                         \n"
"  gegl_buffer_set (output, full_roi, 0, babl_format (PAD_FORMAT), hdr,        \n"
"                   GEGL_AUTO_ROWSTRIDE);                                      \n"
"                                                                              \n"
"  /* Cleanup */                                                               \n"
"  g_free (hdr);                                                               \n"
"                                                                              \n"
"  g_slist_foreach (exposures, (GFunc)gegl_expcombine_destroy_exposure, NULL); \n"
"  g_slist_free    (exposures);                                                \n"
"                                                                              \n"
"  return TRUE;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* Grab each source's dimensions, and update a running bounding box with the  \n"
" * union of itself and the current source.                                    \n"
" * TODO: properly support inputs of varying dimensions                        \n"
" */                                                                           \n"
"static GeglRectangle                                                          \n"
"gegl_expcombine_get_bounding_box (GeglOperation *operation)                   \n"
"{                                                                             \n"
"  GeglRectangle  result = {0,0,0,0};                                          \n"
"  GSList        *inputs = gegl_node_get_input_pads (operation->node);         \n"
"                                                                              \n"
"  for (; inputs; inputs = inputs->next)                                       \n"
"    {                                                                         \n"
"      GeglPad             *pad = inputs->data;                                \n"
"      const GeglRectangle *newrect;                                           \n"
"                                                                              \n"
"      if (!gegl_expcombine_is_exposure_pad (pad))                             \n"
"        continue;                                                             \n"
"                                                                              \n"
"      /* Get the source bounds and update with the union */                   \n"
"      newrect = gegl_operation_source_get_bounding_box (operation,            \n"
"                                                        gegl_pad_get_name (pad));\n"
"      if (!newrect)                                                           \n"
"        continue;                                                             \n"
"                                                                              \n"
"      if (!gegl_rectangle_is_empty (&result) &&                               \n"
"          !gegl_rectangle_equal (newrect, &result))                           \n"
"        {                                                                     \n"
"          g_warning (\"expcombine inputs are of varying dimensions\");        \n"
"        }                                                                     \n"
"      gegl_rectangle_bounding_box (&result, newrect, &result);                \n"
"    }                                                                         \n"
"                                                                              \n"
"  if (gegl_rectangle_is_empty (&result))                                      \n"
"      g_warning (\"Bounding box for exp-combine should not be empty\");       \n"
"  return result;                                                              \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"gegl_expcombine_get_cached_region (GeglOperation       *operation,            \n"
"                                   const GeglRectangle *roi)                  \n"
"{                                                                             \n"
"  return gegl_expcombine_get_bounding_box (operation);                        \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* We have to use all the input to produce any output */                      \n"
"static GeglRectangle                                                          \n"
"gegl_expcombine_get_required_for_output (GeglOperation       *operation,      \n"
"                                         const gchar         *input_pad,      \n"
"                                         const GeglRectangle *output_roi)     \n"
"{                                                                             \n"
"  return gegl_expcombine_get_bounding_box (operation);                        \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"gegl_op_class_init (GeglOpClass *klass)                                       \n"
"{                                                                             \n"
"  GeglOperationClass       *operation_class;                                  \n"
"                                                                              \n"
"  operation_class = GEGL_OPERATION_CLASS (klass);                             \n"
"                                                                              \n"
"  operation_class->attach            = gegl_expcombine_attach;                \n"
"  operation_class->process           = gegl_expcombine_process;               \n"
"  operation_class->get_bounding_box  = gegl_expcombine_get_bounding_box;      \n"
"  operation_class->get_cached_region = gegl_expcombine_get_cached_region;     \n"
"                                                                              \n"
"  operation_class->prepare     = gegl_expcombine_prepare;                     \n"
"  operation_class->get_required_for_output = gegl_expcombine_get_required_for_output;\n"
"                                                                              \n"
"  gegl_operation_class_set_keys (operation_class,                             \n"
"  \"name\"       , \"gegl:exp-combine\",                                      \n"
"  \"title\",       _(\"Combine Exposures\"),                                  \n"
"  \"categories\" , \"compositors\",                                           \n"
"  \"description\",                                                            \n"
"      _(\"Combine multiple scene exposures into one high dynamic range image.\"),\n"
"      NULL);                                                                  \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"#endif                                                                        \n"
;
