//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
//
// -------------------------------------------------------------------
//      GEANT4 Class file
//
//
//      File name:     G4XNDeltastarTable
//
//      Author:      Maria Grazia Pia (MariaGrazia.Pia@genova.infn.it)  
// 
//      Creation date: 4 June 1999
//
//      Modifications: 
//
// Hadron Kinetic Model
// p p -> N Delta* cross section tables
//
// -------------------------------------------------------------------

#include "globals.hh"
#include "G4ios.hh"
#include "G4XNDeltastarTable.hh"
#include "G4PhysicsFreeVector.hh"

const G4int G4XNDeltastarTable::sizeNDeltastar = 121;


// Energies (GeV) corresponding to the cross section table
// Units are assigned while filling the PhysicsVector

const G4double G4XNDeltastarTable::energyTable[121] =
{
  0.0, 
  2.014,  2.014,  2.016,  2.018,  2.022,  2.026,  2.031,  2.037,  2.044,  2.052,   
  2.061,  2.071,  2.082,  2.094,  2.107,  2.121,  2.135,  2.151,  2.168,  2.185,   
  2.204,  2.223,  2.244,  2.265,  2.287,  2.311,  2.335,  2.360,  2.386,  2.413,   
  2.441,  2.470,  2.500,  2.531,  2.562,  2.595,  2.629,  2.664,  2.699,  2.736,
  2.773,  2.812,  2.851,  2.891,  2.933,  2.975,  3.018,  3.062,  3.107,  3.153,   
  3.200,  3.248,  3.297,  3.347,  3.397,  3.449,  3.502,  3.555,  3.610,  3.666,   
  3.722,  3.779,  3.838,  3.897,  3.957,  4.018,  4.081,  4.144,  4.208,  4.273,   
  4.339,  4.406,  4.473,  4.542,  4.612,  4.683,  4.754,  4.827,  4.900,  4.975,
  5.000,  6.134,  7.269,  8.403,  9.538, 10.672, 11.807, 12.941, 14.076, 15.210,  
 16.345, 17.479, 18.613, 19.748, 20.882, 22.017, 23.151, 24.286, 25.420, 26.555, 
 27.689, 28.824, 29.958, 31.092, 32.227, 33.361, 34.496, 35.630, 36.765, 37.899,  
 39.034, 40.168, 41.303, 42.437, 43.571, 44.706, 45.840, 46.975, 48.109, 49.244
};

// Cross-sections in mb, from S.A. Bass et al., Prog.Part.Nucl.Phys.41:225-370,1998 
// Units are assigned while filling the PhysicsVector

const G4double G4XNDeltastarTable::sigmaND1600[121]= 
{
  0.0, 
  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,
  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,
  0.000,  0.000,  0.000,  0.000,  0.000,  0.001,  0.001,  0.001,
  0.001,  0.002,  0.003,  0.004,  0.006,  0.009,  0.014,  0.022,
  0.036,  0.060,  0.104,  0.179,  0.290,  0.418,  0.543,  0.656,
  0.755,  0.841,  0.914,  0.976,  1.029,  1.073,  1.110,  1.139,
  1.163,  1.181,  1.195,  1.204,  1.210,  1.212,  1.212,  1.209,
  1.203,  1.196,  1.187,  1.176,  1.164,  1.150,  1.136,  1.120,
  1.104,  1.087,  1.070,  1.052,  1.034,  1.016,  0.997,  0.978,
  0.959,  0.940,  0.921,  0.903,  0.884,  0.865,  0.847,  0.828,
  0.822,  0.597,  0.447,  0.345,  0.274,  0.223,  0.184,  0.155,
  0.132,  0.114,  0.099,  0.087,  0.077,  0.069,  0.062,  0.056,
  0.050,  0.046,  0.042,  0.039,  0.036,  0.033,  0.030,  0.028,
  0.026,  0.025,  0.023,  0.022,  0.020,  0.019,  0.018,  0.017,
  0.016,  0.015,  0.015,  0.014,  0.013,  0.013,  0.012,  0.011
};

const G4double G4XNDeltastarTable::sigmaND1620[121] = 
{
  0.0,
  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,
  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.001,  0.001,
  0.001,  0.001,  0.002,  0.002,  0.003,  0.003,  0.004,  0.005,
  0.006,  0.007,  0.009,  0.010,  0.013,  0.016,  0.021,  0.028,
  0.040,  0.060,  0.096,  0.154,  0.228,  0.304,  0.372,  0.431,
  0.483,  0.526,  0.563,  0.595,  0.621,  0.643,  0.660,  0.674,
  0.685,  0.693,  0.699,  0.702,  0.704,  0.704,  0.702,  0.699,
  0.694,  0.689,  0.682,  0.675,  0.669,  0.661,  0.651,  0.642,
  0.632,  0.621,  0.611,  0.600,  0.589,  0.577,  0.566,  0.555,
  0.544,  0.532,  0.521,  0.510,  0.499,  0.488,  0.477,  0.466,
  0.463,  0.332,  0.247,  0.189,  0.150,  0.121,  0.100,  0.084,
  0.071,  0.061,  0.053,  0.047,  0.041,  0.037,  0.033,  0.030,
  0.027,  0.025,  0.022,  0.021,  0.019,  0.018,  0.016,  0.015,
  0.014,  0.013,  0.012,  0.012,  0.011,  0.010,  0.010,  0.009,
  0.009,  0.008,  0.008,  0.007,  0.007,  0.007,  0.006,  0.006 
};

const G4double G4XNDeltastarTable::sigmaND1700[121] = 
{
  0.0,
  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,
  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,
  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.001,
  0.001,  0.002,  0.003,  0.005,  0.008,  0.013,  0.020,  0.029,
  0.043,  0.062,  0.090,  0.131,  0.187,  0.258,  0.338,  0.418,
  0.495,  0.564,  0.626,  0.680,  0.726,  0.766,  0.800,  0.828,
  0.852,  0.871,  0.886,  0.898,  0.906,  0.911,  0.914,  0.914,
  0.913,  0.909,  0.904,  0.898,  0.890,  0.881,  0.871,  0.861,
  0.849,  0.837,  0.825,  0.812,  0.798,  0.785,  0.771,  0.756,
  0.742,  0.728,  0.713,  0.699,  0.685,  0.670,  0.656,  0.642,
  0.637,  0.462,  0.345,  0.266,  0.211,  0.171,  0.141,  0.119,
  0.101,  0.087,  0.076,  0.066,  0.059,  0.052,  0.047,  0.042,
  0.038,  0.035,  0.032,  0.029,  0.027,  0.025,  0.023,  0.021,
  0.020,  0.019,  0.017,  0.016,  0.015,  0.015,  0.014,  0.013,
  0.012,  0.012,  0.011,  0.010,  0.010,  0.009,  0.009,  0.009 
};


const G4double G4XNDeltastarTable::sigmaND1900[121] = 
{
  0.0,
  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,
  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,
  0.001,  0.001,  0.001,  0.001,  0.001,  0.001,  0.002,  0.002,
  0.002,  0.003,  0.003,  0.004,  0.005,  0.006,  0.007,  0.008,
  0.010,  0.012,  0.015,  0.019,  0.025,  0.034,  0.048,  0.069,
  0.097,  0.129,  0.162,  0.193,  0.220,  0.243,  0.263,  0.280,
  0.294,  0.305,  0.315,  0.322,  0.328,  0.333,  0.336,  0.337,
  0.338,  0.338,  0.337,  0.336,  0.334,  0.331,  0.328,  0.325,
  0.321,  0.316,  0.312,  0.307,  0.302,  0.297,  0.292,  0.287,
  0.282,  0.276,  0.271,  0.265,  0.260,  0.255,  0.249,  0.244,
  0.242,  0.175,  0.131,  0.101,  0.080,  0.064,  0.053,  0.045,
  0.038,  0.033,  0.028,  0.025,  0.022,  0.020,  0.018,  0.016,
  0.014,  0.013,  0.012,  0.011,  0.010,  0.009,  0.009,  0.008,
  0.007,  0.007,  0.007,  0.006,  0.006,  0.005,  0.005,  0.005,
  0.005,  0.004,  0.004,  0.004,  0.004,  0.004,  0.003,  0.003
};

const G4double G4XNDeltastarTable::sigmaND1905[121] = 
{
  0.0,
  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,
  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,
  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,
  0.000,  0.000,  0.000,  0.000,  0.000,  0.001,  0.001,  0.001,
  0.002,  0.003,  0.005,  0.008,  0.013,  0.023,  0.041,  0.075,
  0.131,  0.206,  0.286,  0.362,  0.430,  0.491,  0.542,  0.587,
  0.626,  0.659,  0.687,  0.710,  0.730,  0.745,  0.757,  0.767,
  0.774,  0.778,  0.781,  0.782,  0.781,  0.778,  0.775,  0.770,
  0.764,  0.757,  0.749,  0.741,  0.732,  0.722,  0.712,  0.702,
  0.691,  0.680,  0.669,  0.658,  0.646,  0.635,  0.623,  0.611,
  0.607,  0.454,  0.346,  0.270,  0.216,  0.176,  0.146,  0.123,
  0.105,  0.091,  0.079,  0.070,  0.066,  0.055,  0.050,  0.045,
  0.041,  0.037,  0.034,  0.031,  0.029,  0.027,  0.025,  0.023,
  0.021,  0.020,  0.019,  0.018,  0.016,  0.016,  0.015,  0.014,
  0.013,  0.012,  0.012,  0.011,  0.011,  0.010,  0.010,  0.009 
};

const G4double G4XNDeltastarTable::sigmaND1910[121] = 
{
  0.0,
  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,
  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,
  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,
  0.000,  0.000,  0.000,  0.001,  0.001,  0.001,  0.001,  0.002,
  0.002,  0.003,  0.004,  0.006,  0.008,  0.012,  0.017,  0.026,
  0.041,  0.063,  0.089,  0.116,  0.142,  0.164,  0.184,  0.201,
  0.215,  0.227,  0.238,  0.246,  0.253,  0.259,  0.263,  0.266,
  0.268,  0.270,  0.270,  0.270,  0.270,  0.269,  0.267,  0.265,
  0.263,  0.260,  0.257,  0.254,  0.251,  0.247,  0.243,  0.240,
  0.236,  0.232,  0.228,  0.224,  0.219,  0.215,  0.211,  0.207,
  0.206,  0.152,  0.115,  0.089,  0.071,  0.058,  0.048,  0.040,
  0.034,  0.030,  0.027,  0.023,  0.020,  0.018,  0.016,  0.015,
  0.013,  0.012,  0.011,  0.010,  0.009,  0.009,  0.008,  0.007,
  0.007,  0.006,  0.006,  0.006,  0.005,  0.005,  0.005,  0.004,
  0.004,  0.004,  0.004,  0.004,  0.003,  0.003,  0.003,  0.003 
};

const G4double G4XNDeltastarTable::sigmaND1920[121] = 
{
  0.0,
  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,
  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,
  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,
  0.000,  0.000,  0.000,  0.000,  0.000,  0.001,  0.001,  0.001,
  0.001,  0.002,  0.003,  0.004,  0.006,  0.009,  0.013,  0.022,
  0.037,  0.068,  0.123,  0.194,  0.263,  0.322,  0.371,  0.412,
  0.446,  0.473,  0.496,  0.515,  0.529,  0.540,  0.549,  0.555,
  0.559,  0.561,  0.561,  0.560,  0.558,  0.555,  0.551,  0.546,
  0.540,  0.534,  0.527,  0.519,  0.512,  0.504,  0.495,  0.487,
  0.478,  0.470,  0.461,  0.452,  0.443,  0.434,  0.425,  0.416,
  0.413,  0.301,  0.225,  0.174,  0.138,  0.112,  0.092,  0.078,
  0.066,  0.057,  0.049,  0.043,  0.038,  0.034,  0.031,  0.028,
  0.025,  0.023,  0.021,  0.019,  0.018,  0.016,  0.015,  0.014,
  0.013,  0.012,  0.011,  0.011,  0.010,  0.009,  0.009,  0.008,
  0.008,  0.008,  0.007,  0.007,  0.007,  0.006,  0.006,  0.006
};

const G4double G4XNDeltastarTable::sigmaND1930[121] = 
{
  0.0,
  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,
  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,
  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,
  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.001,  0.001,
  0.001,  0.002,  0.003,  0.004,  0.006,  0.010,  0.016,  0.028,
  0.051,  0.092,  0.154,  0.227,  0.298,  0.363,  0.419,  0.468,
  0.509,  0.545,  0.575,  0.605,  0.626,  0.643,  0.658,  0.665,
  0.674,  0.680,  0.684,  0.687,  0.687,  0.687,  0.685,  0.681,
  0.677,  0.672,  0.666,  0.659,  0.651,  0.644,  0.635,  0.626,
  0.617,  0.608,  0.598,  0.588,  0.578,  0.568,  0.558,  0.547,
  0.544,  0.407,  0.310,  0.242,  0.193,  0.157,  0.131,  0.110,
  0.094,  0.081,  0.075,  0.062,  0.055,  0.049,  0.044,  0.040,
  0.036,  0.033,  0.030,  0.028,  0.025,  0.024,  0.022,  0.020,
  0.019,  0.018,  0.017,  0.016,  0.015,  0.014,  0.013,  0.012,
  0.012,  0.011,  0.010,  0.010,  0.009,  0.009,  0.009,  0.008 
};

const G4double G4XNDeltastarTable::sigmaND1950[121] = 
{
  0.0,
  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,
  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,
  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,
  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.000,  0.001,
  0.001,  0.001,  0.002,  0.003,  0.005,  0.008,  0.013,  0.023,
  0.041,  0.078,  0.142,  0.229,  0.319,  0.402,  0.475,  0.538,
  0.597,  0.643,  0.678,  0.711,  0.739,  0.762,  0.781,  0.797,
  0.809,  0.818,  0.824,  0.828,  0.830,  0.830,  0.829,  0.825,
  0.821,  0.815,  0.809,  0.801,  0.793,  0.784,  0.774,  0.764,
  0.753,  0.742,  0.730,  0.718,  0.706,  0.694,  0.682,  0.670,
  0.666,  0.500,  0.380,  0.297,  0.237,  0.193,  0.160,  0.135,
  0.115,  0.099,  0.087,  0.076,  0.068,  0.060,  0.054,  0.049,
  0.044,  0.040,  0.037,  0.034,  0.031,  0.029,  0.027,  0.025,
  0.023,  0.022,  0.020,  0.019,  0.018,  0.017,  0.016,  0.015,
  0.014,  0.013,  0.013,  0.012,  0.012,  0.011,  0.010,  0.010 
};



G4XNDeltastarTable::G4XNDeltastarTable() 
{
  xMap["delta(1600)-"] = (G4double*) sigmaND1600;
  xMap["delta(1600)0"] = (G4double*) sigmaND1600;
  xMap["delta(1600)+"] = (G4double*) sigmaND1600;
  xMap["delta(1600)++"] = (G4double*) sigmaND1600;

  xMap["delta(1620)-"] = (G4double*) sigmaND1620;
  xMap["delta(1620)0"] = (G4double*) sigmaND1620;
  xMap["delta(1620)+"] = (G4double*) sigmaND1620;
  xMap["delta(1620)++"] = (G4double*) sigmaND1620;

  xMap["delta(1700)-"] = (G4double*) sigmaND1700;
  xMap["delta(1700)0"] = (G4double*) sigmaND1700;
  xMap["delta(1700)+"] = (G4double*) sigmaND1700;
  xMap["delta(1700)++"] = (G4double*) sigmaND1700;

  xMap["delta(1900)-"] = (G4double*) sigmaND1900;
  xMap["delta(1900)0"] = (G4double*) sigmaND1900;
  xMap["delta(1900)+"] = (G4double*) sigmaND1900;
  xMap["delta(1900)++"] = (G4double*) sigmaND1900;

  xMap["delta(1905)-"] = (G4double*) sigmaND1905;
  xMap["delta(1905)0"] = (G4double*) sigmaND1905;
  xMap["delta(1905)+"] = (G4double*) sigmaND1905;
  xMap["delta(1905)++"] = (G4double*) sigmaND1905;

  xMap["delta(1910)-"] = (G4double*) sigmaND1910;
  xMap["delta(1910)0"] = (G4double*) sigmaND1910;
  xMap["delta(1910)+"] = (G4double*) sigmaND1910;
  xMap["delta(1910)++"] = (G4double*) sigmaND1910;

  xMap["delta(1920)-"] = (G4double*) sigmaND1920;
  xMap["delta(1920)0"] = (G4double*) sigmaND1920;
  xMap["delta(1920)+"] = (G4double*) sigmaND1920;
  xMap["delta(1920)++"] = (G4double*) sigmaND1920;

  xMap["delta(1930)-"] = (G4double*) sigmaND1930;
  xMap["delta(1930)0"] = (G4double*) sigmaND1930;
  xMap["delta(1930)+"] = (G4double*) sigmaND1930;
  xMap["delta(1930)++"] = (G4double*) sigmaND1930;

  xMap["delta(1950)-"] = (G4double*) sigmaND1950;
  xMap["delta(1950)0"] = (G4double*) sigmaND1950;
  xMap["delta(1950)+"] = (G4double*) sigmaND1950;
  xMap["delta(1950)++"] = (G4double*) sigmaND1950;
}


G4XNDeltastarTable::~G4XNDeltastarTable()
{ }


const G4PhysicsVector* G4XNDeltastarTable::CrossSectionTable(const G4String& particleName) const
{
  // NOTE: the returned pointer is owned by the client

  if (xMap.find(particleName) != xMap.end())
    {
      // Cross section table for the requested particle available in the Map
      G4PhysicsFreeVector* sigmaVector = new G4PhysicsFreeVector(sizeNDeltastar);
      std::map <G4String, G4double*, std::less<G4String> >::const_iterator iter;
      G4double* sigmaPointer = 0;
      for (iter = xMap.begin(); iter != xMap.end(); ++iter)
	{
	  G4String str = (*iter).first;
          if (str == particleName)
	    {
	      sigmaPointer = (*iter).second; 
	    }
	}

      G4int i;
      for (i=0; i<sizeNDeltastar; i++)
	{
	  G4double value = *(sigmaPointer + i) * 0.5* millibarn;
	  G4double energy = energyTable[i] * GeV;
	  sigmaVector->PutValue(i,energy,value);
	}	    
      return sigmaVector;
    }
  else
    // No cross section table for the requested particle is available in the Map
    return 0;
} 










