/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef SEG_STACK_HH
#define SEG_STACK_HH

#include "calc_engine/eval_stack.hh"
#include "SegList.hh"
#include <string>
#include <vector>

/**  Segment calculation data token.
  *  @brief seg_calc eval token.
  *  @author John Zweizig (john.zweizig@ligo.org)
  *  @version $Id: seg_stack.hh 6426 2011-05-12 01:40:42Z john.zweizig@LIGO.ORG $
  */
class seg_token : public eval_token {
public:
    /**  Data types. For now the only data types are string, numeric and 
      *  logical (boolean values). In the future this should be templated 
      *  to a more descriptive class.
      */
    enum type_enum {
	kNone,    ///< Undefined token type
	kSegment, ///< Segment list token type
	kNumeric  ///< Numeric token type
    };

    /**  eval_stack token type data type
      */
    using eval_token::datype;

public:
    /**  Construct an empty token.
      *  @brief Default constructor.
      */
    seg_token(void);

    /**  Construct a token from a segment list.
      *  @brief Segment list token constructor.
      *  @param s Segment list
      */
    seg_token(const LockSegList& s);

    /**  Construct a token from a numeric value.
      *  @brief Numeric token constructor.
      *  @param d Numeric data.
      */
    seg_token(double d);

    /**  Destroy the token and release any allocated storage.
      *  @brief Destructor.
      */
    virtual ~seg_token(void);

    //----------------------------------  eval_type virtual functions.
    /**  Allocate storage for and construct a token object identical to
      *  this object.
      *  @brief Clone a token object.
      *  @return Pointer to the clone.
      */
    seg_token* clone(void) const;

    /**  Get the token data type.
      *  @brief Token type.
      *  @return Token type enumerator.
      */
    datype getType(void) const;

    /**  Get the numeric value of the token.
      *  @brief Numeric token value.
      *  @return Token value represented as a double.
      */
    double getNumeric(void) const;

    /**  Get string value of the token.
      *  @brief String token value.
      *  @return Token string value.
      */
    std::string getString(void) const;

    /**  Set token to a numeric value.
      *  @brief Set token.
      *  @param val New numeric value of the token.
      */
    void setNumeric(double val);

    /**  Set token to a string value.
      *  @brief Set token.
      *  @param s New string value of the token.
      */
    void setString(const std::string& s);

    //---------------------------------- Specialized mutators/accessors.
    /**  Get a reference to the token segment list.
      *  @brief Segment list reference.
      *  @return Reference to the segment list.
      */
    LockSegList& refSegment(void);

    /**  Get a constant reference to the token segment list.
      *  @brief Constant segment list reference.
      *  @return Constant reference to the segment list.
      */
    const LockSegList& refSegment(void) const;

private:
    LockSegList mValue;
    double      mNumeric;
    type_enum   mType;
};

//====================================== seg_token inline functions
inline const LockSegList& 
seg_token::refSegment(void) const {
    return mValue;
}

inline LockSegList& 
seg_token::refSegment(void) {
    return mValue;
}

/**  The evaluation stack holds currently active intermediate results.
  *  In the initial implementation, data are stored as strings.
  *  @memo Evaluation stack.
  *  @author John Zweizig
  *  @version 1.0; Last modified July 6, 2004
  */
class seg_stack : public eval_stack {
public:
    using eval_stack::push;

    /**  Segment evaluation stack constructor.
      *  @brief Default (empty) constructor.
      */
    seg_stack(void);

    /**  Destroy the segment evaluation stack
      *  @brief Evaluation stack destructor. 
      */
    ~seg_stack(void) {}

    /**  Write a formatted dump of the stack contents to a specified 
      *  output stream.
      *  @brief Dump stack contents.
      *  @param out Output stream to which the stack will be dumped.
      */
    void dump(std::ostream& out) const;

    /**  Perform a specified operation on the top stack entries. The number
      *  of argument appropriate for the operator are popped off the stack 
      *  and the result is pushed onto the top of the stack.
      *  @brief Perform an operation on the top tokens.
      *  @param op Enumerated operation to be performed on the stack.
      */
    void evaluate(OpsEnum op);

    /**  Test whether the specified stack entry is numeric. Stack entries 
      *  are numbered from the top (most recent is 0) to the bottom (least
      *  recent).
      *  @brief Test for numeric operand.
      *  @param lvl Stack entry number.
      *  @return True if stack entry is numeric.
      */
    bool numeric(int lvl) const;

    /**  Push a string token onto the top of the stack.
      *  @brief Push a string.
      *  @param str String to be pushed onto the stack.
      */
    void push(const std::string& str);

    /**  Push a numeric value. The double precision float value is converted
      *  to a string before being pushed on to the stack.
      *  @brief Push a numeric value
      *  @param val Value to be pushed on top of the stack.
      */
    void push(double val);

    /**  Get a reference to the top element of the stack.
      *  @brief  Reference to top element
      *  @return Reference to top element.
      */  
    seg_token& top(void);

    /**  Get a constant reference to the top element of the stack.
      *  @brief  Constant reference to top element
      *  @return Constant reference to top element.
      */  
    const seg_token& top(void) const;

};

inline bool 
seg_stack::numeric(int off) const {
    return type(off) == seg_token::kNumeric;
}

inline seg_token& 
seg_stack::top(void) {
    return dynamic_cast<seg_token&>(eval_stack::top());
}

inline const seg_token& 
seg_stack::top(void) const {
    return dynamic_cast<const seg_token&>(eval_stack::top());
}

#endif // !defined(STR_STACK_HH)
