/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "seg_iobase.hh"
#include "Time.hh"
#include <iostream>
#include <fstream>
#include <sstream>
#include <stdexcept>
#include <cstdlib>

using namespace std;

bool segid_debug = true;

//======================================  Name component constructor
segID::segID(const std::string& name, const std::string ifo, int ver) 
    : _name(name), _ifo(ifo), _version(ver)
{
    string::size_type inx = _name.find(':');
    if (inx != string::npos) {
	_ifo = _name.substr(0, inx);
	_name.erase(0, inx+1);
    }
}

//======================================  Full name constructor
segID::segID(const std::string& full) 
    : _version(kUnknown)
{

    //----------------------------------  Look for a colon
    string::size_type inx = full.find(':');

    //----------------------------------  Colon delimits ifo name
    if (inx != string::npos) {
	_ifo = full.substr(0, inx);
	inx++;
    }

    //----------------------------------  Otherwise parse from start of string
    else {
	inx = 0;
    }

    //----------------------------------  Look for a semicolon
    string::size_type semi = full.substr(inx).find(';');
    _name = full.substr(inx, semi);

    //----------------------------------  Semi-colon delimits the version
    if (semi != string::npos) {
	inx += semi + 1;
	string vsn = full.substr(inx);
	if (vsn == "any") {
	    _version = kAny;
	} else if (vsn  == "unkn") {
	    _version = kUnknown;
	} else {
	    _version = strtol(vsn.c_str(), 0, 0);
	}
    }
}

//======================================  Destructor
segID::~segID(void) {
}

//======================================  Return the full name
std::string 
segID::full_name(void) const {
    ostringstream full;
    if (!_ifo.empty()) full << _ifo << ":";
    full << _name;
    if (_version > 0) {
	full << ";" << _version;
    } else if (_version == kUnknown) {
	full << ";unkn";
    } else if (_version == kAny) {
	full << ";any";
    }
    return full.str();
}


//======================================  Compare segIDs
bool 
segID::operator< (const segID& x) const {
    if (_name != x._name) {
	return _name < x._name;
    }

    if (_ifo != x._ifo) {
	return _ifo < x._ifo;
    }

    return _version < x._version;
}

//======================================  Test equal segIDs
bool 
segID::operator==(const segID& x) const {
    bool rc = _name == x._name && _ifo == x._ifo && _version == x._version;
    return rc;
}

//======================================  Get the root segID
segID
segID::root_segid(void) const {
    segID seg(*this);
    size_t len = _name.size();
    if (len > 8 && _name.substr(len-8) == "_summary") {
	seg._name.erase(len);
    }
    return seg;
}

//======================================  Get the root segID
bool
segID::summary(void) const {
    size_t len = _name.size();
    return (len > 8 && _name.substr(len-8) == "_summary");
}


//======================================  Test for compatibility
bool 
segID::test(const std::string& name, const std::string ifo, int ver) const {
    segID rhs(name, ifo, ver);
    return test(rhs);
}

//======================================  Test for compatibility
bool 
segID::test(const segID& x) const {
    bool rc = ((x._name.empty()     || _name == x._name) &&
	       (x._ifo.empty()      || _ifo  == x._ifo) &&
	       (x._version == kAny  || x._version == kUnknown ||
		_version  == x._version)
	       );
    if (!rc && segid_debug) {
	cout << "segID test failed: " << endl;
	if (!x._name.empty() && _name == x._name) {
	    cout << "name: \"" << _name << "\" != \"" <<  x._name << "\""
		 << endl;
	}
	if (!x._ifo.empty() && _ifo == x._ifo) {
	    cout << "ifo: \"" << _ifo << "\" != \"" <<  x._ifo << "\"" << endl;
	}
	if (x._version != kAny && _version != x._version) {
	    cout << "version: \"" << _version << "\" != \"" <<  x._version
		 << "\"" << endl;	    
	}
    }
    return rc;
}

//======================================  Set ifo field
void  
segID::set_ifo(const std::string& Ifo) {
    _ifo = Ifo;
}

//======================================  Set name field
void  
segID::set_name(const std::string& Name) {
    _name = Name;
}

//======================================  Set version number
void  
segID::set_version(int vrsn) {
    _version = vrsn;
}


//======================================  Print a segId
std::ostream&
operator<<(std::ostream& out, const segID& sid) {
    return out << sid.full_name();
}



//======================================  Constructor
seg_iobase::seg_iobase(void)
    : mDebug(0)
{}

//======================================  Destructor
seg_iobase::~seg_iobase(void)
{}

//======================================  Merge data into argument map
void 
seg_iobase::merge(seg_map& outmap, bool dosort) {
    //--------------------------------- Merge the files
    for (seg_iter p=mSegMap.begin(); p != mSegMap.end(); ++p) {
	p->second.sort();
	p->second.coalesce();
	if (mDebug) cerr << "SegList: " << p->first.full_name() 
			 << " defined with " << p->second.size() 
			 << " segments, " << p->second.live() 
			 << " seconds live." << endl;

	//----------------------------- Add new list to map.
	seg_iter pm = outmap.find(p->first);
	if (pm == outmap.end()) {
	    outmap.insert(*p);
	}
	else if (dosort) {
	    pm->second |= p->second;
	}
	else {
	    pm->second.stuff(p->second);
	}
    }
}

//======================================  Set debug print level
void 
seg_iobase::set_debug(int lvl) {
    mDebug = lvl;
}
