/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    Bit test function
//
#ifndef FRWRITE_HH
#define FRWRITE_HH

#include "DMTBase.hh"
#include "Interval.hh"
#include <iosfwd>
#include <list>
#include <vector>
#include <string>

#if LDAS_VERSION_NUMBER >= 109360
#if LDAS_VERSION_NUMBER > 119013
#include "framecpp/Common/FrameBuffer.hh"
#endif
#include "framecpp/OFrameStream.hh"
namespace FrameCPP {
    namespace Common {
	class FrameBufferInterface;
    }
}
#endif

/**  \page FrWrite
  *
  *  FrWrite writes frames containing a user-defined subset of the channels
  *  from the input frames. The input data source may be either an online 
  *  shared memory partition or one or more frame files. The data are 
  *  written to one or more frame files, the names of which may include a 
  *  time stamp generated from the GPS time of the first frame in the file. 
  *  The list of channels to be written is specified in an optional 
  *  configuration file. If no configuration file is specified, all available
  *  channels are written.
  *
  *  <b>Note:</b> At present, no pre-processing is performed on the data. 
  *  Filtering, decimation and time deskewing are expected to be 
  *  added at some time in the future. (January 27, 2000).
  *
  *  <h2> The FrWrite Configuration File </h2>
  *
  *  The FrWrite configuration file specifies which channels are to be 
  *  written to the output file. Each line of the configuration file
  *  specifies a single channel name. Comment lines may be interspersed 
  *  throughout the file and must have a '#' in the first column. A single 
  *  error message is generated if a requested channel doesn't exist in a 
  *  given frame, although a message will be printed each time a channel is
  *  not found if the debug flag is non-zero.
  *
  *  <h2> Running FrWrite</h2>
  *
  *  The syntax of the FrWrite command is as follows:
  *  \verbatim
FrWrite [-partition <part-name>] [-infile <file>] [-debug <dbg-level>] \
        [-cfile <conf-file>] [-nframe <nFrame>] [-repeat <nFile>] [+toc] \
        [-ofile <out-file>] [-sequence <nSeq>] [-trigger <osc-file>] \
	[-resize <nSec>]
     \endverbatim
  *  Where the arguments have the following meaning:
  *
  *  <table>
  *  <tr><td>\<part-name\></td>
  *      <td>Shared memory partition with data to be read.</td></tr>
  *  <tr><td>\<file\></td>
  *      <td>Input frame file (exclusive of <part-name>).</td></tr>
  *  <tr><td>\<dbg-level\></td>
  *      <td>Debug level [0].</td></tr>
  *  <tr><td>\<conf-file\></td>
  *      <td>Configuration file name [].</td></tr>
  *  <tr><td>\<nFrame\></td>
  *      <td>Number of frames to write in each file [1].</td></tr>
  *  <tr><td>\<nFile\></td>
  *      <td>The number of files to be written [1].</td></tr>
  *  <tr><td>\<out-file\></td>
  *      <td>Output file template [FrWrite-%Y.%m.%d-%H.%N.%S.F]</td></tr>
  *  <tr><td>\<nSeq\></td>
  *      <td>The number of output files in a sequence [0].</td></tr>
  *  <tr><td>\<osc-file\></td>
  *      <td>Trigger definition file [].</td></tr>
  *  <tr><td>\<nSec\></td>
  *      <td>New output frame size [0]</td></tr>
  *  </table>
  *
  *  The partition name is mutually exclusive with the input frame file name.
  *  If both are specified, data are read from the specified shared memory 
  *  partition. Multiple input files may be specified either with several
  *  "-infile \<file\>" arguments or with wild-cards (e.g. "-infile '*.F' ").
  *  The debug level defaults to 0 (no debug messages). Any other value for 
  *  \<dbg-level\> will cause debugging messages to be printed to cout. The
  *  \<out-file\> argument is treated as a template for generating file names.
  *  The name is generated with TimeStr
  *  using the template and the GPS start-time of the first frame as 
  *  arguments. You can thus include the GPS time in seconds by using a file 
  *  name template of e.g. "MyFrame-%s.F". If \<nFile\> is zero, FrWrite 
  *  will continue to write frame files until it is terminated with a \e ctrl/c 
  *  or a kill command. Output files may be divided into sequences of \<nSeq\> 
  *  files. This is most useful for limiting the number of files in a single 
  *  directory. The sequence number may be incorporated into the output file 
  *  name with a "%q" token. The \<osc-file\> defines an optional condition 
  *  that the input frame must satisfy for the data to be written to the 
  *  output. the \c +toc argument indicates that a table of contents is to
  *  be written at the end of each output frame file.
  *  @author J.Zweizig
  *  @version 2.2; Modified September 19, 2000
  */  
//@{
//@}

class Time;
class OperStateCondList;

class ChanDesc {
public:
    typedef frameh_pointer    frame_ptr_type;
    typedef fradcdata_pointer fradc_ptr_type;
  public:
    ChanDesc(void);
    ChanDesc(const ChanDesc& cd);
    ChanDesc(const char* configstr);
    ChanDesc& operator=(const ChanDesc& x);
    void Copy(frame_ptr_type in, frame_ptr_type out, Interval Off, Interval dT);
    void Copy(fradc_ptr_type in, frame_ptr_type out, Interval Off, Interval dT);
    void setError(bool err);
    bool getError(void) const;
    const char* getName(void) const;
  private:
    //  Channel name
    std::string mName;

    //  Sample rate (after decimation).
    double mRate;

    //  Error state (Channel temporarily disabled)
    bool mError;
};

class FrWrite : public DMTBase {
public:
    typedef frameh_pointer frame_ptr_type;
public:
    FrWrite(int argc, const char *argv[]);
    ~FrWrite();
    void ProcessFrame(frame_ptr_type frame);
    frame_ptr_type MakeFrame(frame_ptr_type frame, Interval Off, Interval dT);
    void WriteFrame(frame_ptr_type frame);

    /*  Close a FrameCPP frame writer.
      *  @memo Open a frame writer.
      */
    void closeWriter(void);

    /*  Open a FrameCPP frame writer. The file name is generated from the 
      *  file name template in mOutFile and the GPS time specified by the
      *  gps argument.
      *  @memo Open a frame writer.
      *  @return true if an error occurred in opening he file.
      *  @param gps GPS time to be used to generate the file name.
      */
    bool openWriter(const Time& gps);

    bool readConfig(std::istream& in);

  private:
    typedef std::list<ChanDesc>           ChanList;
    typedef ChanList::iterator            chan_iter;
    typedef std::vector<frame_ptr_type>   BackVect;

  private:
    // Number of frames to write per file.
    int mFrOut;

    // Number of frames written in the current file
    int mFrDone;

    // Total number of frames written.
    int mFrTotal;

    // Current file number. First file is 0.
    int mFileOut;

    /*  Number of files to write. mRepeat defaults to 1. If set to zero,
     *  the program will repeat indefinitely.
     */
    int mRepeat;

    /*  Number of files in a sequence. The only effect a sequence has is
     *  the sequence number which may be added to the file name.
     */
    int mSequence;

    /*  Output mode.
     */
    enum {m_CopyAll, m_Select} mMode;

    // New frame length
    Interval mResize;

    // Output file name format.
    std::string mOutFile;

    //  Current output file name.
    std::string mCurFile;

    //  FrameCPP output writer.
    bool mTOC;

    //  FrameCPP output writer.
    FrameCPP::OFrameStream* mWriter;

#if LDAS_VERSION_NUMBER < 109360
    //  Current output stream.
    std::ostream*      mStream;
#else
    //  Current output buffer
    FrameCPP::Common::FrameBufferInterface* mFrameBuf;
#endif

    ChanList           mChannel;

    //  Current output stream.
    OperStateCondList* mOSC;

    //--- History and information.
    BackVect           mBacklog;

    //    Number of earlier frames to write when a trigger is found
    int mBackFrame;
    //    Backlog of frames left to be written
    int mTrigFrame;
    //    Backlog of frames left to be written
    int mBackWrite;
    //    Number of valid frames in backlog
    int mBackValid;
};

inline bool
ChanDesc::getError(void) const {
    return mError;
}

inline const char*
ChanDesc::getName(void) const {
    return mName.c_str();
}
#endif
