#ifndef TRIGRAW_HH
#define TRIGRAW_HH
//
//    Define external classes
#include <string>
#include <iostream>
#include "Interval.hh"

namespace trig { 

/**  Trigger raw data descriptor class.
  *  Describes the location of the raw data in which the trigger was found.
  */
class TrigRaw {
  public:
    /**  Create an empty data descriptor.
      *  @memo Default raw data descriptor constructor.
      */
  TrigRaw(void);

    /**  Create a descriptor for the data. The frame containing the data 
      *  is in 'frame' of file 'File' and the epoch is defined by 
      *  {Start, Length}.
      *  @memo Raw data descriptor constructor.
      */
    TrigRaw(const char* File, const char* FrameID, Interval Start=0.0, 
	    Interval Length=0.0);

    /**  Copy an object into an existing object.
      *  @memo Copy constructor.
      */
    TrigRaw(const TrigRaw& x);

    /**  Copy an object into an existing object.
      *  @memo Assignment operator.
      */
    TrigRaw& operator=(const TrigRaw& x);

    /**  The serializer is a function that write the object to a data stream.
      *  @memo Object serializer.
      */
    void Serialize(std::ostream &serial) const;

    //----------------------------  Accessors
    /**  Returns a pointer to a channel name string.
      *  @memo Get the channel name
      */
    const std::string& getChannel(void) const {return mChannel;}

    /**  Returns a pointer to a frame name.
      *  @memo Get the channel name
      */
    const std::string& getFrame(void) const {return mFrameID;}

    /**  Returns a time offset within the specified frame where the 
      *  epoch of interest starts.
      *  @memo Get the epoch starting offset.
      */
    Interval getEpochStart(void) const {return mEpochStart;}

    /**  Returns the epoch time interval.
      *  @memo Get the epoch length.
      */
    Interval getEpochLength(void) const {return mEpochLength;}

    /**  Fill the channel name string.
      *  @memo Set the channel name
      */
    void setChannel(const std::string& Name);

  private:
    /**  The file nameof the data source.
      *  Note that the online data may see a different data path than 
      *  the Frame ID.
      *  @memo File providing input data.
      */
    std::string mFile;

    /**  The frame name is the standard naming convention for the frame. 
      *  Note that the online data may see a different data path than the
      *  named recorded frame.
      *  @memo Input data sample frame name.
      */
    std::string mFrameID;

    /**  The name of the channel analysed by the process. Only the principal
      *  channel name is listed if more than one channel is used in the 
      *  pre-processing.
      *  @memo Principal channel name.
      */
    std::string mChannel;

    /**  The pre-analysis may divide or combine data frames to form epochs.
      *  EpochStart defines the offset in seconds from the start of the 
      *  frame to the start of the used epoch.
      *  @memo Epoch start time.
      */
    Interval mEpochStart;

    /**  The length (in seconds) of the analyzed time epoch
      *  @memo Epoch length time.
      */
    Interval mEpochLength;

};  // trig::TrigRaw

}   // namespace trig

#endif   //  TRIGRAW_HH
