#ifndef TRIGPROC_HH
#define TRIGPROC_HH
//
//   Trigger process class
//
//   The trigger process class describes the process used to generate 
//   the trigger.
//
#include <string>
#include <iosfwd>
#ifndef __CINT__
#include <sys/types.h>
#else
typedef int pid_t;
#endif
#include "Time.hh"

namespace trig {

/**  The trigger process class is used to describe the process used to 
  *  generate a trigger. The current version of the TrigProc class 
  *  maps directly into the LDAS meta-database process table record as
  *  defined in the November 30, 1999 version of Peter Shawhans database
  *  memo "Table Definitions for LDAS Metadata / Event Database" 
  *  (LIGO-T990101-02-E).
  *  @memo Trigger process class.
  *  @author J. Zweizig
  *  @version 1.2; Modified August 12, 2010
  */
class TrigProc {
public:
  /** Type definition for gps  times in seconds.
    */
  typedef Time::ulong_t gps_t;

  /**  Default constructor.
    *  @memo Default constructor.
    */
  TrigProc(void) {}

  /**  Make a connection to the trigger broker and get a process ID.
    *  @memo Create a trigger process identifier.
    *  @param cvsid Id string supplied by svn Header symbol.
    *  @param title Process title strin
    */
  TrigProc(const char* cvsid, const char* title=0);

  /**  Construct the process ID from a detailed list of data.
    *  @memo Trigger process ID constructor.
    *  @param Source  Source file name or URL.
    *  @param Version Source file version (from svn/cvs)
    *  @param Start   Process start time.
    *  @param Host    Node name.
    *  @param PID     Pricess identifier.
    *  @param Online  Online status (tru for online processes)
    *  @param IFOs    List of IFOs available for processing.
    *  @param Author  Author of record for the source code.
    *  @param Mod     Source code modification time.
    *  @param Title   Process title.
    *  @param Exec    Executable being run by process.
    *  @param End     Process End time (if available).
    */
  TrigProc(const char* Source, const char* Version, gps_t Start, 
	   const char* Host, pid_t PID=0, bool Online=false, 
	   const char* IFOs=0, const char* Author=0, gps_t Mod=0, 
	   const char* Title=0, const char* Exec=0, gps_t End=0);

  /**  Remove the trigger process. Unregister with the trigger broker.
    *  @memo Destroy a trigger process.
    */
  ~TrigProc(void);

  /**  The display method prints the contents of the %TrigProc structure to
    *  the specified output stream.
    *  \brief Get the author's name.
    *  \param out Output stream
    *  \return Output stream reference.
    */
  std::ostream& display(std::ostream& out) const;

  /**  getAuthor returns a static pointer to the author's name.
    *  @memo Get the author's name.
    *  @return Constant pointer to character string containing author's name.
    */
  const char* getAuthor(void) const;

  /**  getEndTime returns the time at which the process described by 
    *  this TrigProc object ended. The end time is Time(0) if the 
    *  process is still running or if the end time was not recorded.
    *  @memo Process start time.
    *  @return Process end tiem or zero (see text).
    */
  gps_t getEndTime(void) const;


  /**  Get the segment comment.
    *  @memo Get the process comment.
    *  @return Pointer to the comment string.
    */
  const char* getComment(void) const;

  /**  Get the Cvs ID for the program Source file.
    *  @memo Get the cvs ID.
    *  @return Pointer to svn (or cvs) Header string.
    */
  const char* getCvsID(void) const;

  /**  Get a list of interferometers that this process monitors.
    *  @memo Interferometer list.
    *  @return Constant pointer to IFO list character string.
    */
  const char* getIFOs(void) const;

  /**  Get the time at which the cvs sources were last modified.
    *  @memo Process start time.
    *  @return GPS time when the source code was last modified.
    */
  gps_t getModTime(void) const;

  /**  Get a pointer to the program name.
    *  @memo Get the program name.
    *  @return A static pointer to the program name.
    */
  const char* getName(void) const;

  /**  getNode returns a constant pointer to the process node.
    *  @memo Get the process node.
    *  @return Constant pointer to execution node name.
    */
  const char* getNode(void) const;

  /**  Returns the parameter set string.
    *  @memo Get the parameter set string.
    *  @return Constant pointer to parameter set string.
    */
  const char* getParamSet(void) const;

  /**  Returns the database ID of this process as provided by the trigger 
    *  broker.
    *  @memo Get the database record ID.
    *  @return Constant pointer to trigger broker Process ID string.
    */
  const char* getProcessID(void) const;

  /**  Returns the unix process ID of this process.
    *  @memo Get the unix process ID.
    *  @return Unix process ID on local node. 
    */
  pid_t getProcID(void) const;

  /**  Get the name of the cvs source file (repository and file name) from
    *  which the main object was compiled to make the executable.
    *  @memo Source file name.
    *  @return Constant pointer to source file name string.
    */
  const char* getSource(void) const;

  /**  getStartTime returns the time at which the process described 
    *  by this %TrigProc instance started.
    *  @memo Process start time.
    *  @return GPS time of process start.
    */
  gps_t getStartTime(void) const;

  /**  getName returns a static pointer to the program title.
    *  @memo Get the program title.
    *  @return Constant pointer to process title string.
    */
  const char* getTitle(void) const;

  /**  Get the program version number retrieved from the CVS header.
    *  @memo Get the program version.
    *  @return a static pointer to the cvs version string.
    */
  const char* getVersion(void) const;

  /**  See if two TrigProcs define the identical process. The fields 
    *  compared are the Program name, the unix process ID, the node name 
    *  and the start time. These fields are used to generate the primary
    *  key in the database, so equal process IDs will necessarily refer
    *  to the same metadata table row.
    *  @memo Compare two process identifiers.
    *  @param x Process to be compared to this.
    *  @return true if the processes are the same.
    */
  bool operator==(const TrigProc& x) const;

  /**  Test whether the specified is (was) an online process.
    *  @memo Test for online process.
    *  @return True if this process is online.
    */
  bool isOnline(void) const;

  /**  Set a comment for this process.
    *  @memo Set the comment.
    *  @param comment Comment for this process.
    */
  void setComment(const std::string& comment);

  /**  Specify the cvs / svn Header ID string.
    *  @memo Set the cvs ID string
    *  @param ID Svn Header string.
    */
  void setCvsID(const std::string& ID);

  /**  Specify the process termination GPS time.
    *  @memo set process end time.
    *  @param tEnd Process end time.
    */
  void setEndTime(gps_t tEnd);

  /**  Set the list of IFOs analyzed by this process.
    *  @memo Set ifo list.
    *  @param ifos List of ifos.
    */
  void setIFOs(const std::string& ifos);

  /**  Set the process database foreign key string used to identify this 
    *  process.
    *  @memo Set process database key.
    *  @param ID Dataase foreign key.
    */
  void setProcessID(const std::string& ID);

private:
  /**  Creator Data Base.
    */
  int mCreatorDB;

  /**  Program name.
    */
  std::string mProgram;

  /**  Source file version.
    */
  std::string mVersion;

  /**  CVS repository file name
    */
  std::string mSource;

  /**  GPS time of last update of the cvs source.
    */
  gps_t mTime;

  /**  Process Author name.
    */
  std::string mAuthor;

  /**  Process title.
    */
  std::string mComment;

  /**  Is online.
   */
  bool mIsOnline;

  /** Node name.
   */
  std::string mNode;

  /** Unix process ID
   */
  pid_t mProcID;

  /**  Start time.
   */
  gps_t mStartTime;

  /**  End time.
   */
  gps_t mEndTime;

  /**  Cvs ID string for process source code.
    *  @memo Process source ID.
    */
  std::string mCvsID;

  /**  A unique ID is assigned to each process when it registers with 
    *  with the trigger broker. 
    *  @memo Trigger process ID.
    */
  std::string mProcessID;

  /**  Parameter set.
    */
  std::string mParamSet;

  /**  List of online IFOs available to the monitor.
    *  @memo IFO list.
    */
  std::string mIfos;

};  // trig::TrigProc

//=======================================  Inline accessors
inline const char*
TrigProc::getComment(void) const {
    return mComment.c_str();
}

inline const char*
TrigProc::getCvsID(void) const {
    return mCvsID.c_str();
}


}  //  namespace trig

extern const char* getIFO(void);
// extern trig::TrigProc ProcIdent;

#endif // TRIGPROC_HH
