/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: xsilEvent	     					*/
/*                                                         		*/
/* Module Description: Classes for LIGO-LW input/output to events.	*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/*                Daniel Sigg   Initial revision                        */
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: none 						*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Rauha Rahkola (509) 372-8141  (509) 372-8137  rrahkola@darkwing.     */
/*                                                        uoregon.edu   */
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 2001.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#ifndef _GDS_XSILEVENT_HH
#define _GDS_XSILEVENT_HH

#include <string>
#include <iosfwd>
#ifndef __CINT__
#include "xml/Xsil.hh"
#else
  #include <map>
namespace xml {
  class xsil_base {
  public:
    xsil_base(const char* name, int level=0);
    xsil_base(const char* name, const char* units, int level=0);
  };
  class xsilHandler {
  public:
    typedef std::map<std::string, std::string> attrlist;
    xsilHandler(bool ignore=false);
  };
   class xsilHandlerQuery {
   public:
    typedef std::map<std::string, std::string> attrlist;
      xsilHandlerQuery();
      virtual ~xsilHandlerQuery();
      virtual xsilHandler* GetHandler (const attrlist& attr) = 0;
      virtual xsilHandler* GetTableHandler (const attrlist& attr) {
         return 0; }
   };
}
#endif
#include "events/Event.hh"
#include "events/List.hh"
#include "events/Chain.hh"
#include "events/Set.hh"
#include "events/Name.hh"

namespace xml {

/** \ingroup IO_xml
    \defgroup xml_events XML IO for events
    Output manipulators and input handler for writing and reading
    events in XML.
    @author Written January 2002 by Masahiro Ito and Daniel Sigg
    @version 1.0
  *************************************************************************/
//@{


/** xsilEvent converts an event object to an output stream in LIGO_LW
    format as described in LIGO-T990023.
    @memo Writes events in LIGO_LW format
  *************************************************************************/
   class xsilEvent : public xsil_base {
   protected:
      /// Begin iterator
      events::ConstIterator	fBeg;
      /// End iterator
      events::ConstIterator	fEnd;
      /// Current iterator
      mutable events::ConstIterator	fCur;
      /// Maximum number of events to write
      int		fMax;
   
   public:
      /// Constructor for a single Event
      xsilEvent (const char* name, const events::Event& event, 
                int level = 1);
      /// Constructor for an event array
      xsilEvent (const char* name, const events::Event* event, 
                int n, int level = 1);
      /// Constructor for List
      xsilEvent (const char* name, const events::List& list, 
                int max = -1, int level = 1)
      : xsil_base (name, level), fBeg (list.Begin()), fEnd (list.End()),
      fCur (list.Begin()), fMax (max) {
      }
      /// Constructor for Chain
      xsilEvent (const char* name, const events::Chain& chain, 
                int max = -1, int level = 1)
      : xsil_base (name, level), fBeg (chain.Begin()), fEnd (chain.End()),
      fCur (chain.Begin()), fMax (max) {
      }
      /// Constructor for Set
      xsilEvent (const char* name, const events::Set& set, 
                int max = -1, int level = 1)
      : xsil_base (name, level), fBeg (set.Begin()), fEnd (set.End()),
      fCur (set.Begin()), fMax (max) {
      }
      /// Constructor for Iterator and ConstIterator
      xsilEvent (const char* name, events::ConstIterator beg,
                events::ConstIterator end, int max = -1, int level = 1)
      : xsil_base (name, level), fBeg (beg), fEnd (end), 
      fCur (beg), fMax (max) {
      }
      /// Write the event(s) to output stream
      std::ostream& write (std::ostream &os) const;
      /// Get the current iterator
      const events::ConstIterator& get() const {
         return fCur; }
   };


/** Write events to an output stream.
    @param os output stream
    @param ev xsilEvent object
    @memo XML output of TSeries
    @return output stream
  *************************************************************************/
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilEvent& ev) {
      return ev.write(os); }


/** Callback for adding events. This class is used by the
    event handler to add the read events to the parent.
    @memo Callback for adding events
  *************************************************************************/
   class xsilEventAdd {
   public:
      xsilEventAdd (){
      }
      virtual ~xsilEventAdd (){
      }
      virtual bool AddEvent (const events::Event& event) = 0;
   };


/** Handler for Event class
    @memo Event data type handler
  *************************************************************************/
   class xsilHandlerEvent : public xsilHandler,
   protected xsilEventAdd {
   public:
      enum redirect_info {
      col_normal = 0,
      col_name = 1,
      col_time = 2,
      col_time_ns = 3,
      col_ifo = 4
      }; // mito
   
      /// Column type, column access, special rule
      struct column_item { // mito
         /// Column type
         events::ColumnType::Enum 	fType;
         /// Column access class
         events::Column 		fColumn;
         /// Special transformation rule
         redirect_info 			fRedirect;
         /// Default constructor
         column_item () 
         : fType (events::ColumnType::kInvalid), 
         fRedirect (col_normal) {
         }      
         /// Constructor
         column_item (events::ColumnType::Enum ctype, events::Column col, 
                     redirect_info redir = col_normal)
         : fType (ctype), fColumn (col), fRedirect (redir) {
         }
      };
      /// List of columns
      typedef std::vector<column_item> column_list;
   protected:
      /// Parent handler
      xsilEventAdd*		fParent;
      /// Event layout
      events::Layout		fLayout;
      /// Event
      events::Event		fEvent;
      /// Row number of event column
      int			fEventRow;
      /// Column number of event column
      int			fEventCol;
      /// List of columns and types
      column_list		fCols;
   
   public:
      /// Constructor
      explicit xsilHandlerEvent (const attrlist *attr = 0,
                        xsilEventAdd* parent = 0);
      /// Destructor
      virtual ~xsilHandlerEvent();
   
      /// comment callback
      virtual bool CommentHandler (const std::string& comment);
      /// Parameter callback (not used)
      virtual bool ParameterHandler (int type, void* x, int size,
                        const attrlist& attr) {
         return false; }
      /// Data callback (not used)
      virtual bool DataHandler (const std::string& name,
                        int type, void* x, int size, 
                        int dim1, int dim2 = 0, int dim3 = 0,
                        int dim4 = 0) {
         return false; }
      /// table column callback
      virtual bool HandleTableColumn (int col, const std::string& name,
                        int type, const attrlist& attr);
      /// table entry callback (will call correct type)
      virtual bool TableEntryHandler (int row, int col, int type, 
                        void* x);
      /// bool table entry callback (not handled)
      virtual bool HandleTableEntry (int row, int col, bool p) {
         return HandleTableEntry (row, col, (int)p); }
      /// byte table entry callback (not handled)
      virtual bool HandleTableEntry (int row, int col, char p) {
         return false; }
      /// short table entry callback (not handled)
      virtual bool HandleTableEntry (int row, int col, short p) {
         return false; }
      /// int table entry callback (must return true if handled)
      virtual bool HandleTableEntry (int row, int col, int p);
   #ifndef __CINT__
      /// long table entry callback (not handled)
      virtual bool HandleTableEntry (int row, int col, long long p) {
         return false; }
   #endif
      /// float table entry callback (not handled)
      virtual bool HandleTableEntry (int row, int col, float p) {
         return false; }
      /// double table entry callback (must return true if handled)
      virtual bool HandleTableEntry (int row, int col, double p);
#ifndef __CINT__
      /// complex float table entry callback (not handled)
      virtual bool HandleTableEntry (int row, int col,
                        const std::complex<float>& p) {
         return false; }
      /// complex double table entry callback (must return true if handled)
      virtual bool HandleTableEntry (int row, int col,
                        const std::complex<double>& p);
#endif
      /// Time table entry callback (must return true if handled)
      virtual bool HandleTableEntry (int row, int col, const Time& p);
      /// string table entry callback (must return true if handled)
      virtual bool HandleTableEntry (int row, int col,
                        const std::string& p);
    /// Name table entry callback (must return true if handled)
      virtual bool HandleTableEntry (int row, int col, const events::Name& p);
   
     /// handler for nested data objects (not used)
      virtual xsilHandler* GetHandler (const attrlist& attr) {
         return 0; }
      /// handler for nested table/event
      virtual xsilHandler* GetTableHandler (const attrlist& attr) {
         return new xsilHandlerEvent (&attr, this); }
   
   protected:
      /// Add a column value
   template <class T>
      bool AddColumnValue (int row, int col, const T& p);
      /// Add an event (called by child handlers)
      virtual bool AddEvent (const events::Event& event);
   };


   // get a table handler for an event
   xsilHandler* queryEventHandler (const xsilHandlerQuery::attrlist& attr,
                     xsilEventAdd* parent);

/** Xsil Event handler query class.
    The query will return a handler if the data object type is a table.
    @memo Xsil Event handler query
  *************************************************************************/
template <class Output>
   class xsilHandlerQueryEvent: public xsilHandlerQuery, 
   protected xsilEventAdd {
   public:
       // Constructor
      xsilHandlerQueryEvent () : fOutput (0), fMax (0) {
      }
       /// Constructor
      explicit xsilHandlerQueryEvent (Output& output, int max = -1) :
      fOutput (&output), fMax (max) {
      }
      /// returns a handler for the specified object (not used)
      virtual xsilHandler* GetHandler (const attrlist& attr) {
         return 0; }
      /// returns a handler for the specified table (or 0 if not)
      virtual xsilHandler* GetTableHandler (const attrlist& attr) {
         return queryEventHandler (attr, this); }
   
   protected:
      /// Output class
      Output*		fOutput;
      /// Maximum number of events to read
      int		fMax;
   
      /// Add an event (called by a handler)
      virtual bool AddEvent (const events::Event& event) {
         if (fMax != 0) {
            *(*fOutput) = event; ++(*fOutput); } // Output iterator
         if (fMax > 0) --fMax;
         return (fMax != 0); }
   };


// Template Specialization
// 
//__________________________________________________________________________
template <>
   inline bool xsilHandlerQueryEvent<events::List>::AddEvent (
                     const events::Event& event) {
      if (fMax != 0) fOutput->PushBack (event); 
      if (fMax > 0) --fMax;
      return (fMax != 0);
   }

//__________________________________________________________________________
template <>
   inline bool xsilHandlerQueryEvent<events::Chain>::AddEvent (
                     const events::Event& event) {
      if (fMax != 0) fOutput->PushBack (event); 
      if (fMax > 0) --fMax;
      return (fMax != 0);
   }

//__________________________________________________________________________
template <>
   inline bool xsilHandlerQueryEvent<events::Set>::AddEvent (
                     const events::Event& event) {
      if (fMax != 0) fOutput->PushBack (event); 
      if (fMax > 0) --fMax;
      return (fMax != 0);
   }

//@}

} // namespace xml

#ifdef __CINT__
   inline std::ostream& operator<< (std::ostream& os, 
                     const xml::xsilEvent& ev) {
      return ev.write(os); }
#endif

#endif // define _GDS_XSILEVENT_HH
