#ifndef _LIGO_EVENTTIMEWINDOW_H
#define _LIGO_EVENTTIMEWINDOW_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: TimeWindow						*/
/*                                                         		*/
/* Module Description: Manages a time window				*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: TimeWindow.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include "Time.hh"
#include "Interval.hh"


namespace events {

/** A time window is a utility class which manages a time duration and
    time offset. It is used by event analysis algorithms to select
    the analysis, the veto and the coincidence window.
       
    @memo Defines a time window
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class TimeWindow {
   public:
      // Interval type
      typedef Interval interval_type;
   
      /** Creates a default time window. Duration = 1s, Offset = -0.5s.
          @memo Default constructor
       ******************************************************************/
      TimeWindow ()
      : mWidth (1.0), mOffset (-0.5) {
      }
      /** Creates a symmetric time window.
          @memo Constructor
          @param width Width of window
       ******************************************************************/
      TimeWindow (double width)
      : mWidth (width), mOffset (-width/2.) {
      }
      /** Creates a symmetric time window.
          @memo Constructor
          @param width Width of window
       ******************************************************************/
      TimeWindow (const interval_type& width)
      : mWidth (width), mOffset (-width/2.) {
      }
      /** Creates time window. For a symmetric window set the 
          offset to the negative of half the window width.
          @memo Constructor
          @param width Width of window
          @param offset Window offset
       ******************************************************************/
      TimeWindow (double width, double offset)
      : mWidth (width), mOffset (offset) {
      }
      /** Creates time window. For a symmetric window set the 
          offset to the negative of half the window width.
          @memo Constructor
          @param width Width of window
          @param offset Window offset
       ******************************************************************/
      TimeWindow (const interval_type& width, 
                 const interval_type& offset)
      : mWidth (width), mOffset (offset) {
      }
   
      /** Set Window.
          @memo Set window width
       ******************************************************************/
      void SetWindow (const interval_type& width,
		      const interval_type& offset) {
         mWidth = width; mOffset = offset; }
      /** Set width/duration of window.
          @memo Set window width
       ******************************************************************/
      void SetWidth (const interval_type& width) {
         mWidth = width; }
      /** Get width/duration of window.
          @memo Get window width
       ******************************************************************/
      interval_type GetWidth() const {
         return mWidth; }
      /** Set window offset.
          @memo Set window offset
       ******************************************************************/
      void SetOffset (const interval_type& offset) {
         mOffset = offset; }
      /** Window offset.
          @memo Window offset
       ******************************************************************/
      interval_type GetOffset() const {
         return mOffset; }
      /** Get stop time of window.
          @memo Window end
       ******************************************************************/
      interval_type GetEnd() const {
         return mOffset + mWidth; }
   
   private:
      /// Width of window in sec
      interval_type mWidth;
      /// Window offset in sec
      interval_type mOffset;
   };


}

#endif // _LIGO_EVENTTIMEWINDOW_H
