#ifndef _LIGO_EVENTLISTPTR_H
#define _LIGO_EVENTLISTPTR_H

/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: ListPtr							*/
/*                                                         		*/
/* Module Description: Defines a smart pointer for an event list	*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: ListPtr.html						*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include "events/List.hh"


namespace events {

/** A smart pointer for an event list. When copying an event list 
    smart pointer, a new event list is created which is an exact copy
    of the original.
   
    @memo Smart pointer for event lists.
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class ListPtr {
   
   public:
      /** Creates a pointer to an event list. The event list gets adopted.
          @memo Default constructor
       ******************************************************************/
      explicit ListPtr (List* ptr = 0) : mPtr (0) {
         Reset (ptr); }
      /** Creates a new copy of the event. The event is not adopted.
          @memo Constructor
       ******************************************************************/
      ListPtr (const List& eventlist) : mPtr (0) {
         Reset (eventlist); }
      /** Copies an event list.
          @memo Copy constructor
       ******************************************************************/
      ListPtr (const ListPtr& ptr) : mPtr (0) {
         *this = ptr; }
      /** Destructs the event list.
          @memo Destructor
       ******************************************************************/
      ~ListPtr () {
         Reset (0); }
      /** Assignment. Makes a new pointer with a true copy.
          @memo Assignment operator
       ******************************************************************/
      inline ListPtr& operator= (const ListPtr& ptr) {
         if (this != &ptr) Reset (ptr.Get() ? ptr->Copy() : 0); 
         return *this; }
      /** Equality. Compares the pointers (needed by CINT).
          @memo Equality operator
       ******************************************************************/
      bool operator== (const ListPtr& ptr) const {
         return mPtr == ptr.mPtr; }
     /** Smaller. Compares the pointers (needed by CINT).
         @memo Smaller operator
       ******************************************************************/
      bool operator< (const ListPtr& ptr) const {
         return mPtr < ptr.mPtr; }
      /** Get a pointer to the event.
          @memo Get
       ******************************************************************/
      List* Get() {
         return mPtr; }
      /** Get a pointer to the event.
          @memo Get
       ******************************************************************/
      const List* Get() const {
         return mPtr; }
      /** Dereference operator.
          @memo Dereference oprator
       ******************************************************************/
      List& operator*() {
         return *mPtr; }
      /** Dereference operator.
          @memo Dereference oprator
       ******************************************************************/
      const List& operator*() const {
         return *mPtr; }
      /** Member access oprator.
          @memo Member access oprator
       ******************************************************************/
      List* operator->() {
         return mPtr; }
      /** Member access oprator.
          @memo Member access oprator
       ******************************************************************/
      const List* operator->() const {
         return mPtr; }
      /** Release the event.
          @memo Release
       ******************************************************************/
      List* Release() {
         List* e = mPtr; mPtr = 0; 
         return e; }
      /** Resets the event. The new event gets adopted.
          @memo Reset
       ******************************************************************/
      void Reset (List* ptr = 0) {
         if (mPtr) delete mPtr; 
         mPtr = ptr; }
      /** Resets the event. The new event is copied.
          @memo Reset
       ******************************************************************/
      void Reset (const List& eventlist) {
         if (mPtr) delete mPtr; 
         mPtr = eventlist.Copy(); }
      /** Swap the pointers.
          @memo Swap
       ******************************************************************/
      void Swap (ListPtr& ptr) {
         List* tmp = mPtr; mPtr = ptr.mPtr; ptr.mPtr = tmp; }
   
   private:
      /** Event pointer.
          @memo Pointer to event
       ******************************************************************/
      List* mPtr;
   };


}

#endif // _LIGO_EVENTLISTPTR_H
