#ifndef _LIGO_EVENTFIXEDCOLUMNS_H
#define _LIGO_EVENTFIXEDCOLUMNS_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: FixedColumns						*/
/*                                                         		*/
/* Module Description: Defines fixed event columns			*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: FixedColumns.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include "events/ColumnType.hh"
#include "events/ColumnInfo.hh"


namespace events {


/** Fixed column handling routines. A set of routines to add and query
    the list of fixed columns. The list of fixed columns is
    handled by the event factory. This is just the interface.

    Fixed columns are stored before any additional event specific
    columns. They are stored in the order they are declared.
    A fixed column has a fixed column index and a fixed
    offset into the event data block. They are common to all
    events and can be accessed fast. Since they are common, fixed
    columns can only be added before the first event layout gets
    registered. Fixed columns cannot be removed once they have
    been added.

    Predefined fixed columns are
    \begin{verbatim}
    ColumnNumber - The number of columns within the event data block
    SubTypeId - Represents the event subtype identification number
    Time - Represents the event time
    Ifo - Represents the interferometer information (bit set)
    \end{verbatim}
   
    @memo Fixed column management
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   struct FixedColumns : public ColumnType {
      /** Add a fixed column.
          @memo Add a fixed column
          @param info Column information record
          @return true if succesful
       ******************************************************************/
      static bool Add (const ColumnInfo& info);
      /** Add a fixed column.
          @memo Add a fixed column
          @param name Column name
          @param type Column type
          @param csize Array size for string type
          @return true if succesful
       ******************************************************************/
      static bool Add (const char* name, Enum type) {
         return Add (ColumnInfo (name, type)); }
      /** Get a fixed column by name.
          @memo Get a fixed column
          @param name Column name
          @return pointer to column information record (0 if failed)
       ******************************************************************/
      static const ColumnInfo* Get (const char*);
      /** Get the fixed column list.
          @memo Get the fixed column list
          @return List of fixed columns
       ******************************************************************/
      static const ColumnInfoList& List();
      /** Get a index of the first varying column. This is the
          first index available for event specific columns.
          @memo Get a varying column index
          @return Varying column index
       ******************************************************************/
      static int VaryingNumber();
      /** Get a byte offset of the first varying column. This is the
          first address available for event specific columns.
          @memo Get a varying column offset
          @return Varying column offset
       ******************************************************************/
      static int VaryingOffset();
   };


}

#endif // _LIGO_EVENTFIXEDCOLUMNS_H
