#ifndef _LIGO_EVENTCOMPARISION_H
#define _LIGO_EVENTCOMPARISION_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: Comparison						*/
/*                                                         		*/
/* Module Description: Defines a boolean event condition for comparing	*/
/*                     event functions		           		*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: Comparison.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include "events/Condition.hh"
#include "events/ConditionPtr.hh"
#include "events/Function.hh"
#include "events/FunctionPtr.hh"
#include "events/Value.hh"


namespace events {


/** A event conditions for comparing event functions.
   
    @memo Defines an event condition for comparing event functions
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class Comparison : public Condition {
   public:
      /** Comparison operations supported by the condition.
          @memo Comparison operations.
       ******************************************************************/
      enum operation {
      /// Always true
      opEqual = 0,
      /// Always false
      opUnequal = 1,
      /// identity condition
      opLess = 2,
      /// not condition
      opLessEqual = 3,
      /// and condition
      opGreater = 4,
      /// or condition
      opGreaterEqual = 5
      };
   
      /** Creates an column condition between two columns.
          @memo Constructor
       ******************************************************************/
      Comparison (const Function& f1, const Function& f2, 
                 operation op, double tolerance = 0.0)
      : mOp (op), mFPtr1 (f1), mFPtr2 (f2), mCPtr1 (0),
      mTolerance (tolerance) {
      }
      Comparison (const FunctionPtr& f1, const FunctionPtr& f2, 
                 operation op, double tolerance = 0.0)
      : mOp (op), mFPtr1 (f1), mFPtr2 (f2), mCPtr1 (0),
      mTolerance (tolerance) {
      }
      /** Creates an column condition between a condition and a fucntion.
          This allows for expression such as 
          "3 < Column ("Amplitude") < 5".
          @memo Constructor
       ******************************************************************/
      Comparison (const Comparison& cond, 
                 const Function& f, operation op, 
                 double tolerance = 0.0)
      : mOp (op), mFPtr1 (0), mFPtr2 (f), mCPtr1 (cond),
      mTolerance (tolerance) {
      }
      Comparison (const Comparison& cond, 
                 const FunctionPtr& f, operation op, 
                 double tolerance = 0.0)
      : mOp (op), mFPtr1 (0), mFPtr2 (f), mCPtr1 (cond),
      mTolerance (tolerance) {
      }
   
      /** Returns a copy of the event condition. This method must be 
          overriden by all descendents.
          @memo Copy the event condition
          @return event copy
       ******************************************************************/
      virtual Comparison* Copy() const {
         return new Comparison (*this); }
   
      /** Evaluate the condition.
          @memo Evaluate
          @param arg Event argument list
          @param val Condition value (return)
          @return True if condition could be evaluated
       ******************************************************************/
      virtual bool Evaluate (const Argument& arg, bool& val) const;
   
   protected:
      /** Default constructor.
          @memo Default constructor
       ******************************************************************/
      Comparison () : mOp (opEqual) {
      }
      /** Get the right hand side value of the left hand side argument!
          @memo Right hand side of left hand side
          @param arg Event argument list
          @param x Right hand side value
          @return True if value is defined
       ******************************************************************/
      virtual bool ValueR (const Argument& arg, Value& val) const;
   
   private:
      /// Operation
      operation 	mOp;
      /// First argument if event function
      FunctionPtr	mFPtr1;
      /// Second argument if event function
      FunctionPtr	mFPtr2;
      /// First argument if comparision condition
      ConditionPtr 	mCPtr1;
      /// Use for determining if two real values are the same
      double 		mTolerance;
   };


#ifndef __CINT__
/** @name Comparison operators
    The following comparison operations are supported:
    \begin{verbatim}
    == (equal)        != (unequal)     < (less)
    <= (less equal)   > (greater)      >= (greater equal)
    \end{verbatim}
    The arguments of a comparision operation can be:
    \begin{verbatim}
    Function    <op> Function
    Function    <op> column type
    column type <op> Function
    Comparison <op> Function
    Comparison <op> column type
    \end{verbatim}
    The last two operator groups are necessary to support constructs
    such as 
    \begin{verbatim}
    5 < Column ("Amplitude") < 10
      or
    Column ("Time(0)") == 
       Column ("Time(1)") ==
          Column ("Time(2)")
    \end{verbatim}
    The last expression tests whether Time(0) and Time(1) are
    equal and whether Time(1) and Time(2) are equal as well.
   
    @memo Defines an event condition comparison operators
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
//@{
//@Include: ComparisonOps.hh
//@}
#include "ComparisonOps.hh"
#endif


/** @name WithIn
    The WithIn function test whether two fcuntion values
    are within a certain tolerance. Example:
    \begin{verbatim}
    WithIn ("Time(0)", "Time(1)", 2.0)
    \end{verbatim}
    returns an event condition which is true, if the time stamps of 
    the two events are within 2 sec of each other.
   
    @memo WithIn operator
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
//@{

   /** Compare event functions (equal with tolerance).
       @memo Compare event functions (equal with tolerance)
       @return true if within tolerance
    *********************************************************************/
   inline Comparison WithIn (const Function& f1,
                     const Function& f2, double d) {
      return Comparison (f1, f2, Comparison::opEqual, d); }
   inline Comparison WithIn (const FunctionPtr& f1,
                     const FunctionPtr& f2, double d) {
      return Comparison (f1, f2, Comparison::opEqual, d); }

   /** Compare event function and real value (equal with tolerance).
       @memo Compare event functions (equal with tolerance)
       @return true if within tolerance
    *********************************************************************/
   inline Comparison WithIn (const Function& f, double x, double d) {
      return Comparison (f, Value (x), Comparison::opEqual, d); }
   inline Comparison WithIn (const FunctionPtr& f, double x, double d) {
      return Comparison (f, Value (x), Comparison::opEqual, d); }

   /** Compare real value and event function (equal with tolerance).
       @memo Compare event functions (equal with tolerance)
       @return true if within tolerance
    *********************************************************************/
   inline Comparison WithIn (double x, const Function& f, double d) {
      return Comparison (Value (x), f, Comparison::opEqual, d); }
   inline Comparison WithIn (double x, const FunctionPtr& f, double d) {
      return Comparison (Value (x), f, Comparison::opEqual, d); }

//@}
}

#endif // _LIGO_EVENTCOMPARISION_H
