/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "PConfig.h"
#include "TrigStream.hh"
#include "TrigTable.hh"
#include "xsil/ligolw.hh"
#include "xsil/Xwriter.hh"
#include "html/table.hh"
#include "html/document.hh"
#include <sys/stat.h>
#include <sys/types.h>
#include <fstream>
#include <sstream>
#include <unistd.h>
#include <signal.h>
#include <cmath>
#include <cstdlib>

using namespace std;

//======================================  Get a unique ID
static unsigned long mUniqueID(0);

string
getUnique(const Time& idTime) {
    Time temp(idTime);
    temp.setN(mUniqueID++);
    char line[128];
    TimeStr(temp, line, "%Y%02m%02d%02H%02N%02S%06n010000");
    return string(line);
}

//======================================  Print first N digits of an integer
static int
put_int(string& s, unsigned long d, int count) {
    unsigned long  dig = d%10;
    unsigned long rest = d/10;
    int     N = 0;
    if (rest) N = put_int(s, rest, count);
    if (!count || N < count) s += char('0' + dig);
    return N+1;
}

void
frog(const string& s) {
#if defined (P__DARWIN)
    sigignore(SIGCHLD);
#else
    sigignore(SIGCLD);
#endif

    pid_t pid = fork();
    if (pid != 0) {
	if (pid == -1) perror("fork failed");
	return;
    }
    string a(s);
    a += char(0);
    const char* argv[32];
    argv[0] = 0;
    char* p = &a[0];
    for (int i=0; i<31 && *p; ) {
        while (*p == ' ') *p++ = 0;
	if (*p) argv[i] = p;
	while (*p && *p != ' ') p++;
	argv[++i] = 0;
    }
    exit(execvp(argv[0], const_cast<char* const*>(argv)));
}

//======================================  Parameter constructor
TrigStream::param::param(void)
  : _Inactive(300.0), _MaxDelay(300.0), _TableCount(defTableCount)
{}

//======================================  Set parameter mode
Time
TrigStream::param::align(const Time& t) const {
    Time::ulong_t mod = t.getS() % Time::ulong_t(_MaxDelay);
    return Time(t.getS() - mod);
}

//======================================  Set parameter mode
void
TrigStream::param::setMode(time_mode m, Interval i, Interval d) {
    _Mode     = m;
    _Inactive = i;
    _MaxDelay = d;
}

//======================================  Statistics constructor
TrigStream::stream_stats::stream_stats(void)
  : errTot(0), fileTot(0), segTot(0), trigTot(0)
{
    lastUpdate = Now();
    lastWrite  = lastUpdate;
}

//======================================  Count a segment
void
TrigStream::stream_stats::count_seg(void) {
    segTot++;
    lastUpdate = Now();
}

//======================================  Count a trigger
void
TrigStream::stream_stats::count_trig(void) {
    trigTot++;
    lastUpdate = Now();
}

//======================================  Count a trigger
void
TrigStream::stream_stats::count_write(const string& file, const Time& end) {
    fileTot++;
    lastWrite = end;
    lastFile  = file;
}

//======================================  Stream constructor
TrigStream::TrigStream(const std::string& name, mask_type mask)
    : mName(name), mDebug(0), mDestMsk(mask), mExcludeMsk(0)
{
}

TrigStream::~TrigStream(void) {
}

//======================================  Add a process
void 
TrigStream::addProcess(const Process& pdb) {
    mProcDB[pdb.refData().getProcessID()] = pdb;
}

//======================================  Add a segment
bool
TrigStream::addSegment(const trig::Segment& seg, mask_type enable) {
   return true;
}

//======================================  Add a Trigger 
bool
TrigStream::addTrigger(const trig::TrigBase& trg, mask_type enable) {
    return true;
}

//======================================  Create the directory path
bool 
TrigStream::createPath(const std::string& path, int depth) const {
    string::size_type inx = path.rfind("/");
    if (inx == string::npos) return true;
    string dpath = path.substr(0, inx);
    if (!access(dpath.c_str(), W_OK)) return true;
    if (!depth || !createPath(dpath, depth-1)) return false;
    if (!mkdir(dpath.c_str(), 0775)) return true;
    return false;
}

//=======================================  Get trigger and segment counts
bool
TrigStream::empty(const Time& t) const {
    return !getNSegs(t) && !getNTrigs(t);
}

TrigStream::count_type
TrigStream::getNSegs(const Time& t) const {
    return 0;
}

TrigStream::count_type
TrigStream::getNTrigs(const Time& t) const {
    return 0;
}

//======================================  Generate the output file path
string 
TrigStream::getPath(const Time& start, const Time& end) const {
    Time::ulong_t tStart = start.getS();
    Time::ulong_t tEnd   = end.getS();
    if (end.getN()) tEnd++;
    string s;
    string::size_type N = mParam._OutDir.size();
    for (string::size_type i=0; i<N; ++i) {
	if (mParam._OutDir[i] != '%') {
	    s += mParam._OutDir[i];
	} else {
	    int count = 0;
	    char fmt =  mParam._OutDir[++i];
	    while (fmt >= '0' && fmt < '9') {
		count *= 10;
		count += fmt - '0';
		fmt =  mParam._OutDir[++i];
	    }
	    switch(fmt) {
	    case 'g':
		put_int(s, tStart, count);
		break;
	    case 'r':
		put_int(s, tStart/int(pow(10.0,count)+0.5), 0);
		break;
	    case 'd':
		put_int(s, tEnd-tStart, count);
		break;
	    case 'u':
		s += getUnique(start);
		break;
	    default:
		s += fmt;
		break;
	    }
	}
    }
    return s;
}

//======================================  Get estimated timeout
Interval 
TrigStream::getTimeOut(const Time& now) const {
    Interval timeOut(0);
    switch (mParam._Mode) {
    case param::kCount: {
	Interval dTWrite  = mStats.lastWrite  + mParam._MaxDelay - now;
	if (double(dTWrite) > 0) timeOut = dTWrite;
	Interval dTUpdate = mStats.lastUpdate + mParam._Inactive - now;
	if (double(dTUpdate) > 0 && dTUpdate < timeOut) timeOut = dTUpdate;
	break;
    }
    case param::kTimer: {
	Time t0 = mStats.lastWrite;
	if (!t0) t0  = mParam.align(now - mParam._Inactive);
	if (now < t0 + mParam._Inactive) timeOut = t0 - now + mParam._Inactive;
    }
    }
    return timeOut;
}

//======================================  Perform trigger file post-processing
void 
TrigStream::postProcess(const std::string& file) const {
    const char* 
	symc="abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ_0123456789";

    //----------------------------------  Get the insertion command
    string insert = refParams()._Insert;
    if (insert.empty()) return;

    //----------------------------------  Scan string, replace environment vbls
    for (string::size_type i=0; i<insert.size();) {
	string::size_type k = i+1;
	if (insert[i] == '$') {
	    string::size_type j=k;
	    string env;
	    if (insert[j] == '{') {
		j++;
		k = insert.find("}", j);
		if (k == string::npos) {
		    cerr << "Missing '}' in insert string" << endl;
		    return;
		}
		env = insert.substr(j, k-j);
		k++;
	    } else {
		k = insert.find_first_not_of(symc, j);
		if (k == string::npos) k = insert.size();
		env = insert.substr(j, k-j);
	    }
	    if (! env.empty()) {
		const char* enval = getenv(env.c_str());
		if (enval) {
		    insert.replace(i, k-i, enval);
		} else if (env == "Xml_File") {
		    insert.replace(i, k-i, file);
		} else {
		    cerr << "Undefined symbol: " << env << endl;
		    insert.erase(i, k-i);
		}
	    } else {
		insert.erase(i, k-i);
	    }
	}
	i = k;
    }

    //----------------------------------  Issue the insertion command.
    frog(insert);
}

//======================================  Purge the process DB
void 
TrigStream::PurgeProcDB(void) {
}

//======================================  Get estimated timeout
void 
TrigStream::removeProc(const std::string& ident) {
    if (mProcDB.find(ident) == mProcDB.end()) return;
    mProcDB[ident].setClosed();
    if (!mProcDB[ident].isReleasable()) mProcDB.erase(ident);  
}

//======================================  Default segment printout
void 
TrigStream::segStats(html::document& doc) const {
    return;
}

//======================================  Set parameters
void 
TrigStream::setDebug(int lvl) {
    mDebug=lvl;
}

void 
TrigStream::setDestMsk(mask_type mask) {
    mDestMsk = mask;
}

void 
TrigStream::setExcludeMsk(mask_type mask) {
    mExcludeMsk = mask;
}

void 
TrigStream::setParam(const param& par) {
    mParam = par;
}

//======================================  Test destination flags
bool 
TrigStream::testWrite(const trig::Segment& seg, mask_type dest) const {
    return (dest & mDestMsk) != 0 && (dest & mExcludeMsk) == 0;
}

bool 
TrigStream::testWrite(const trig::TrigBase& tr, mask_type dest) const {
    mask_type msgd = tr.getDisposition();
    return (dest & mDestMsk & msgd) != 0 && (dest & mExcludeMsk & msgd) == 0;
}

//======================================  See if its time to write
bool
TrigStream::writeNow(bool force) const {
    bool rc = false;
    Time now = Now();
    if (mParam._Mode == param::kTimer) {
	Time t(mParam.align(now - mParam._Inactive));
	rc = !empty(t) && (force || t > mStats.lastWrite + mParam._MaxDelay);
    } else {
	if (empty(Time(0))) {
	    rc = false;
	} else if (force) {
	    rc = true;
	} else if (getNTrigs(Time(0)) >= mParam._TableCount || 
		   getTimeOut(now) <= Interval(0)) {
	    rc = true;
	}
    }
    return rc;
}

//======================================  Write DB
void 
TrigStream::WriteDB(bool force) {
}

//======================================  Write data to stream
bool
TrigStream::write(bool force, trig::TrigWriter& wrt) {
    //----------------------------------  Get the start time.
    Time start(0), end(0);
    if (!mStats.lastWrite) start = wrt.getEarly();
    else                   start = mStats.lastWrite;

    Time now = Now();
    Interval delta = mParam._MaxDelay;
    if (mParam._Mode == param::kTimer) {
	//------------------------------  Get the end time
	start = mParam.align(start);
	end   = start + delta;
	if (now < end + mParam._Inactive) {
	    if (!force) {
		return false;
	    } else {
		end = now;
	    }
	}
	if (empty(end)) {
	    mStats.lastWrite = end;
	    return false;
	}
    } else {
	end = now;
	if (empty(Time(0))) return false;
    }

    string path = getPath(start, end);
    if (createPath(path, 1)) {
	wrt.write(path, start, end);
	wrt.clear(start, end);
	mStats.count_write(path, end);
    } else { 
	cerr << "Unable to create path for: " << path << endl;
    }
    return true;
}

//======================================  Update
void
TrigStream::tick(const Time& now) {
    if (empty(now)) {
	mStats.lastUpdate = now;
    }
}

//======================================  Epics stream class
EpxStream::EpxStream(const std::string& name, mask_type mask)
  : TrigStream(name, mask)
{}

//======================================  Epics stream destructor
EpxStream::~EpxStream(void) {
}

//======================================  Clone an Epics stream
EpxStream*
EpxStream::clone(void) const {
    return new EpxStream(*this);
}

//======================================  Send the trigger to Epics
bool
EpxStream::addTrigger(const trig::TrigBase& trg, mask_type enable) {
    if (!testWrite(trg, enable)) return false;
    string procid = trg.getProcess();
    proc_iter   iProc  = mProcDB.find(procid);
    if (iProc == mProcDB.end()) return false;
 
    stringstream ingest;
    ingest << "epics_alarm ";      // submission command
    switch (trg.getPriority()) {
    case trig::p_info:
        ingest << "Information ";
	break;
    case trig::p_warn:
        ingest << "Warning ";
	break;
    case trig::p_error:
        ingest << "Error ";
	break;
    case trig::p_severe:
        ingest << "Severe ";
	break;
    }
    ingest << trg.getTime().getS() << " \"";
    ingest << iProc->second.refData().getName() << " ";
    ingest << trg.getID() << " " << trg.getSubID();
    ingest << "\" > /dev/null";
    if (getDebug()) cout << "Epics Alarm: " << ingest.str() << endl;
    frog(ingest.str());
    mStats.count_trig();
    return true;
}

//======================================  Ldas DB stream class
LdasStream::LdasStream(const std::string& name, mask_type mask)
  : TrigStream(name, mask)
{
    mParam._Insert = "SeqInsert -host ${DMTCOMNODE} submit ${Xml_File}";
}

//======================================  Ldas DB stream destructor
LdasStream::~LdasStream(void) {
}

//======================================  Clone an Ldas stream
LdasStream*
LdasStream::clone(void) const {
    return new LdasStream(*this);
}

//======================================  Add a segment
bool
LdasStream::addSegment(const trig::Segment& seg, mask_type enable) {
    if (!testWrite(seg, enable)) return false;
    std::string procid = seg.getProcess();
    proc_iter   iProc  = mProcDB.find(procid);
    if (iProc == mProcDB.end()) return false;
    iProc->second.incrementWrite();
    trig::Segment s(seg);
    string grpName(s.getGroup());
    if (*s.getIfos() != 0 && grpName.find(":") == string::npos) {
	grpName.insert(0,":");
	grpName.insert(0,s.getIfos());
	s.setGroup(grpName.c_str());
    }
    mWriter.addSegment(s, iProc->second.refData());
    mStats.count_seg();
    return true;
}

//======================================  Add a Trigger 
bool
LdasStream::addTrigger(const trig::TrigBase& trg, mask_type enable) {
    if (!testWrite(trg, enable)) return false;
    std::string procid = trg.getProcess();
    proc_iter   iProc  = mProcDB.find(procid);
    if (iProc == mProcDB.end()) return false;
    iProc->second.incrementWrite();
    mWriter.addTrigger(trg, iProc->second.refData());
    mStats.count_trig();
    return true;
}

//======================================  Data size methods
LdasStream::count_type
LdasStream::getNSegs(const Time& t) const {
    return mWriter.getNSegs(t);
}

LdasStream::count_type
LdasStream::getNTrigs(const Time& t) const {
    return mWriter.getNTrigs(t);
}

//======================================  Write DB
void 
LdasStream::WriteDB(bool force) {
    while ( writeNow(force) ) {
	if (write(force, mWriter)) {
	    postProcess(mStats.lastFile);
	}
    }
}

//======================================  Segment DB stream class
SegDBStream::SegDBStream(const string& name, mask_type mask)
  : TrigStream(name, mask)
{
    mParam._Insert="dmtdq_seg_insert --server $DMTSEGSERVER --file $Xml_File";
}

//======================================  Segment DB stream destructor
SegDBStream::~SegDBStream(void) {
}

//======================================  Clone an Segment DB stream
SegDBStream*
SegDBStream::clone(void) const {
    return new SegDBStream(*this);
}

//======================================  Add a segment
bool
SegDBStream::addSegment(const trig::Segment& seg, mask_type enable) {
    if (!testWrite(seg, enable)) return false;
    std::string procid = seg.getProcess();
    proc_iter   iProc  = mProcDB.find(procid);
    if (iProc == mProcDB.end()) return false;
    iProc->second.incrementWrite();
    mWriter.addSegment(seg, iProc->second.refData());
    mStats.count_seg();
    return true;
}

//======================================  Data size methods
SegDBStream::count_type
SegDBStream::getNSegs(const Time& t) const {
    return mWriter.getNSegs(t);
}

//======================================  Write DB
void 
SegDBStream::WriteDB(bool force) {
    while ( writeNow(force) ) {
	if (write(force, mWriter)) {
	    postProcess(mStats.lastFile);
	}
    }
}

//======================================  Segment stats
struct segstat {
    string _name;
    Time   _first;
    Time   _last;
    long   _count;
};

typedef vector<segstat> stat_list;
typedef stat_list::iterator stat_iter;

//======================================  Default segment printout
void 
SegDBStream::segStats(html::document& doc) const {
    stat_list StatList;
    const trig::S6SegWriter::seg_list* p = &mWriter.refSegList();
    for (trig::S6SegWriter::const_seg_iter i=p->begin(); i!=p->end(); ++i) {
	int N=StatList.size();
        int f=N;
	for (int j=0; j<f; ++j) {
	    if (StatList[j]._name == i->getGroup()) f=j;
	}
	if (f < N) {
	    StatList[f]._count++;
	    if (i->getStartTime() < StatList[f]._first) 
		StatList[f]._first = i->getStartTime();
	    if (i->getEndTime() > StatList[f]._last) 
		StatList[f]._last = i->getEndTime();
	} else {
	    segstat nstat;
	    nstat._name = i->getGroup();
	    nstat._first = i->getStartTime();
	    nstat._last = i->getEndTime();
	    nstat._count = 1;
	    StatList.push_back(nstat);
	}
    }
    if (StatList.empty()) return;

    //----------------------------------  Create an html table
    html::table tbl;
    tbl.addColumn("Segment Name");
    tbl.addColumn("Begin Time");
    tbl.addColumn("End Time");
    tbl.addColumn("Count");

    //----------------------------------  Fill the table
    int N=StatList.size();
    for (int j=0; j<N; ++j) {
	int row = tbl.addRow();
	tbl.insertData(row, 0, StatList[j]._name);
	tbl.insertData(row, 0, long(StatList[j]._first.getS()));
	tbl.insertData(row, 0, long(StatList[j]._last.getS()));
	tbl.insertData(row, 0, StatList[j]._count);
    }
    doc.addObject(tbl);
}
