/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "TrigClient.hh"
#include "DQSegWriter.hh"
#include "LdasDBWriter.hh"
#include "S6SegWriter.hh"
#include "SBTrigWriter.hh"
#ifndef DMTOFFLINE
#include "MsgWriter.hh"
#endif
#include "lmsg/ErrorList.hh"
#include <iostream>
#include <cstdlib>

using namespace trig;
using namespace std;

extern trig::TrigProc* ProcIdent;

trig::TrigProc* ProcIdent = 0;

//======================================  Construct and register the client.
TrigClient::TrigClient(void) 
  : mDebug(0), mWriter(0)
{
    //  Note that the default enrolment is performed when a trigger or 
    //  segment is sent. The monitor writer should use enroll(kNone) 
    //  for early registration of default connection if needed.
    //enroll(TrigWriter::kNone);
}

//======================================  Construct and register the client.
TrigClient::TrigClient(TrigWriter::trig_mode mode) 
  : mDebug(0), mWriter(0)
{
    if (mode != TrigWriter::kNone) enroll(mode);
}

//======================================  Unregister and go away
TrigClient::~TrigClient(void) {
    close();
}

//======================================  Close and delete the writer.
void
TrigClient::close(void) {
    if  (mWriter) {
	mWriter->write(mTableFile, Time(0), Time(0));
	delete mWriter;
	mWriter=0;
    }
}

//======================================  Register with the trigger manager.
lmsg::error_type 
TrigClient::enroll(TrigWriter::trig_mode mode) {
    const char* trigf = getenv("TRIGGER_FILE");
    const char* segf  = getenv("SEGMENT_FILE");
    const char* seg6  = getenv("S6SEGMENT_FILE");
    const char* snglb = getenv("SBTRIGGER_FILE");

    lmsg::error_type rc = lmsg::OK;

    //----------------------------------  Make sure process was identified.
    if (!ProcIdent) {
	cerr << "TrigClient: Process Identifier not specified" << endl;
	return lmsg::Invalid;
    }

    //----------------------------------  Handle ldas trigger writer
    if (mode == TrigWriter::kWriter || (!mWriter && trigf)) {
	if (trigf) mTableFile = trigf;
	if (!mWriter) mWriter = new LdasDBWriter;
	if (mDebug) cout << "Trigger output to file: " << mTableFile << endl;
    }

    //----------------------------------  Handle segment writer
    else if (mode == TrigWriter::kSegWrt || (!mWriter && segf)) {
	if (segf) mTableFile = segf;
	if (!mWriter) mWriter = new DQSegWriter;
	if (mDebug) cout << "Segment output to file: " << mTableFile << endl;
    }

    //----------------------------------  Handle S6 segment writer
    else if (mode == TrigWriter::kS6Seg || (!mWriter && seg6)) {
	if (seg6) mTableFile = seg6;
	if (!mWriter) mWriter = new S6SegWriter;
	if (mDebug) cout << "Segment output to file: " << mTableFile << endl;
    }

    //----------------------------------  Handle sngl_burst trigger writer
    else if (mode == TrigWriter::kSBTrig || (!mWriter && snglb)) {
	if (snglb) mTableFile = snglb;
	if (!mWriter) mWriter = new SBTrigWriter;
	if (mDebug) cout << "Trigger output to file: " << mTableFile << endl;
    }

#ifndef DMTOFFLINE
    //----------------------------------  look up the trigger manager.
    else if (!mWriter) {
	mWriter = new MsgWriter;
	if (mDebug) cout << "Output to Trigger Manager"<< endl;
    }
#endif
    if (mWriter) {
        mWriter->setDebug(mDebug);
	rc = mWriter->setProcess(*ProcIdent);
	if (rc) {
	    cerr << "TrigClient::enroll: setProcess error: " << rc << endl;
	    close();
	}
    }
    return rc;
}

//======================================  Log a segment
lmsg::error_type 
TrigClient::flush(const Time& start, const Time& end) {
    lmsg::error_type rc = 0;
    if (mWriter) {
	char file[1024];
	Time vStart(start);
	vStart.setN(long(end-start));
	TimeStr(vStart, file, mTableFile.c_str());
	cout << "Write data. Start: " << start << " end: " << end 
	     << " file: " << file << endl;
	rc = mWriter->write(file, start, end);
	mWriter->clear(start, end);
    } else {
	rc = lmsg::NotOpen;
    }
    return rc;
}

//======================================  Log a segment
lmsg::error_type
TrigClient::sendSegment(const Segment& seg) {
    lmsg::error_type rc = lmsg::OK;
    if (!mWriter) rc = enroll(TrigWriter::kNone);
    if (mWriter) {
	rc = mWriter->addSegment(seg);
	if (rc == lmsg::BadAddress) {
	    cerr << "TrigClient: Bad address detected, writer closed." << endl;
	    close();
	}
	else if (rc == lmsg::TimeOut) {
	    cerr << "TrigClient: Timeout detected, writer closed." << endl;
	    close();
	}
    }
    return rc;
}

//======================================  Log a trigger
lmsg::error_type
TrigClient::sendTrigger(const TrigBase& trigger) {
    lmsg::error_type rc = lmsg::OK;
    if (!mWriter) rc = enroll(TrigWriter::kNone);
    if (mWriter) {
	rc = mWriter->addTrigger(trigger);
	if (rc == lmsg::BadAddress) {
	    cerr << "TrigClient: Bad address detected, writer closed." << endl;
	    close();
	}
	else if (rc == lmsg::TimeOut) {
	    cerr << "TrigClient: Timeout detected, writer closed." << endl;
	    close();
	}
    }
    return rc;
}

//======================================  Set the output file name
void
TrigClient::setTableFile(const char* name) {
    mTableFile = name;
}

//======================================  Set the debug level.
void 
TrigClient::setDebug(lmsg::index_type level) {
    mDebug = level;
    if (mWriter) mWriter->setDebug(level);
}
