/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "S6SegWriter.hh"
#include "S6SegTable.hh"
#include "xsil/Xwriter.hh"
#include "xsil/ligolw.hh"
#include "lmsg/ErrorList.hh"
#include <fstream>
#include <sstream>
#include <algorithm>
#include <stdexcept>

using namespace std;
using namespace trig;

typedef std::list<S6SegDef>   segdef_list;
typedef segdef_list::iterator segdef_iter;

//======================================  Segment writer constructor
S6SegWriter::S6SegWriter(void) {
}

//======================================  Segment writer destructor
S6SegWriter::~S6SegWriter(void) {
}

//======================================  Accept a trigger for writing.
lmsg::error_type
S6SegWriter::addTrigger(const TrigBase& t, const TrigProc& p) {
    throw std::runtime_error("mySQl database doesn't handle triggers");
    return lmsg::OK;
}

//======================================  Accept a segment for writing.
lmsg::error_type
S6SegWriter::addSegment(const Segment& s,  const TrigProc& p) {
    if (!mSegList.empty() && mSegList.back() == s) {
	cerr << "Segment repeated. Group: " << s.getGroup() << endl;
	return lmsg::Invalid;
    }
    TrigProc& proc = mProcList.insert(p);
    mSegList.push_back(s);
    mSegList.back().setProcess(proc.getProcessID());
    TrigProc::gps_t tEnd = s.getEndTime().getS();
    if (tEnd > proc.getEndTime()) proc.setEndTime(tEnd);
    return lmsg::OK;
}

//======================================  Clear the tables
void 
S6SegWriter::clear(const Time& start, const Time& end) {

    //----------------------------------  Clear the used segments
    for (seg_iter i=mSegList.begin(); i != mSegList.end(); ) {
	if (!end) {
	    mProcList.unuse(i->getProcess());
	    i = mSegList.erase(i);
	} else if (i->getStartTime() >= end) {
	    i++;
	} else if (i->getEndTime() > end) {
	    (i++)->setStartTime(end);
	}  else {
	    mProcList.unuse(i->getProcess());
	    i = mSegList.erase(i);
	}
    }
}

//======================================  Get the earliest segment start time.
Time
S6SegWriter::getEarly(void) const {
    Time t(0);
    for (const_seg_iter i=mSegList.begin(); i!=mSegList.end(); ++i) {
	if (!t || i->getStartTime() < t) t = i->getStartTime();
    }
    return t;
}

//========================================  Data size methods
int
S6SegWriter::getNSegs(const Time& t) const {

    //------------------------------------  Return total size if t == 0 
    if (!t) return mSegList.size();

    //------------------------------------  Count the segments before t.
    int N = 0;
    for (const_seg_iter i=mSegList.begin(); i!=mSegList.end(); ++i) {
	if (i->getStartTime() < t) N++;
    }
    return N;
}

//======================================  Accept a segment for writing.
lmsg::error_type
S6SegWriter::setProcess(const TrigProc& p) {
    refProcess()  = p;
    return lmsg::OK;
}

//======================================  Write out the tables
lmsg::error_type
S6SegWriter::write(const string& file, const Time& start,
		   const Time& end) const {
    if (mSegList.empty()) return lmsg::OK;

    //----------------------------------  Build the process table
    ProcTable* pTab = new ProcTable(true);
    mProcList.put(*pTab);
  
    //----------------------------------  Build the segment tables
    S6SegDefTable* dTab = new S6SegDefTable;
    S6SegTable*    sTab = new S6SegTable;
    S6SegDefList   sdl;
    S6SummaryList  suml;
    int segN(0);
    for (const_seg_iter i=mSegList.begin(); i != mSegList.end(); ++i) {
	if (!end || i->getStartTime() < end) {
	    Segment seg(*i);
	    if (end != Time(0) && seg.getEndTime() > end) seg.setEndTime(end);

	    //--------------------------  Find the segment definition
	    S6SegDefList::const_segdef_iter df = sdl.find(seg);
	    if (df == sdl.end()) {
		S6SegDef sd(seg);
		sd.setDefinerID(dTab->citeTable("segment_def_id",sdl.size()));
		df = sdl.add(sd);
		dTab->addRow(sd);
	    }

	    //--------------------------  Add the segment
	    if (seg.getActivity() > 0) {
		seg.setSegID(sTab->citeTable("segment_id", segN++));
		sTab->addRow(seg, *df);
	    }

	    //--------------------------  Add the segment to the summary list.
	    suml += seg;
	}
    }

    //----------------------------------  Build up a document
    xsil::ligolw lwfile;
    lwfile.addObject(pTab);
    lwfile.addObject(dTab);
    if (segN) lwfile.addObject(sTab);
    else      delete sTab;
    S6SummaryTable* mTab = new S6SummaryTable;
    suml.put(*mTab, sdl);
    lwfile.addObject(mTab);

    //----------------------------------  Fudge a temporary file name.
    string tempfile = file;
    string::size_type inx = tempfile.find_last_of('.');
    if (inx != string::npos) tempfile.erase(inx);
    tempfile += ".tmp";
    inx = tempfile.find_last_of('/');
    if (inx == string::npos) inx = 0;
    else                     inx++;
    tempfile.insert(inx, ".");

    //----------------------------------  Write it to the output file
    std::ofstream str(tempfile.c_str());
    xsil::Xwriter xw(str);
    lwfile.Spew(xw);
    str.close();
    if (str.fail()) {
	cerr << "Writing to file: " << file << " failed!" << endl;
    } else {
	rename(tempfile.c_str(), file.c_str());
    }
    return lmsg::OK;
}
