/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef S6SEGTABLE_HH
#define S6SEGTABLE_HH

#include "xsil/MetaTable.hh"
#include "ProcTable.hh"
#include "Time.hh"
#include <vector>
#include <string>

namespace trig {
    class Segment;
    class TrigBase;
    class TrigProc;

    //=====================================================================
    //
    //   The S6 database needs a few tables
    //
    //=====================================================================

    class S6SegDef {
    public:
	S6SegDef(void);
	S6SegDef(const Segment& seg);
	S6SegDef(const std::string& name, const std::string& ifo, 
		 int version); 
	bool operator==(const S6SegDef& x) const;
	int getVersion(void) const;
	const std::string& refProcess(void) const;
	const std::string& refDefinerID(void) const;
	const std::string& refIfos(void) const;
	const std::string& refName(void) const;
	const std::string& refComment(void) const;
	void setDefinerID(const std::string& defid);
    private:
	std::string mProcID;
	std::string mDefID;
	std::string mIfos;
	std::string mName;
	int         mVersion;
	std::string mComment;
    };

    inline int 
    S6SegDef::getVersion(void) const {
	return mVersion;
    }

    inline const std::string& 
    S6SegDef::refProcess(void) const {
	return mProcID;
    }
    inline const std::string& 
    S6SegDef::refDefinerID(void) const {
	return mDefID;
    }

    inline const std::string& 
    S6SegDef::refIfos(void) const {
	return mIfos;
    }

    inline const std::string& 
    S6SegDef::refName(void) const {
	return mName;
    }

    inline const std::string& 
    S6SegDef::refComment(void) const {
	return mComment;
    }

    class S6SegDefList {
    public:
	typedef std::vector<S6SegDef> segdef_list;
	typedef segdef_list::iterator segdef_iter;
	typedef segdef_list::const_iterator const_segdef_iter;
    public:
	S6SegDefList(void);
	~S6SegDefList(void);
	const_segdef_iter add(const S6SegDef& sd);
	const_segdef_iter end(void) const;
	const_segdef_iter find(const Segment& s) const;
	const_segdef_iter find(const S6SegDef& sd) const;
	int size(void) const;
    private:
	segdef_list mList;
    };

    inline int
    S6SegDefList::size(void) const {
	return mList.size();
    }

    inline S6SegDefList::const_segdef_iter
    S6SegDefList::end(void) const {
	return mList.end();
    }

    inline S6SegDefList::const_segdef_iter 
    S6SegDefList::find(const Segment& s) const {
	return find(S6SegDef(s));
    }

    /**  The SegTable class is a meta-database interface producing a segment
     *  database table.
     */
    class S6SegTable : public xsil::MetaTable {
    public:
	S6SegTable(void);
	~S6SegTable(void);
	void addRow(const Segment& s, const S6SegDef& sd);
	void check(const char* title) const;
    private:
	DBTypes::integer_4 start_time;
	DBTypes::integer_4 end_time;
	DBTypes::varchar   segid;
	DBTypes::varchar   defid;
	DBTypes::varchar   procid;
    };

    /**  The S6SegDefTable class is a meta-database interface producing
      *  a segment definition database table.
      */
    class S6SegDefTable : public xsil::MetaTable {
    public:
	S6SegDefTable(void);
	~S6SegDefTable(void);
	void addRow(const S6SegDef& df);
	void check(const char* title) const;
    private:
	DBTypes::varchar   defid;
	DBTypes::varchar   ifos;
	DBTypes::varchar   name;
	DBTypes::integer_4 version;
	DBTypes::varchar   comment;
	DBTypes::varchar   procid;
    };

    /**  The SummaryTable class is a meta-database interface producing a
     *  segment summary table.
     */
    class S6SummaryTable : public xsil::MetaTable {
    public:
	S6SummaryTable(void);
	~S6SummaryTable(void);
	void addRow(long start_time, long end_time, const std::string& defid, 
		    const std::string& procid);
	void check(const char* title) const;
	void setSummaryID(const std::string& sid);
    private:
	DBTypes::varchar procid;
	DBTypes::varchar defid;
	DBTypes::varchar sumid;
	DBTypes::integer_4 start_time;
	DBTypes::integer_4 end_time;
	DBTypes::varchar   comment;	
    };


    /**  The S6SummaryList class keeps the summary status for a set of 
      *  segments.
      */
    class S6SummaryList : public xsil::MetaTable {
    public:
	typedef unsigned long gps_type;
    public:
	S6SummaryList(void);
	~S6SummaryList(void);
	void clear(void);
	void operator+=(const Segment& seg);
	void put(S6SummaryTable& table, const S6SegDefList& sdl) const;
    private:
	struct list_entry {
	    std::string name;
	    std::string ifo;
	    std::string procid;
	    long version;
	    gps_type start_time;
	    gps_type end_time;
	    bool operator<(const list_entry& le) const;
	    bool operator==(const list_entry& le) const;
	};
	typedef std::vector<list_entry> summary_list;
	typedef summary_list::iterator summary_iter;
	typedef summary_list::const_iterator const_summary_iter;
	void operator+=(const list_entry& seg);
    private:
	summary_list mList;
    };

    inline bool 
    S6SummaryList::list_entry::operator==(const list_entry& le) const {
	return name   == le.name && ifo == le.ifo && version == le.version && 
	       procid == le.procid;
    }

    inline bool 
    S6SummaryList::list_entry::operator<(const list_entry& le) const {
	if (name    != le.name   ) return name    < le.name;
	if (ifo     != le.ifo    ) return ifo     < le.ifo;
	if (version != le.version) return version < le.version;
	return procid < le.procid;
    }
}

#endif // S6SEGTABLE_HH
