/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "S6SegTable.hh"
#include "TrigBase.hh"
#include "TrigProc.hh"
#include "Segment.hh"
#include <fstream>
#include <stdlib.h>

using namespace xsil;
using namespace std;
using namespace trig;

//======================================  Null S6 Segment definer construtor
S6SegDef::S6SegDef(void) 
    : mVersion(0)
{}

//======================================  S6 Segment definer data construtor
S6SegDef::S6SegDef(const Segment& seg) {
    mProcID  = seg.getProcess();
    mIfos    = seg.getIfos();
    mName    = seg.getGroup();
    mVersion = seg.getVersion();
    mComment = seg.getComment();
}

//======================================  S6 Segment definer data construtor
S6SegDef::S6SegDef(const string& name, const string& ifo, int version)
    : mIfos(ifo), mName(name), mVersion(version)
{}

//======================================  Set segment definer ID
void 
S6SegDef::setDefinerID(const std::string& defid) {
    mDefID = defid;
}

//======================================  Empty Definer list
S6SegDefList::S6SegDefList(void) 
{}

//======================================  Destroy definer list
S6SegDefList::~S6SegDefList(void) {
}

//======================================  S5 Segment definer construtor
S6SegDefList::const_segdef_iter
S6SegDefList::add(const S6SegDef& def) {
    mList.push_back(def);
    return --mList.end();
}

//======================================  S5 Segment definer construtor
S6SegDefList::const_segdef_iter
S6SegDefList::find(const S6SegDef& sd) const {
    string name    = sd.refName();
    string ifo     = sd.refIfos();
    int    version = sd.getVersion();
    for (const_segdef_iter i=mList.begin(); i!=mList.end(); ++i) {
	if (i->refName() == name && 
	    i->refIfos() == ifo && 
	    i->getVersion() == version) return i;
    }
    return mList.end();
}

//======================================  Define the MySql segment table
S6SegTable::S6SegTable(void) 
  : MetaTable("segment")
{
    defineColumn("segment_id", &segid, true);
    defineColumn("start_time", &start_time);
    defineColumn("end_time",   &end_time);
    defineColumn("segment_def_id", &defid, true);
    defineColumn("process_id", &procid, true);
}

//======================================  Delete the trigger table
S6SegTable::~S6SegTable(void) {
    MetaTable::clear();
}

//======================================  Fill the trigger table.
void
S6SegTable::addRow(const trig::Segment& s, const S6SegDef& sd) {
    if (s.getActivity() > 0) {
	start_time = s.getStartTime().getS();
	end_time   = s.getEndTime().getS();
	segid      = s.getSegID();
	defid      = sd.refDefinerID();
	procid     = sd.refProcess();
	putRow();
    }
}

//======================================  Check that the pointers are OK
void
S6SegTable::check(const char* title) const {
    MetaTable::check(title, sizeof(S6SegTable));
}

//======================================  MySql segment definer table
S6SegDefTable::S6SegDefTable(void) 
  : MetaTable("segment_definer")
{
    defineColumn("process_id",     &procid, true);
    defineColumn("segment_def_id", &defid, true);
    defineColumn("ifos",           &ifos);
    defineColumn("name",           &name);
    defineColumn("version",        &version);
    defineColumn("comment",        &comment);
}

//======================================  Delete the trigger table
S6SegDefTable::~S6SegDefTable(void) {
    MetaTable::clear();
}

//======================================  Fill the trigger table.
void
S6SegDefTable::addRow(const S6SegDef& d) {
    procid  = d.refProcess();
    defid   = d.refDefinerID();
    ifos    = d.refIfos();
    name    = d.refName();
    version = d.getVersion();
    comment = d.refComment();
    putRow();
}

//======================================  Check that the pointers are OK
void
S6SegDefTable::check(const char* title) const {
    MetaTable::check(title, sizeof(S6SegDefTable));
}

//======================================  MySql segment definer table
S6SummaryTable::S6SummaryTable(void) 
  : MetaTable("segment_summary")
{
    defineColumn("segment_sum_id", &sumid,  true);
    defineColumn("start_time",     &start_time);
    defineColumn("end_time",       &end_time);
    defineColumn("comment",        &comment);
    defineColumn("segment_def_id", &defid,  true);
    defineColumn("process_id",     &procid, true);
}

//======================================  Delete the trigger table
S6SummaryTable::~S6SummaryTable(void) {
    MetaTable::clear();
}

//======================================  Fill the segment summary
void
S6SummaryTable::addRow(long s, long e, const string& d, const string& p) {
    start_time = s;
    end_time   = e;
    defid      = d;
    procid     = p;
    putRow();
}

//======================================  Set the summary entry id
//
//    Kludge note: the ID must be set before calling addRow!!!
void 
S6SummaryTable::setSummaryID(const std::string& sid) {
    sumid      = sid;
}

//======================================  Check that the pointers are OK
void
S6SummaryTable::check(const char* title) const {
    MetaTable::check(title, sizeof(S6SummaryTable));
}

//======================================  Construct a summary list.
S6SummaryList::S6SummaryList(void) {
}

//======================================  Destroy a summary list
S6SummaryList::~S6SummaryList(void) {
}

//======================================  Clear list
void 
S6SummaryList::clear(void) {
    mList.clear();
}

//======================================  Add a segment duration to the summary
void 
S6SummaryList::operator+=(const Segment& seg) {
    list_entry le;
    le.name       = seg.getGroup();
    le.ifo        = seg.getIfos();
    le.start_time = seg.getStartTime().getS();
    le.end_time   = seg.getEndTime().getS();
    le.version    = seg.getVersion();
    le.procid     = seg.getProcess();
    operator+=(le);
}

//======================================  Add a segment duration to the summary
void 
S6SummaryList::operator+=(const list_entry& le) {
    gps_type tEnd   = le.end_time;
    gps_type tStart = le.start_time;
    int           N = mList.size();
    for (int i=0; i<N; ++i) {
	if (mList[i] < le || (le == mList[i] && mList[i].end_time < tStart)) {
	    continue;
	} else if (le == mList[i] && tStart <= mList[i].end_time) {
	    if (tEnd > mList[i].end_time) mList[i].end_time = tEnd;
	    return;
	} else {
	    mList.insert(mList.begin()+i, le);
	    return;
	}
    }
    mList.insert(mList.begin()+N, le);
}

//======================================  put the summary to a table
void 
S6SummaryList::put(S6SummaryTable& table, const S6SegDefList& sl) const {
    int N = mList.size();
    for (int i=0; i<N; ++i) {
	S6SegDef sd(mList[i].name, mList[i].ifo, mList[i].version);
	S6SegDefList::const_segdef_iter p = sl.find(sd);
	string procid = p->refProcess();
	string defid  = p->refDefinerID();
	table.setSummaryID(table.citeTable("segment_sum_id", i));
	table.addRow(mList[i].start_time, mList[i].end_time, defid, procid);
    }
}
