/* -*- mode: c++; c-basic-offset: 4; -*- */
#include <stdio.h>
#include "NameServer.hh"
#include "lmsg/MsgHeader.hh"
#include "lmsg/ErrorList.hh"
#include <iostream>
#include <iomanip>
#include <cstdlib>

using namespace std;

static string stamp(time_t now);

//======================================  Server main program.
int 
main(int argc, const char* argv[]) {
    NameServer app(argc, argv);
    app.ProvideService();
    cout << stamp(0) << " Name server terminating" << endl;
}

//======================================  Time stamp generator
static string stamp(time_t now=0) {
    if (now == 0) now = time(0);
    string datel(::ctime(&now));
    return datel.substr(4, 15);
}

//======================================  Dictionary entry constructor
NameEntry::NameEntry(const NameData& x) 
  : NameData(x), mIdent(-1)
{
}

//======================================  Set dictionary entry identifier
void 
NameEntry::setIdent(int id) {
    mIdent = id;
}

//======================================  Application constructor
NameServer::NameServer(int argc, const char* argv[]) 
  : mDebug(0), mActive(false), mLastIdent(0)
{
    bool syntax(false);
    MsgAddr::ipport_t port(default_server_port);
    const char* hosttable = 0;
    for (int i=1 ; i<argc ; i++) {
      string argi(argv[i]);
        if (argi == "-debug") {
	    if (++i < argc) mDebug = strtol(argv[i],0,0);
	    else            mDebug = 1;
        } else if (argi == "-hosts") {
	    if (++i < argc) hosttable = argv[i];
	} else if (argi =="-port") {
	    if (++i < argc) port   = strtol(argv[i],0,0);
            else            syntax = true;
	} else {
	    cout << stamp(0) << " Unrecognized argument " << argi << endl;
	    syntax = true;
	}
    }

    mTerm.add(SIGINT);
    mTerm.add(SIGTERM);

    //--- Replaced SIGPOLL with SIGIO Keith Thorne June, 2011
    // on Linux, Solaris SIGPOLL same as SIGIO, Mac OS X only has SIGIO 
    mPoll.add(SIGIO);

    if (syntax) {
        cerr << "Error in NameServer command arguments" << endl;
	cerr << "Command Syntax: " << endl;
	cerr << argv[0] << " [-port <portid>] [-debug <debug-level>]" 
	     << " [-hosts <host-table>]" << endl;
	return;
    }

    //----------------------------------  Set up the message server interface
    setServerPort(MsgAddr(0, port));
    unsigned int rc = open(AppServer::o_none);
    if (rc) {
        cerr << stamp(0) << " Unable to open specified port." << endl;
	return;
    }
    AppServer::setDebug(getDebug());
    AppServer::addHandler(m_addName, new HandleNameAdd);
    AppServer::addHandler(m_lookup,  new HandleNameLookup);
    AppServer::addHandler(m_remName, new HandleNameRemove);
    AppServer::addHandler(m_index,   new HandleNameIndex);
    AppServer::addHandler(m_IDget,   new HandleIDget);

    //----------------------------------  Set up address translation table
    string HostFile;
    if (hosttable) {
        HostFile = hosttable;
    } else {
        if (!getenv("DMTHOME")) {
	    HostFile = getenv("HOME");
	} else {
	    HostFile = getenv("DMTHOME");
	    if (getenv("DMTVERSION")) {
	        HostFile += "/";
		HostFile += getenv("DMTVERSION");
	    }
	}
	HostFile += "/etc/HostTable";
    }
    cout << stamp() << " Opening host translation file: " << HostFile << endl;
    ifstream tabfile(HostFile.c_str());
    if (mDebug) mTable.setDebug(true);
    mTable.setTable(tabfile);
    mActive = true;
}

//======================================  Application destructor.
NameServer::~NameServer(void) {
    close();
}

//======================================  Service provision method
void
NameServer::ProvideService(void) {
    MsgHeader reqhdr;

    while (mActive && !mTerm) {
        try {
            unsigned int rc = handleMessage();
	    switch (rc) {
	    case OK:
	        break;
	    case TimeOut:
	    case Continue:
	        if (mDebug) cout << "Returned from handleMessage" << endl;
	        break;
	    default:
	        cerr << stamp() << " Error " << rc << " handling message" 
		     << endl;
	    }
	} catch (exception& e) {
	    cerr << stamp() << " Caught exception " << e.what() << endl;
	    mActive = false;
	}
    }
}

//=====================================  Find a dictionary entry
NameServer::dict_iter
NameServer::find(const char* name) {
    string sName = name;
    for (dict_iter r=mDictionary.begin() ; r!=mDictionary.end() ; r++) {
        if (sName == r->getName()) return r;
    }
    return mDictionary.end();
}

NameServer::dict_iter
NameServer::findByID(int id) {
    for (dict_iter r=mDictionary.begin() ; r!=mDictionary.end() ; r++) {
        if (r->getIdent() == id) return r;
    }
    return mDictionary.end();
}

//======================================  Add Handler implementation.
template<>
error_type 
HandleIDget::handleMsg(AppServer& app, const MsgHeader& hdr) {
    return reinterpret_cast<NameServer&>(app).idGetHandler(hdr, *this);
}

error_type 
NameServer::idGetHandler(const MsgHeader& hdr, const IDget& msg) {
    error_type rc=OK;
    dict_iter entry = findByID(msg.refData());
    if (getDebug()) {
        cout << stamp(0) << " ID: \"" << msg.refData()
	     << "\" lookup requested." << endl;
    }
    if (entry != mDictionary.end()) {
        MsgAddr requestor = hdr.getSource();
        MsgAddr target    = entry->getAddr();
	target.setSubProcess(entry->getIdent());

        MsgAddr resolve = mTable.getRoute(requestor, target);
	if (resolve == MsgAddr(0)) {
	    cout << stamp() << " Unable to address IP node " << target
		 << " from node " << requestor << endl;
	    rc = reply(hdr, NameStatus(*entry));
	} else if (resolve == entry->getAddr() ) {
	    rc = reply(hdr, NameStatus(*entry));
	} else {
	    NameStatus reply_stat(*entry);
	    reply_stat.refData().setAddr(resolve);
	    rc = reply(hdr, reply_stat);
	}
    } else {
        if (mDebug) cout << stamp() << " ID: " << msg.refData() 
			 << " is not in dictionary." << endl;
        rc = reply(hdr, HdrOnlyMsg<m_NO>());
    }
    return rc;
}

//======================================  Add Handler implementation.
template<>
error_type 
HandleNameAdd::handleMsg(AppServer& app, const MsgHeader& hdr) {
    return reinterpret_cast<NameServer&>(app).addHandler(hdr, *this);
}

error_type 
NameServer::addHandler(const MsgHeader& hdr, AddName& msg) {
    dict_iter entry = find(msg.refData().getName());
    msg.refData().fixAddr(hdr.getSource());
    if (entry != mDictionary.end()) {
        if (entry->getPType() == p_Server) {
	    cout << stamp() << " Error - Can't replace server: " 
		 << msg.refData().getName() << endl;
	    return reply(hdr, HdrOnlyMsg<m_NO>());
	} else if (entry->getPType() == p_Tunnel) {
	    mTable.remTunnel(entry->getAddr());
	    cout << stamp(0) << " Removed tunnel " << msg.refData().getName() 
		 << endl;
	    *entry = msg.refData();
        } else {
	    *entry = msg.refData();
	    cout << stamp() << " Replaced name " << msg.refData().getName() 
		 << " with " << msg.refData().getAddr() << endl;
	}
    } else {
        mDictionary.push_back(msg.refData());
	cout << stamp() << " Added name \"" << msg.refData().getName() 
	     << "\" with address " << msg.refData().getAddr() << endl;
	entry = mDictionary.end();
	entry--;
    }
    entry->setIdent(++mLastIdent);
    if (entry->getPType() == p_Tunnel) {
        MsgAddr::ipaddr_t netip(0), netmask(0);
	sscanf(entry->getName(), "_Tunnel_%08x%08x", &netip, &netmask);
	mTable.addTunnel(netip, netmask, entry->getAddr());
	if (mDebug) cout << "Tunnel " << entry->getName() 
			 << " registered with NetIP:NetMask=" << hex
			 << netip << ":" << netmask << dec << endl;
    }
    return reply(hdr, HdrOnlyMsg<m_OK>());
}

//======================================  Lookup Handler implementation
template<>
error_type 
HandleNameLookup::handleMsg(AppServer& app, const MsgHeader& hdr) {
    return reinterpret_cast<NameServer&>(app).findHandler(hdr, *this);
}

error_type 
NameServer::findHandler(const MsgHeader& hdr, const FindName& msg) {
    error_type rc;
    dict_iter entry = find(msg.refData().getName());
    if (getDebug()) {
        cout << stamp(0) << " Name: \"" << msg.refData().getName() 
	     << "\" lookup requested." << endl;
    }
    if (entry != mDictionary.end()) {
        MsgAddr requestor = hdr.getSource();

        MsgAddr target    = entry->getAddr();
	target.setSubProcess(entry->getIdent());

        MsgAddr resolve = mTable.getRoute(requestor, target);
	if (resolve == MsgAddr(0) ) {
	    cout << stamp() << " Unable to address IP node " << target
		 << " from node " << requestor << endl;
	    rc = reply(hdr, NameStatus(*entry));
	} else if (resolve == entry->getAddr()) {
	    rc = reply(hdr, NameStatus(*entry));
	} else {
	    NameStatus reply_stat(*entry);
	    reply_stat.refData().setAddr(resolve);
	    rc = reply(hdr, reply_stat);
	}
    } else {
        if (mDebug) cout << stamp() << " Name: " << msg.refData().getName() 
			 << " is not in dictionary." << endl;
        rc = reply(hdr, HdrOnlyMsg<m_NO>());
    }
    return rc;
}

//======================================  Remove handler implementation.
template<>
error_type 
HandleNameRemove::handleMsg(AppServer& app, const MsgHeader& hdr) {
    return reinterpret_cast<NameServer&>(app).remHandler(hdr, *this);
}

error_type 
NameServer::remHandler(const MsgHeader& hdr, const RemoveName& msg) {
    dict_iter entry = find(msg.refData().getName());
    if (entry != mDictionary.end()) {
        if (entry->getPType() == p_Tunnel) {
	    mTable.remTunnel(entry->getAddr());
	    mDictionary.erase(entry);
	    cout << stamp(0) << " Removed tunnel " << msg.refData().getName() 
		 << endl;
        } else {
	    mDictionary.erase(entry);
	    cout << stamp(0) << " Removed name " << msg.refData().getName() 
		 << endl;
	}
    }
    return reply(hdr, HdrOnlyMsg<m_OK>());
}

//======================================  Index handler implementation.
template<>
error_type 
HandleNameIndex::handleMsg(AppServer& app, const MsgHeader& hdr) {
    return reinterpret_cast<NameServer&>(app).indexHandler(hdr, *this);
}

error_type 
NameServer::indexHandler(const MsgHeader& hdr, const NameIndex& msg) {
    string Index;
    NameProcs         type_select = msg.refData().getPType();
    MsgAddr::ipaddr_t node_select = msg.refData().getAddr().getIPAddr();
    if (getDebug()) {
        cout << stamp(0) << " Index requested for: " << node_select 
	     << " process type: " << type_select << endl;
    }
    for (dict_iter i=mDictionary.begin() ; i != mDictionary.end() ; i++) {
        if ((type_select == p_Any || type_select == i->getPType()) &&
	    (node_select == 0 || node_select == i->getAddr().getIPAddr())) {
	    if (Index.empty()) {
	        Index = i->getName();
	    } else {
	        Index += ";";
		Index += i->getName();
	    }
	}
    }
    return reply(hdr, NameStatus(NameData(Index, msg.refData().getAddr(), 
					  msg.refData().getPType())));
}

