#ifndef LMSG_NAMECLIENT_HH
#define LMSG_NAMECLIENT_HH

#ifndef __CINT__
#include "AppClient.hh"
#include "NameData.hh"
#endif

namespace lmsg {

  /**  The Name service client API allows ligo message users to look-up 
    *  the address of dmt server processes, including monitors, Trigger
    *  managers, etc.
    *  @memo Name service client API.
    *  @author John Zweizig
    *  @version 1.0; Modfied January 28, 2000
    */
  class NameClient : private AppClient {
  public:
    /**  Release the port and the client sattus information.
      *  @memo Destructor.
      */
    virtual ~NameClient(void);

    /**  Create a client object. Get the default name server (domain) 
      *  address from the DMTNAMESERVER environment variable.
      *  @memo Construct a client.
      */
    NameClient(void);

    /**  Get the current name server (domain) IP host name.
      *  @memo get the Domain name.
      */
    const char* getDomain(void) const;

    /**  Set the client interface debug level.
      *  @set the debug level.
      */
    void setDebug(index_type level);

    /**  Provide a name server port adddress in the form "<ip-node>:<port>",
      *  where <ip-node> is an IP node name, and <port> is a vlid port 
      *  number.
      *  @memo Set the name server port address.
      *  @return Error status (as described in ErrorList.hh)
      *  @param domain Node and port number specified as descibed above.
      */
    error_type setDomain(const char* domain);

    /**  Request the server to add a name to its dictionary.
      *  @memo Add a name.
      *  @return Error status (as described in ErrorList.hh)
      *  @param name  Name to be added to the dictionary
      *  @param addr  Address corresponding to the name.
      *  @param ptype Process type of the added name.
      */
    error_type addName(const char* name, const MsgAddr& addr, 
		    NameProcs Type=p_Any);

    /**  Request the server to add a name to its dictionary. All information 
      *  pertining to the process is encoded in a NameMsg object.
      *  @memo Add a name.
      *  @return Error status (as described in ErrorList.hh)
      *  @param name NameMsg with information to be added to the dictionary
      */
    error_type addName(const NameData& name);

    /**  Request a list of server names from the dictionary. The names are
      *  optionally selected by process type and ip node.
      *  @memo Get a list of names.
      *  @return Error status (as described in ErrorList.hh)
      *  @param name  List of selected names from the dictionary
      *  @param addr  IP address for names.
      *  @param ptype Process types.
      */
    error_type getIndex(std::string& list, MsgAddr::ipaddr_t addr=0, 
		     NameProcs Type=p_Any);

    /**  Request the server to remove a name from its dictionary.
      *  @memo Remove a name.
      *  @return Error status (as described in ErrorList.hh)
      *  @param name  Name to be removed from the dictionary
      */
    error_type remName(const char* name);

    /**  Request the server to remove a name from its dictionary.
      *  @memo Remove a name.
      *  @return Error status (as described in ErrorList.hh)
      *  @param name NameMsg with name to be removed from the dictionary
      */
    error_type remName(const NameData& name);

    /**  Request the server to look-up a name in its dictionary.
      *  @memo find a name.
      *  @return Error status (as described in ErrorList.hh)
      *  @param name Name to be found in the dictionary
      *  @param addr MsgAddr to receive the address of the named process.
      */
    error_type lookup(const char* name, MsgAddr& addr);

    /**  Request the server to look-up a name in its dictionary.
      *  @memo find a name.
      *  @return Error status (as described in ErrorList.hh)
      *  @param name Name to be found in the dictionary
      *  @param addr MsgAddr to receive the requested information.
      */
     error_type lookup(const char* name, NameData& msg);

    /**  Request the server to look-up an entry in its dictionary.
      *  @memo find a name.
      *  @return Error status (as described in ErrorList.hh)
      *  @param ident Server ID to be found in the dictionary
      *  @param addr  MsgAddr to receive the address of the named process.
      */
    error_type lookupID(int ident, MsgAddr& addr);

    /**  Request the server to look-up an entry in its dictionary.
      *  @memo find a name.
      *  @return Error status (as described in ErrorList.hh)
      *  @param ident Server ID to be found in the dictionary
      *  @param addr  MsgAddr to receive the requested information.
      */
     error_type lookupID(int ident, NameData& msg);

  private:
    /**  Look up the address of the the current name server.
      *  @memo Find the current name server address.
      *  @return Error status (as described in ErrorList.hh)
      */
    error_type findDomainServer(void);

    /**  Issue a request to the server and wait for a reply.
      *  @memo Issue a name server request.
      *  @param type  Type of request to be issued.
      *  @param req   Request message contents.
      *  @param reply pointer to NameMsg to receive the server reply.
      */
    error_type doRequest(const Message& req, NameData* reply=0);

  private:
    void    localInit(void);
    error_type localAdd(const char* name, const MsgAddr& addr);
    error_type localRemove(const char* name);
    error_type localResolve(const char* name, NameData& addr) const;

  private:
    std::string mDomainName;
    MsgAddr     mDomainServer;
    bool        mLocal;
    std::string mLocalDir;
  };
} // namespace lmsg

#ifndef __CINT__
inline const char* 
lmsg::NameClient::getDomain(void) const {
    return mDomainName.c_str();
}

inline void 
lmsg::NameClient::setDebug(lmsg::index_type level) {
    lmsg::AppClient::setDebug(level);
}
#endif

#endif // LMSG_NAMECLIENT_HH
