#ifndef LMSG_APPCLIENT_HH
#define LMSG_APPCLIENT_HH

#include "lmsg/MsgTypes.hh"

namespace lmsg {
  class Buffer;
  class BufferPool;
  class MsgAddr;
  class Message;
  class MsgHeader;
  class SocketPool;
#ifdef TCP_TRANSPORT
  class TransportTCP;
#else
  class TransportMsg;
#endif

  /**  The client application class provides a generic client API. 
    *  AppClient provides the primitive client request operation 
    *  and server name resolution functions. If a server does not
    *  reply to a request within a specified amount of time, the 
    *  request is retried. A transaction ID is sent with the request 
    *  and returned with the reply. The client API verifies that the 
    *  trasaction ID of  the reply is correct. The port used by the 
    *  client API is allocated dynamically from a specified socket 
    *  pool or from the defaul pool. If a request fails. the socket 
    *  is deleted and a new socket is allocated to retry the request.
    *  In non-linux systems, the socket is logically connected to the 
    *  server socket before the request is made. This allows the client
    *  to receive notification if the server socket is not available.
    *  @memo Application client API.
    *  @author John Zweizig
    *  @version 1.2; Modified May 12, 2000
    */
  class AppClient {
  public:
    /**  Client destructor.
      *  @memo Destructor.
      */
    virtual ~AppClient(void);

    /**  Construct a disconnected client interface.
      *  @memo Default constructor.
      */
    AppClient(void);

    /**  Send a request message to the specified server and wait for a 
      *  reply. If no reply is received in the specified timeout period,
      *  the request is resent. The reply header and message are stored 
      *  in the buffers provided by the user.
      *  @memo Send a request message to a server and wait for the reply.
      *  @return Error code from lmsg::ErrorList.
      *  @param to     Address of server to handle request.
      *  @param req    Request message.
      *  @param hdr    Reply header.
      *  @param data   Pointer to buffer to receive reply text.
      *  @param length Length of reply buffer.
      */
    error_type request(const MsgAddr& to, const Message& req, 
		    MsgHeader& hdr, char* data, index_type length);

    /**  Send a request message to the specified server and wait for a 
      *  reply. If no reply is received in the specified timeout period,
      *  the request is resent. The reply header and message are stored 
      *  in the buffers provided by the user.
      *  @memo Send a request message to a server and wait for the reply.
      *  @return Error code from lmsg::ErrorList.
      *  @param ReqHdr  Request header (with destination, type, length).
      *  @param ReqData Request message text.
      *  @param hdr     Reply header.
      *  @param data    Pointer to buffer to receive reply text.
      *  @param length  Length of reply buffer.
      */
    error_type request(const MsgHeader& ReqHdr, const void* ReqData, 
		    MsgHeader& hdr, char* data, index_type length);

    /**  Send a request message to the specified server and wait for a 
      *  reply. If no reply is received in the specified timeout period,
      *  the request is resent. The reply header and message are stored 
      *  in the buffers provided by the user.
      *  @memo Send a request message to a server and wait for the reply.
      *  @return Error code from lmsg::ErrorList.
      *  @param to    Request header (including det, type, length).
      *  @param Req   Request message.
      *  @param Reply Reply message.
      */
    error_type request(const MsgAddr& to, const Message& Req, Message& Reply);

    /**  Send a request message to the specified server and wait for a 
      *  reply. If no reply is received in the specified timeout period,
      *  the request is resent. The reply header and message are stored 
      *  in the buffers provided by the user.
      *  @memo Send a request message to a server and wait for the reply.
      *  @return Error code from lmsg::ErrorList.
      *  @param to     Request header (including det, type, length).
      *  @param Req    Request message.
      *  @param RepHdr Reply message header.
      *  @param Reply  Reply message.
      */
    error_type request(const MsgAddr& to, const Message& Req, 
		    MsgHeader& RepHdr, Message& Reply);

    /**  Get the Debug flag contents.
      *  @memo Get the debug flag.
      *  @return The debug flag contents.
      */
    error_type getDebug(void) const;

    /**  Get the maximum number of retries.
      *  @memo Get number of retries.
      *  @return The maximum number of retries.
      */
    error_type getRetry(void) const;

    /**  Get the maximum response time.
      *  @memo Maximum response time.
      *  @return Maximum reply time.
      */
    wtime_type getTimeOut(void) const;

    /**  Test whether the client interface is open.
      *  @memo Test whether Client is open.
      *  @return true if client interface is ready for work.
      */
    bool    isOpen(void) const;

    /**  Specify the buffer pool to be used by this client.
      *  @memo Select a buffer pool.
      *  @param pool Pointer to the buffer pool to be used.
      */
    void setBufferPool(BufferPool* pool);

    /**  Set the Debug flag contents. If the debug flag is non-zero
      *  status messages will be printed each time a request is made.
      *  @memo Set the debug flag.
      *  @param level New debug flag contents.
      */
    void setDebug(index_type level);

    /**  Specify the maximum number of times a request is to be retried 
      *  by this client if no valid reply is received.
      *  @memo Set the maximum number of retries..
      *  @param n Maximum number of retries.
      */
    void setRetry(index_type n);

    /**  Specify the maximum wait time before retrying a request.
      *  @memo Set the maximum response time.
      *  @param time Wait time.
      */
    void setTimeOut(wtime_type time);

    /**  Specify the socket pool to be used by this client.
      *  @memo Select a socket pool.
      *  @param pool Pointer to the socket pool to be used by this client.
      */
    void setSocketPool(SocketPool* pool);

  private:
    /**  Get a client socket and set the appropriate debug level.
     */
    error_type openClient(const MsgAddr&  to);

  private:
    SocketPool*   mSockPool;
    BufferPool*   mBuffPool;
#ifdef TCP_TRANSPORT
    TransportTCP* mClient;
#else
    TransportMsg* mClient;
#endif
    wtime_type       mTimeout;
    index_type       mRetryMax;
    index_type       mTransID;
    index_type       mDebug;
  };
} // namespace lmsg

inline lmsg::error_type
lmsg::AppClient::getDebug(void) const {
    return mDebug;
}

inline lmsg::error_type
lmsg::AppClient::getRetry(void) const {
    return mRetryMax;
}

inline lmsg::wtime_type
lmsg::AppClient::getTimeOut(void) const {
    return mTimeout;
}

#endif // LMSG_APPCLIENT_HH
