/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "AppClient.hh"
#include "lmsg/Buffer.hh"
#include "lmsg/BufferPool.hh"
#include "lmsg/ErrorList.hh"
#include "lmsg/MsgAddr.hh"
#include "lmsg/Message.hh"
#include "lmsg/MsgHeader.hh"
#include "lmsg/Socket.hh"
#include "lmsg/SocketPool.hh"
#ifdef TCP_TRANSPORT
#include "lmsg/TransportTCP.hh"
#else
#include "lmsg/TransportMsg.hh"
#endif
#include "lmsg/TransInput.hh"
#include <iostream>
#include <stdio.h>
#include <cstdlib>

using namespace lmsg;
using namespace std;

AppClient::~AppClient(void) {
    if (mClient) delete mClient;
}

AppClient::AppClient(void) 
  : mSockPool(&defaultLMsgSocketPool), mBuffPool(&defaultLMsgBufferPool),
    mTimeout(1.0), mRetryMax(1), mTransID(0), mDebug(0)
{
    const char* deblvl=getenv("APPCLIENT_DEBUG");
    if (deblvl) {
	if (*deblvl) mDebug = strtol(deblvl, 0, 0);
	else         mDebug = 1;
    }
#ifdef TCP_TRANSPORT
    mClient = new TransportTCP();
#else
    mClient = new TransportMsg();
#endif
    mClient->setDebug(mDebug);
}

error_type 
AppClient::openClient(const MsgAddr& to) {
    //----------------------------------  Get a client port.
    if (!mClient) {
        return NotOpen;
    } else if (!isOpen()) {
#ifdef TCP_TRANSPORT
        mClient->open(TransportTCP::o_client);
#else
        mClient->open(TransportMsg::o_client);
#endif
	if (!isOpen()) return NotOpen;
	mClient->setDebug(mDebug);
    }

    //----------------------------------  Connect if not connected already
#if defined(TCP_TRANSPORT) || !defined(__linux)
    error_type rc = mClient->connect(to);
    if (rc) mClient->close();
    return rc;
#else
    return OK;
#endif
}

error_type 
AppClient::request(const MsgAddr& to, const Message& req, MsgHeader& rhdr, 
		   char* rdata, index_type maxlen) 
{
    MsgHeader h(req.getType(), req.size(), to);
    error_type rc=OK;
    h.setTransID(++mTransID);

    //----------------------------------  repeat as necessary.
    for (index_type trial=0 ; trial<=mRetryMax ; trial++) {

        rc = openClient(to);
	if (rc) return rc;

        //------------------------------  Send the request.
	rc = mClient->send(h, req);
	if (rc) {
	    mClient->Purge();
	    continue;
	}
	if (mDebug) cerr << "Sent request." << endl;

	//------------------------------  Get a reply, free the client port
	do {
	    rc = mClient->receive(rhdr, rdata, maxlen, mTimeout);
	    if (rc && mDebug) {
	        if (rc == SystemError) perror("AppClient: receive error");
		else cerr<<"AppClient: receive error: "<<rc<<endl;
	    }
	} while (rc == Continue);

	if (rc) {
	    mClient->Purge();
	    if (rc == TimeOut) continue;
	    else               break;
	}

	//------------------------------  Check the transaction ID
	if (rhdr.getTransID() != mTransID) {
	    mClient->Purge();
	    rc = Invalid;
	    if (mDebug) {
	        cerr << "Received invalid reply (good tranID= " 
		     << mTransID << ")" << endl;
	    }
	} else {
	    if (mDebug) cerr << "Received valid reply. " << endl;
	    break;
	}
    }
    if (isOpen()) mClient->close();
    return rc;
}

error_type 
AppClient::request(const MsgAddr& to, const Message& req, Message& reply) 
{
    MsgHeader rhdr;
    return request(to, req, rhdr, reply);
}

error_type 
AppClient::request(const MsgAddr& to, const Message& req, 
		   MsgHeader& rhdr, Message& reply) 
{
    MsgHeader h(req.getType(), req.size(), to);
    index_type rc=OK;
    h.setTransID(++mTransID);

    //----------------------------------  repeat as necessary.
    for (index_type trial=0 ; trial<=mRetryMax ; trial++) {

        //------------------------------  Get a client port.
        rc = openClient(to);
	if (rc) return rc;

        //------------------------------  Send the request.
	rc = mClient->send(h, req);
	if (rc) {
	    mClient->Purge();
	    continue;
	}
	if (mDebug) cerr << "Sent request." << endl;

	//------------------------------  Get a reply, free the client port
	do {
	    rc = mClient->receive(rhdr, reply, mTimeout);
	    if (rc && mDebug) {
	        if (rc == SystemError) perror("AppClient: receive error");
		else cerr<<"AppClient: receive error: "<<rc<<endl;
	    }
	} while (rc == Continue);

	if (rc) {
	    mClient->Purge();
	    if (rc == TimeOut) continue;
	    else               break;
	}

	//------------------------------  Check the transaction ID
	if (rhdr.getTransID() != mTransID) {
	    mClient->Purge();
	    rc = Invalid;
	    if (mDebug) {
	        cerr << "Received invalid reply (good tranID= " 
			  << mTransID << ")" << endl;
	    }
	} else {
	    if (mDebug > 1) cerr << "Message received." << endl;
	    break;
	}
    }
    if (isOpen()) mClient->close();
    return rc;
}

error_type 
AppClient::request(const MsgHeader& hdr, const void* data, MsgHeader& rhdr, 
		   char* rdata, index_type maxlen) 
{
    index_type rc=OK;
    MsgHeader h(hdr);
    h.setTransID(++mTransID);

    //----------------------------------  repeat as necessary.
    for (index_type trial=0 ; trial<=mRetryMax ; trial++) {

        //------------------------------  Get a client port.
        rc = openClient(h.getDest());
	if (rc) return rc;

        //------------------------------  Send the request.
	rc = mClient->send(h, (const char*)data);
	if (rc) {
	    mClient->Purge();
	    continue;
	}
	if (mDebug) {
	    cerr << "Sent request." << endl;
	}

	//------------------------------  Get a reply, free the client port
	do {
	    rc = mClient->receive(rhdr, rdata, maxlen, mTimeout);
	    if (rc && mDebug) {
	        if (rc == SystemError) perror("AppClient: receive error");
		else cerr << "AppClient: receive error: " << rc << endl;
	    }
	} while (rc == Continue);

	if (rc) {
	    mClient->Purge();
	    if (rc == TimeOut) continue;
	    else               break;
	}

	//------------------------------  Check the transaction ID
	if (rhdr.getTransID() != mTransID) {
	    mClient->Purge();
	    rc = Invalid;
	    if (mDebug) cerr << "Received invalid reply (good tranID= " 
			     << mTransID << ")" << endl;
	} else {
	    if (mDebug) cerr << "Received valid reply." << endl;
	    break;
	}
    }
    if (isOpen()) mClient->close();
    return rc;
}

void 
AppClient::setBufferPool(BufferPool* pool) {
    if (pool) mBuffPool = pool;
    if (mClient) mClient->setBufferPool(mBuffPool);
}

void 
AppClient::setSocketPool(SocketPool* pool) {
    if (pool) mSockPool = pool;
#ifndef TCP_TRANSPORT
    if (mClient) mClient->setSocketPool(mSockPool);
#endif
}

void 
AppClient::setRetry(index_type count) {
    mRetryMax = count;
}

void 
AppClient::setTimeOut(double time) {
    mTimeout = time;
}

bool
AppClient::isOpen(void) const {
    return mClient && mClient->isOpen();
}

void 
AppClient::setDebug(index_type level) {
    mDebug = level;
    if (mClient) mClient->setDebug(level);
}
