/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef ALARMCLIENT_HH
#define ALARMCLIENT_HH

#ifndef DMTOFFLINE
#include "AppClient.hh"
#else
#include "lmsg/MsgTypes.hh"
#endif
#include "AlarmData.hh"
#include "lmsg/MsgAddr.hh"
#include <vector>
#include <stdexcept>

/**  The alarm client API provides an interface to the alarm manager. Refer
  *  to xxxx for a full description of the Alarm Manager functionality.
  *  The AlarmClient API allows the client process to:
  *  <ul>
  *    <li> <b> Define Alarm Types:</b>
  *      The alarm manager keeps a list of currently defined alarm types.
  *      The alarm must be defined by a process before it can be set. The
  *      definition specifies defaul values for all alarm parameters.
  *    </li>
  *    <li> <b> Set Alarms:</b>
  *      Alarms are set or asserted when an abnormal condidstion is detected.
  *      The alarm remains in effect until it times out or is cancelled 
  *      explicitly.
  *    </li>
  *    <li> <b> Prolong Alarms:</b>
  *      Alarms are created with an expected timeout period. If the error
  *      condition is not resolved in the expected time, the monitor that
  *      detected the alarm may prolong the alarm duration.
  *    </li>
  *    <li> <b> Cancel Alarms:</b>
  *      Alarms may be cancelled explicitly if the condition is resolved 
  *      before the timeout period.
  *    </li>
  *    <li> <b> Interrogate alarm status:</b>
  *      The status of an individual alarm and the default parameters of an
  *      alarm type may be interrogated.
  *    </li>
  *  </ul>
  *
  *  @memo Alarm Client API
  *  @author J. Zweizig
  *  @version $Revision: 6273 $; Last Modified $Date: 2010-08-01 00:47:51 -0700 (Sun, 01 Aug 2010) $
  */
class AlarmClient 
#ifndef DMTOFFLINE
  : private lmsg::AppClient 
#endif
{
public:
  /**  Construct an alarm client interface. The optional name argument 
    *  specifies the monitor name to be used in all Alarm requests. The
    *  monitor name must be specified if the interface is to be used to 
    *  define or set alarms. 
    *  @memo Construct an alarm client interface.
    *  @param name Monitor name to be used in defining and setting alarms.
    */
  AlarmClient(const char* name=0);

  /** Destroy the alarm client. 
    * @memo Alarm client destructor
    */
  virtual ~AlarmClient(void);

  /**  Cancel the alarm specified by the alarm handle. The Alarm manager
    *  is instructed to cancel the alarm. No action is taken if the alarm 
    *  has already expired.
    *  @memo Cancel an alarm.
    *  @return Code indicating success or failure of the cancel request.
    *  @param handle Identifier of alarm to be canceled. 
    */
  lmsg::error_type cancelAlarm(const AlarmHandle& handle);

  /**  Define an alarm type and specify the default parameter values.
    *  The Alarm manager is instructed to add the alarm type to its 
    *  database. 
    *  @memo define an alarm type
    *  @return Code indicating success or failure of the cancel request.
    *  @param ID      Alarm name.
    *  @param timeout Default time before alarm expiry.
    *  @param severe  Default alarm severity.
    *  @param desc    Alarm description string file or URL.
    *  @param params  Parameter substitution string.
    */
  lmsg::error_type defineAlarm(const std::string& ID, Interval timeout=0.0, 
			    int severe=0, const std::string& desc="",
			    const std::string& params="");


  /**  Define an alarm type and specify the default parameter values.
    *  The Alarm manager is instructed to add the alarm type to its 
    *  database. 
    *  @memo define an alarm type
    *  @return Code indicating success or failure of the define request.
    *  @param x  AlarmData object containing the alarm default values.
    */
  lmsg::error_type defineAlarm(const AlarmData& x);

  /**  Get a list of active alarms known by the alarm manager.
    *  @memo Get active alarm handles.
    *  @return Code indicating success or failure of the request.
    *  @param x     AlarmID object specifying a pattern for alarms.
    *  @param reply Vector of alarm handles.
    */
  lmsg::error_type getActive(const AlarmID& x, std::vector<AlarmHandle>& reply);

  /**  Get a list of alarm types known by the alarm manager.
    *  @memo Get defined alarm type
    *  @return Code indicating success or failure of the request.
    *  @param x     AlarmID object specifying a pattern for alarms.
    *  @param reply string of alarm IDs.
    */
  lmsg::error_type getDefined(const AlarmID& x, std::string& reply);

  /**  Get the template alarm data of an alarm type known by the alarm manager.
    *  @memo Get defined alarm template.
    *  @return Code indicating success or failure of the request.
    *  @param aid   AlarmID object specifying an alarm type.
    *  @param reply Template infomation kept by the manager.
    */
  lmsg::error_type getTemplate(const AlarmID& aid, AlarmData& reply);

  /**  Prolong the alarm specified by the alarm handle. The Alarm manager
    *  is instructed to increase the expiry time of the existing alarm by 
    *  the specified time duration. No action is taken if the alarm has 
    *  already expired, although an \c Invalid code is returned.
    *  @memo Prolong an alarm.
    *  @return Code indicating success or failure of the prolongation request.
    *  @param handle Identifier of alarm to be prolonged. 
    *  @param dt     Duration increment. 
    */
  lmsg::error_type prolongAlarm(const AlarmHandle& handle, Interval dt);

  /**  The alarm manager is requested to remove the specified alarm type
    *  definition from its table. 
    *  @memo Remove an alarm type.
    *  @return Code indicating success or failure of the removal request.
    *  @param aid Identifier of alarm to be removed.
    */
  lmsg::error_type remove(const AlarmID& aid);

  /**  The alarm manager is instructed to activate an alarm. The alarm
    *  remains active until it expires or is explicitly cancelled. Any
    *  parameters not specified in the alarm must have default values
    *  specified.
    *  @memo Issue an alarm.
    *  @return Code indicating success or failure of the set request.
    *  @param alarm AlarmData specifying the alarm to be issued.
    *  @param reply Handle to the alarm is successfully issued.
    */
  lmsg::error_type setAlarm(const AlarmData& alarm, AlarmHandle& reply);

  /**  Set the level of debugging output to print.
    *  @memo Set the debug level.
    *  @param level Debug level.
    */
  void setDebug(int level);

  /**  The alarm manager is instructed to enable or disable an alarm type. 
    *  Any attempts to set an alarm that has been disabled will be ignored.
    *  The alarm type may be reenabled with the same 
    *  @memo Disable an alarm type.
    *  @return Code indicating success or failure.
    *  @param aid  AlarmID specifying the alarm type to be disabled.
    *  @param flag If true the alarm is diabled, if false it is enabled.
    */
  lmsg::error_type setDisable(const AlarmID& aid, bool flag=true);

  /**  Set the monitor name to be used any time an alarm is defined or set.
    *  @memo set the monitor name.
    *  @param name Monitor name.
    */
  void setName(const std::string& name);

  /**  Get the current status of the alarm specified by the handle.
    *  @memo Get alarm status.
    *  @return Code indicating success or failure of the request.
    *  @param handle Handle of alarm to be interrogated.
    *  @param reply  Status of alaram (returned).
    */
  lmsg::error_type testAlarm(const AlarmHandle& handle, AlarmData& reply);


private:
  /**  Mark connection as no longer active.
    *  @memo end connection to the Alarm Manager.
    */
  void close(void);

  /**  Establish a routing to the Alarm Manager;
    *  @memo Comnnec to the Alarm Manager.
    *  @return Code indicating success or failure of the request
    */
  lmsg::error_type open(void);

#ifndef DMTOFFLINE
  /**  Send the specified message and set the return code according to the 
    *  reply. #sendMessage# expects a reply of an empty message of type 
    *  #almAck# or #almNAck# depending on the success or failure of the 
    *  the request.
    *  @memo Send a specified request to the Alarm Manager.
    *  @return Code indicating success or failure of the request.
    *  @param msg Request message.
    */
  lmsg::error_type sendMessage(const lmsg::Message& msg);
#endif

private:
  std::string   mName;
#ifndef DMTOFFLINE
  lmsg::MsgAddr mServer;
#endif
  int           mDebug;
};
#endif  // !defined(ALARMCLIENT_HH)
