#include "AlarmClient.hh"
#include "lmsg/ErrorList.hh"
#ifndef DMTOFFLINE
#include "AlarmMsg.hh"
#include "NameClient.hh"
#endif
#include <iostream>

using namespace std;
using namespace lmsg;

//======================================  Default constructor
AlarmClient::AlarmClient(const char* name) 
  : mDebug(0)
{
#ifndef DMTOFFLINE
    mServer = lmsg::MsgAddr(0, 0, 0);
#endif
    if (name) mName = name;
}

//======================================  Destructor
AlarmClient::~AlarmClient(void) {
}

//======================================  Cancel an existing alarm
error_type
AlarmClient::cancelAlarm(const AlarmHandle& x) {
#ifndef DMTOFFLINE
    return sendMessage(ALM_Cancel(x));
#else
    return NotOpen;
#endif
}

//======================================  Define a new alarm type
error_type 
AlarmClient::defineAlarm(const string& ID, Interval timeout, int severe, 
			 const string& desc, const string& params) {
    if (mName.empty()) return Invalid;
    return defineAlarm(AlarmData(mName, ID, timeout, severe, desc, params));
}

error_type 
AlarmClient::defineAlarm(const AlarmData& x) {
#ifndef DMTOFFLINE
    return sendMessage(ALM_Define(x));
#else
    return NotOpen;
#endif
}

//======================================  Get a list of active alarms
error_type 
AlarmClient::getActive(const AlarmID& req, vector<AlarmHandle>& vec) {
    error_type rc = NotOpen;
#ifndef DMTOFFLINE
    try {
        rc = open();
	if (rc) return rc;
        ALM_ActiveReply reply;
        rc = request(mServer, ALM_GetActive(req), reply);
	if (!rc) vec = reply.refData();
    } catch (exception& e) {
        cerr << mName << ": Caught exception in AlarmClient::getActive: "
	     << e.what() << endl;
	rc = Invalid;
    }
    if (rc == BadAddress) {
	cerr << mName << ": Alarm manager moved!" << endl;
	mServer = 0;
    }
#endif
    return rc;
}

//======================================  Get a list of defined alarms
error_type 
AlarmClient::getDefined(const AlarmID& req, string& repstr) {
    error_type rc = NotOpen;
#ifndef DMTOFFLINE
    try {
        rc = open();
	if (rc) return rc;
        ALM_DefinedReply reply;
        rc = request(mServer, ALM_GetDefined(req), reply);
	if (!rc) repstr = reply.refData();
    } catch (exception& e) {
        cerr << mName << ": Caught exception in AlarmClient::getDefined: "
	     << e.what() << endl;
	rc = Invalid;
    }
    if (rc == BadAddress) {
	cerr << mName << ": Alarm manager moved!" << endl;
	mServer = 0;
    }
#endif
    return rc;
}

//======================================  Get alaram data template for an ID
error_type 
AlarmClient::getTemplate(const AlarmID& req, AlarmData& data) {
    error_type rc = NotOpen;
#ifndef DMTOFFLINE
    try {
        rc = open();
	if (rc) return rc;
        ALM_Status reply;
        rc = request(mServer, ALM_GetTemplate(req), reply);
	if (!rc) data = reply.refData();
    } catch (exception& e) {
        cerr << mName << ": Caught exception in AlarmClient::getTemplate: "
	     << e.what() << endl;
	rc = Invalid;
    }
    if (rc == BadAddress) {
	cerr << mName << ": Alarm manager moved!" << endl;
	mServer = 0;
    }
#endif
    return rc;
}

//======================================  Prolong an existing alarm
error_type 
AlarmClient::prolongAlarm(const AlarmHandle& x, Interval dt) {
#ifndef DMTOFFLINE
    return sendMessage(ALM_Prolong(pair<AlarmHandle,double>(x, dt)));
#else
    return NotOpen;
#endif
}

//======================================  Prolong an existing alarm
error_type 
AlarmClient::remove(const AlarmID& aid) {
#ifndef DMTOFFLINE
    return sendMessage(ALM_Remove(aid));
#else
    return NotOpen;
#endif
}

//======================================  Set an alarm
error_type 
AlarmClient::setAlarm(const AlarmData& x, AlarmHandle& handle) {
    error_type rc = NotOpen;
#ifndef DMTOFFLINE
    try {
        rc = open();
	if (rc) {
	    cerr << "Error opening channel to Alarm Manager" << endl;
	    return rc;
	}
        ALM_Reference reply;
        rc = request(mServer, ALM_Set(x), reply);
	if (!rc) handle = reply.refData();
    } catch (exception& e) {
        cerr << mName << ": Caught exception in AlarmClient::setAlarm: "
	     << e.what() << endl;
	rc = Invalid;
    }
    if (rc == BadAddress) {
	cerr << mName << ": Alarm manager moved!" << endl;
	mServer = 0;
    }
#endif
    return rc;
}

//======================================  Set debug print level
void 
AlarmClient::setDebug(int level) {
    mDebug = level;
#ifndef DMTOFFLINE
    AppClient::setDebug(level);
#endif
}

//======================================  Disable a specified alarm type
error_type 
AlarmClient::setDisable(const AlarmID& aid, bool flag) {
#ifndef DMTOFFLINE
    int yn = flag ? 1 : 0 ;
    return sendMessage(ALM_SetDisable(pair<AlarmID,int>(aid, yn)));
#else
    return NotOpen;
#endif
}

//======================================  Set client monitor name
void 
AlarmClient::setName(const string& name) {
    mName = name;
}

//======================================  Get sppecified alarm status
error_type 
AlarmClient::testAlarm(const AlarmHandle& x, AlarmData& data) {
    error_type rc = NotOpen;
#ifndef DMTOFFLINE
    try {
        rc = open();
	if (rc) return rc;
        ALM_Status reply;
        rc = request(mServer, ALM_Request(x), reply);
	if (!rc) data = reply.refData();
    } catch (exception& e) {
        cerr << mName << ": Caught exception in AlarmClient::testAlarm: "
	     << e.what() << endl;
	rc = Invalid;
    }
    if (rc == BadAddress) {
	cerr << mName << ": Alarm manager moved!" << endl;
	mServer = 0;
    }
#endif
    return rc;
}

//======================================  Close connection to Alarm Manager
void
AlarmClient::close(void) {
#ifndef DMTOFFLINE
    mServer = MsgAddr(0, 0, 0);
#endif
}

//======================================  Open connection to Alarm Manager
error_type
AlarmClient::open(void) {
    error_type rc(OK);
#ifndef DMTOFFLINE
    if (mServer == MsgAddr(0, 0, 0)) {
        NameClient nc;
	if (mDebug) nc.setDebug(mDebug);
	rc = nc.lookup(AlarmServerName, mServer);
	if (mDebug > 1) cout << "Alarm server address: " << mServer << endl;
    }
#endif
    return rc;
}

#ifndef DMTOFFLINE
//======================================  Send message to the Alarm Manager
error_type
AlarmClient::sendMessage(const Message& msg) {
    error_type rc;

    //----------------------------------  Make sure we have a process ID
    MsgHeader reply;
    try {
        rc = open();
	if (rc) return rc;
        rc = request(mServer, msg, reply, 0, 0);
    } catch (exception& e) {
        cerr << mName << ": Caught exception in AlarmClient::sendMessage: "
	     << e.what() << endl;
	rc = Invalid;
    }
    if (rc == BadAddress) {
	cerr << mName << ": Alarm manager moved!" << endl;
    } else if (!rc && reply.getMsgType() != almAck) {
        rc = Invalid;
    }
    return rc;
}
#endif
