/************************
 *** Including standard C++ headers (I/O, STL, math, etc.)
 *** not referenced in susConstructs.hh
 ************************/
#include <string>
#include <sstream>
#include <iomanip>
#include <time.h>
#include <stdio.h>
#include <sys/types.h>

/************************
 *** Including GDS headers (frame/LIGO-LW/trigger I/O, one-use classes, etc.)
 *** not referenced in susConstructs.hh
 ************************/
#include "DVector.hh"

/************************
 *** Including susConstructs header
 ************************/
#include "susConstructs.hh"


using namespace std;
using namespace channel;
//using namespace xml;


/***************************************************************************
 *
 * susStatistic functions
 * (susStatistic, ~susStatistic, operator =, setName, setType,
 *  setRefresh, dumpResults, Reset, updateStats,
 *  isInitialized)
 *
 ***************************************************************************/

//NOTE: to avoid compile-time errors, the default settings in the objects below
//were deleted since they were declared in the susConstructs.hh file
//starting on line 107 or so.

 /// susStatistic default constructor
susStatistic::susStatistic(const std::string& statID,
			   const channel::updateType type, 
			   const Interval& rate,
			   const unsigned int lt, const unsigned int st,
			   const double initval,
			   const unsigned int debug) :
  StatisticsConstruct(statID, type, rate, initval, debug), ltlength(lt), stlength(st)
  // tmp1(0), tmp2(0), mDebug(debug), mInit(0), mInitVal(initval)
 {
  if (mDebug > 2) cout <<"Creating susStatistic -- " <<getName()<<endl;
  if (mDebug > 3) cout <<"   (susStatistic "<<getName()<<") "<<this<<endl;
}

/// susStatistic copy constructor
susStatistic::susStatistic(const susStatistic& stat,
			   const unsigned int debug ):
  StatisticsConstruct(stat, debug) {
  //mDebug = debug;
  if (debug > 2) cout <<"Copying susStatistic -- "<<stat.getName()<<endl;
  *this = stat;
 if (mDebug > 2) cout << "susStatistic copy constructor used" << endl;
}

/// susStatistic destructor
susStatistic::~susStatistic(void) {
  if (mDebug > 2) cout <<"Destroying susStatistic -- " <<getName()<<endl;
  if (mDebug > 3) cout <<"   (susStatistic "<<getName()<<") "<<this<<endl;
}

/// susStatistic "clone" method, to allocate new susStatistics
susStatistic *susStatistic::clone(void) const {
  if (mDebug > 2) cout <<"Cloning susStatistic -- "<<getName()<<endl;
  susStatistic *p = new susStatistic(*this, mDebug);
  if (mDebug > 3) cout <<"   (susStatistic "<<getName()<<") "<<p<<endl;
  return p;
}

/// resets the statistics -- continuous stats must be overridden to be reset
void susStatistic::Reset(const bool override) {
  if (mDebug && override) cout <<"Overriding cumulative stats."<<endl;
  if ((StatisticsConstruct::getType() == uRefreshed) || override) {
    tmp1 = 0.0;
    tmp2 = 0.0;
    results.clear();
    mData.clear();
  }
  if (mDebug > 2) cout << "susStatistic Reset used" << endl;
}


/// updates the statistics for (possibly filtered) data
list<statResult>*
susStatistic::updateStats(const TSeries& ts, const string statID) {
  if (ts.isEmpty()) return &results; // taking care of the obvious stuff
 /* So the procedure here is to calculate the rms of every data stream.  Pretty
  * easy, and it should be easier with constructs like these.
  *
  * The trick is in storing the proper data values.  Since RMS can be construed
  * as <x^2> - <x>^2, we can save the sum(x) in prelim and the sum(x^2)
  * in actual, then we can divided by N, located in statcount, when we want the
  * RMS.  This should minimize the roundoff errors.
  */
  // We should probably check if now is a good time to reset the statistics
  //  if (isDue(ts.getStartTime()) && mDebug)
  //    cout <<"Statistics-- "<<getName()<<" has been refreshed."<<endl;
  //----------- Got to have the right tools for the job
  if ( mData.find(statID) == mData.end() ) {
    susData thisdata;
    mData[statID]= thisdata;
  }
  
  if (mDebug > 2) cout << "susStatistic 'UpdateStats' used" << endl;
  susData* dataptr =&(mData.find(statID)->second);
  TSeries tmpts = ts;
  dataptr->lt.push_back(tmpts);
  dataptr->st.push_back(tmpts);

  if (dataptr->lt.size() > ltlength)  dataptr->lt.pop_front();
  if (dataptr->st.size() > stlength)  dataptr->st.pop_front();

  //------variables for statistics------
  const TSeries* cts;
  //  double ltmean=0;
  //  double ltvar=0;
  double tmpmean=0;
  // double tmpvar=0;
  double ltstd=0;
  unsigned int  N = dataptr->lt.begin()->getNSample();
  double sum=0;
  double diff=0;
  double totsum=0;
  double ststd=0;



  //iterate through all ten time series
 for (unsigned int k=0; k < dataptr->lt.size(); k++) {
    cts = &(dataptr->lt[k]);
    if (mDebug > 2) cout << "looping through deque slot #" << k << endl;
    //record beginning time of computation
   
    //     int t1=time(NULL);
    tmpmean = cts->getAverage(); 
    if (mDebug > 2) cout << "the tmpmean for this slot is:" << tmpmean << endl;
    //now iterate  through time series data points
    for (unsigned int i=0; i < cts->refDVect()->getLength(); i+=2){
         diff =((cts->refDVect()->getDouble(i)) - (tmpmean));
         sum += (diff*diff);
         }
    tmpmean = 0;
    if (mDebug > 2) cout << "tmpmean reset check :"<< tmpmean << endl;
    if (mDebug > 2) cout << "the current sum is:" << sum << endl;
    totsum += sum;
    if (mDebug > 2) cout << "the new totsum is:"  << totsum << endl;
    sum =0;
    if (mDebug > 2) cout << "sum should reset to zero, see: " << sum << endl;
    //end time
    //    int t2=time(NULL);
    //    if (mDebug > 1) cout << "Time to work through a deque= " <<  t2-t1 << endl;
 }
    ltstd = sqrt( totsum / (N*dataptr->lt.size()));
    if (mDebug > 2) cout << "the ltstd is: " << ltstd << endl;
    totsum = 0;
    if (mDebug > 2) cout << "check totsum reset: " << totsum << endl;

  if (mDebug > 2) cout <<"  Statistic-- "<<getName()<<":ltstd="<< ltstd << ", N=" << N << endl; 
 

  sum = 0;
  totsum = 0;
  diff = 0;
  tmpmean = 0;

 //iterate through  time series
 for (unsigned int k=0; k < dataptr->st.size(); k++) {
    cts = &(dataptr->st[k]);
    tmpmean += cts->getAverage(); 
    //now iterate  through time series data points
    for (unsigned int i=0; i < cts->refDVect()->getLength(); i++){
         diff =((cts->refDVect()->getDouble(i)) - (tmpmean));
         sum += (diff*diff);
         }

    totsum += sum;
    sum =0;
 }
    ststd = sqrt( totsum / (N*dataptr->st.size()));
    totsum = 0;

  if (mDebug > 2) cout <<"  Statistic-- "<<getName()<<":ststd="<< ststd << ", N=" << N << endl; 

  //----------------end new statistics routine--------------------------------


      
  // loop to calculate the rms of the short-term deque
/*  
  double stmean=0;
  double stvar=0;
  tmpmean=tmpvar=0;
  double ststd=0;
  N = dataptr->st.begin()->getNSample();

  for (unsigned int k=0; k < dataptr->st.size(); k++) {
    cts = &(dataptr->st[k]);
    for (unsigned int i=0; i < cts->refDVect()->getLength(); i++)
        tmpvar += cts->refDVect()->getDouble(i) * cts->refDVect()->getDouble(i);
    tmpmean += cts->getAverage();
    stvar += tmpvar;
    stmean += tmpmean;
    if (mDebug > 3) cout <<"  tmpStats: k="<<k<<", stmean="<<tmpmean<<", stvar="<<tmpvar<<endl;
  }
  stvar /= (N*dataptr->st.size());
  stmean /= dataptr->st.size();
  ststd = sqrt( stvar-(stmean*stmean));
  if (mDebug > 2) cout <<"  Statistic-- "<<getName()<<":ststd="<<ststd<< " sum(x)="<<stmean
		       <<", sum(x^2)="<<stvar <<", N=" << N << endl;   
*/

 
  if (results.empty()) results.insert(results.end(), statResult());
  list<statResult>::iterator cur=results.begin();
  list<statResult>::iterator next=results.begin();
  // Fortunately, we get to assume double's, since that's what IIRFilter uses
  if ( !(ts.refDVect()->D_data()) ) { cerr<<"Data's not double!"<<endl; throw;}
 //----------------------  Gentlemen, start your iterations! Find that, stat!
  while (next != results.end()) {
   //---------------------  Next we'll check if this is our work-in-progress!
    if ((next->comment == statID) || (next->name == "")) {cur = next; break;}
  //---------------------  Finally, if nothing else, we roll our own.
    next++;

    if (next == results.end()) {
      cur = results.insert(next, statResult());
      cur->name="";		// This initializes the last statResult --
      break;			// it can be used for any TSeries
    }
  }
  //----------- (by now cur should be pointing at the proper statResult)
 
  if (mDebug > 3) cout << "susStatistic updateStats" << endl;
  //-----------  Let's (re)initialize our new statResult if we don't have one yet.
  if (cur->name == "" || cur->state == sNewStat) {
    if (mDebug > 1)
      cout <<"  Starting new statistic-- "<<getName()<<":"<<statID<<endl;
    cur->name     = getName();
    cur->comment  = statID;
    cur->channel  = ts.getName();
    cur->units    = "counts";
    cur->statcount= 0;
    cur->prelim   = ststd;
    cur->actual   = ltstd;
    cur->startTime= dataptr->st.begin()->getStartTime();
    cur->duration = dataptr->st.begin()->getInterval()*(double)N;
    cur->type     = dUnknown;
    cur->data.dV  = dataptr;
    cur->state    = sDoneStat;
  //-----------  Otherwise just start where we left off
  } else {
    cur->statcount += 0;
    cur->actual    += ltstd;
    cur->prelim    += ststd;
    cur->data.dV    = dataptr;
    cur->state      = sDoneStat;
  }

  // Just some debug info... Not to worry!
  if (mDebug)
    cout<<statID<<": cur->name="<<cur->name<<", cur->state="<<cur->state<<endl;

  return &results;
}

/// checks whether the susStatistic object has been setup correctly
bool susStatistic::isInitialized(void) const {
  string scname("susStatistic");
  if (!mInit) {
    bool allinit=true;
    allinit &= StatisticsConstruct::isInitialized();
    if (mDebug > 2)
      cout << "susStatistic "<<getName()<<" allinit = "<<allinit<<endl;
    if (ltlength ==0) allinit = false;
    if (mDebug > 2)
      cout << "susStatistic "<<getName()<<" allinit = "<<allinit<<endl;
    if (stlength ==0) allinit = false;
    if (mDebug > 2)
      cout << "susStatistic "<<getName()<<" allinit = "<<allinit<<endl;
    mInit = allinit;
  }
  if (mDebug > 2)
    cout << "susStatistic "<<getName()<<"'isInitialized' used" << endl;
  return mInit;
}


//---------------------end susStatistic code, begin susLog code----------------


/***************************************************************************
 *  Below are functions adapted from the ResultConstruct class of chInterface
 *  (basically "ResultConstruct" replaced with "susLog")
 **************************************************************************/

//NOTE: as above with susStatistics, the parameters have been removed below 
//but can be found in susConstructs.hh starting around line 67 or so

/// susLog default constructor
susLog::susLog(const string& resultID, const updateType type,
	       const unsigned int outmask, ostream& out,
	       const Interval& update, const unsigned int debug ) :
  ResultConstruct(resultID, type, outmask, out, update, debug) {
  if (debug > 2) cout <<"Creating susLog -- " <<getName()<<endl;
  if (debug > 3) cout <<"   (susLog  "<<getName()<<") "<<this<<endl;
}

/// outputs the result
bool susLog::output(const statResult& result, const unsigned int outmask,
		    const bool override, ostream& out) {
  unsigned int thismask = outmask;
  if (!thismask) thismask = mOutmask;
  string histname = result.name + "_" + result.comment;
  bool success=true;
  if (ResultConstruct::mDebug > 2) cout << "using susLog output" << endl;



  if ((result.state == sDoneStat) || override) {
    if (ResultConstruct::mDebug > 1) cout <<histname<<": state= "<<result.state
			 <<", override="<<override<<endl;

    //---------  Here's the formatted output (sent to any ostream)
    ostringstream output("", ios::ate);
    char tstr[40] = "";
    TimeStr(result.startTime, tstr, " %y.%m.%d %H:%N:%S ");
    // We're making some mighty assumptions here;
    // see StatisticsConstruct:updateStats for details
    //    double sAvg  = result.prelim / double(result.statcount);
    //    double sDev = sqrt(result.actual / double(result.statcount) - sAvg * sAvg);
    output <<result.channel  <<tstr << setw(10)<<setprecision(9)<<setfill('0')
	   <<double(result.startTime.totalS())<<" "
	       //	   <<setw(12)<<setprecision(5)<<setfill(' ')<<result.duration
	   <<setw(12)<<setprecision(5)<<setfill(' ')<<result.actual
      //  <<setw(12)<<setprecision(5)<<setfill(' ')<<result.statcount
	   <<setw(12)<<setprecision(5)<<setfill(' ')<<result.prelim << "  ";
	       //	   <<"'Avg'  = "<<setw(12)<<setprecision(5)<<setfill(' ')<<sAvg
	       //	   <<"'SDev' = "<<setw(12)<<setprecision(5)<<setfill(' ')<<sDev;
    /*    switch (result.type) {
    case dTSeries:    output <<" TSeries    "; break;
    case dFSeries:    output <<" FSeries    "; break;
    case dFSpectrum:  output <<" FSpectrum  "; break;
      //    case dHistogram1: output <<" Histogram1 "; break;
    case dUnknown:
    default:          output <<" Unknown    "; break;
    }*/
    //    if (mDebug) output <<result.data.dTS;
    if (override)
      switch (result.state) {
      case sNewStat:  output <<" NewStat  "; break;
      case sInStat:   output <<" InStat   "; break;
      case sDoneStat: output <<" DoneStat "; break;
      default: break;
      }
    //--------addition to correct for lack of histname
    output << (result.name + "_" + result.comment);

    //---------  standard output check
    if (thismask & 1) {
      cout << output.str() << endl;
    }
    //---------  default output stream check
    if (thismask & 2) {
      if (mOutput && (!(thismask & 1) || mOutput != &cout) )
	*mOutput << output.str() << endl;
      else success = false;
    }
    //---------  given output stream check
    if (thismask & 4) {
      if (!(thismask & 1) || &out != &cout) out << output.str() << endl;
      else success = false;
    }
    //---------  internal trend
    if (thismask & 32) {
      if(ResultConstruct::mDebug > 1)
	cout << "susLog:    outputting to trend" << endl;
      if (mTrend){
	// Long-term RMS datapt
	if(ResultConstruct::mDebug > 2)
	  cout << "\tlongrms value is: " << result.actual << endl;
	string tempstring =result.name + "_" + result.comment + "_longrms";
	// Verify that the trend channel exists already
	if (!mTrend->exists(tempstring.c_str())){
	  if(ResultConstruct::mDebug > 2)
	    cout << "susLog:    adding channel: "<< tempstring  << endl;
	  mTrend->addChannel(tempstring.c_str());
	} else {
	  if(ResultConstruct::mDebug > 2)
	    cout << "\tresult.startTime is: " << result.startTime << endl
		 << "\tpoint appended to: " << tempstring << endl;
	  mTrend->trendData(tempstring.c_str(), result.startTime, (Trend::math_t)result.actual);
	}
	// Short-term RMS datapt
	if(ResultConstruct::mDebug > 2)
	  cout << "\tshortrms value is: " << result.prelim << endl;
	tempstring = result.name + "_" + result.comment + "_shortrms";
	// Verify that the trend channel exists already
	if(!mTrend->exists(tempstring.c_str())){
	  if(ResultConstruct::mDebug > 1)
	    cout << "adding channel: "<< tempstring  << endl;
	  mTrend->addChannel(tempstring.c_str());
	} else {
	  if(ResultConstruct::mDebug > 1)
	    cout << "\tpoint appended to: " << tempstring << endl;
	  mTrend->trendData(tempstring.c_str(),  result.startTime, (Trend::math_t)result.prelim);
	}
      }
      else success = false;
    }
    
    //    if (result.state == sDoneStat) result.state = sNewStat;
}
return success;
}
/// end of output


/// susLog "clone" method, to allocate new susLogs
susLog *susLog::clone( void ) const {
  if (ResultConstruct::mDebug > 2) cout <<"Cloning susLog -- "<<getName()<<endl;
  susLog *p = new susLog(*this, ResultConstruct::mDebug);
  if (ResultConstruct::mDebug > 3) cout <<"   (susLog "<<getName()<<") "<<p<<endl;
  return p;
}


/// checks whether the susLog object has been setup correctly
bool susLog::isInitialized(void) const {
  string rcname("susLog");
  if (!mInit) {
    bool allinit=true;
    if (getName().empty()) {
      if (ResultConstruct::mDebug) cout <<"susLog doesn't have a name."<<endl;
      allinit=false; // this should never happen, but could...
    }
    allinit &= ResultConstruct::isInitialized();

    /* else rcname = getName();
    if (mType == uUnknown) {
      if (ResultConstruct::mDebug) cout <<rcname<<" has unknown update type"<<endl;
      allinit=false;
    }
    if ((mType == uRefreshed) && mUpdate <= Interval(0.0)) {
      if (ResultConstruct::mDebug) cout <<rcname<<" doesn't have a positive refresh rate"<<endl;
      allinit=false;
    }
    if (mOutput == NULL) {
      if (ResultConstruct::mDebug) cout <<rcname<<" has no valid output pointer"<<endl;
      //      allinit=false;
      } */
    mInit = allinit;
  }
  return mInit;
}


//-------------------------------end susLog code and begin susEvent code-------


/// susEvent default constructor
susEvent::susEvent(const string& resultID, const updateType type,
		   const unsigned int outmask, ostream& out,
		   const Interval& update, const unsigned int debug,
		   double shortThresh, double longThresh) :
  ResultConstruct(resultID, type, outmask, out, update, debug),
  stThresh(shortThresh), ltThresh(longThresh) {

  if (debug > 2) cout <<"Creating susEvent -- " <<getName()<<endl;
  if (debug > 3) cout <<"   (susEvent  "<<getName()<<") "<<this<<endl;
}


void susEvent::setOSC(OperStateCondList& oscpointer,const string oscname){
  pOSC = &oscpointer;
  nameOSC = oscname;
}
 
 

/// susEvent assignment operator
susEvent& susEvent::operator =(const susEvent& rhs) {
  //(ResultConstruct)*this = (ResultConstruct)rhs;
  dataPrefix = rhs.dataPrefix;
  logPrefix  = rhs.logPrefix;
  stThresh   = rhs.stThresh;
  ltThresh   = rhs.ltThresh;
  pTrig      = rhs.pTrig;
  /*  setName(rhs.getName());
  mType         = rhs.mType;
  mUpdate       = rhs.mUpdate;
  mEntryTime    = rhs.mEntryTime;
  mOutput       = rhs.mOutput;
  mHistMap      = rhs.mHistMap;
  mDebug        = rhs.mDebug;
  mInit         = rhs.mInit;
  */

  if (ResultConstruct::mDebug > 2) 
    cout <<"Assigned susEvent -- "<<getName()<<endl;
  if (ResultConstruct::mDebug > 3) 
    cout <<"   (susEvent "<<getName()<<") "<<this<<endl;
  return *this;
}

/// outputs the result
bool susEvent::output(const statResult& result, const unsigned int outmask,
		      const bool override, ostream& out) {
  unsigned int thismask = outmask;
  if (!thismask) thismask = mOutmask;
  string histname = result.name + "_" + result.comment;
  bool success=true;
  
  if (ResultConstruct::mDebug > 2) cout << "using susEvent results" << endl;

  if ((result.state == sDoneStat) || override) {
    if (ResultConstruct::mDebug > 1) {
      cout << histname << ": state= " << result.state <<", override="
	   << override << endl;
    }

    //---------  Here's the formatted output (sent to any ostream)
    ostringstream output("", ios::ate);
    char tstr[40] = "";
    TimeStr(result.startTime, tstr, " %y.%m.%d %H:%N:%S ");
    // see StatisticsConstruct:updateStats for details

    // begin subroutine for short-term event generation:
    // short-term events should only alarm when the threshold is exceeded
    // AND the IFOs are in-lock (we expect mirrors to be jostled during lock
    // aquisition and don't want to alarm on those). 

    if (result.prelim > stThresh*result.actual &&
	(nameOSC.empty() || pOSC->satisfied(nameOSC.c_str())) ){
      
      if (ResultConstruct::mDebug > 1) 
	cout << "OUTPUTTING SHORT TERM EVENT " << endl;
  
    //=====================  Send event to Trigger Manager
 
      string subID = result.name;
      string Ifo   = result.name.substr(0,2);
      trig::TrigRslt trigData("susEvent_short",subID.c_str());
      trigData.setTime(result.startTime);
      trigData.setIfos(  Ifo.c_str()  );
      trigData.setIntensity(result.prelim);
      trigData.setSignificance(result.prelim/result.actual);
      trigData.setPriority(    trig::p_warn);
      trigData.setDisposition( trig::d_metaDB);
      trigData.setDuration( getUpdate() );
//        trigData.setData( ch->fTrigComment.c_str(), ch->fTrigComment.size() );
      if (pTrig) {
	int trigerr = pTrig->sendTrigger(trigData);
	trigData.setDisposition( trig::d_alarm);
	trigerr &= pTrig->sendTrigger(trigData);
	if ( trigerr  )
	  output <<"Error Sending Trigger "<<trigerr<<" "<<endl;
      }
    }

    // here is the routine for the long-term     

    if (result.actual > ltThresh ){
      
      if (ResultConstruct::mDebug > 1) {
	cout << "OUTPUTTING LONGTERM EVENT" <<endl;  
      }
  
    //=====================  Send event to Trigger Manager
 
      string subID = result.name;
      string Ifo   = result.name.substr(0,2);
      trig::TrigRslt trigData("susEvent_long",subID.c_str());
      trigData.setTime( result.startTime );
      trigData.setIfos( Ifo.c_str() );
      trigData.setIntensity( result.actual);
      trigData.setSignificance(result.actual/ltThresh);
      trigData.setPriority(    trig::p_warn);
      trigData.setDisposition( trig::d_metaDB);
      trigData.setDuration( getUpdate() );
//        trigData.setData( ch->fTrigComment.c_str(), ch->fTrigComment.size() );
      if (pTrig){
	int trigerr = pTrig->sendTrigger(trigData);
	trigData.setDisposition( trig::d_alarm);
	trigerr &= pTrig->sendTrigger(trigData);
	if ( trigerr )
	  output <<"Error Sending Trigger: "<<trigerr<<" "<<endl;
      }
    }

    //---------  standard output check
    if (thismask & 1) {
      cout << output.str() << endl;
    }
    //---------  default output stream check
    if (thismask & 2) {
      if (mOutput && (!(thismask & 1) || mOutput != &cout) )
	*mOutput << output.str() << endl;
      else success = false;
    }
    //---------  given output stream check
    if (thismask & 4) {
      if (!(thismask & 1) || &out != &cout) out << output.str() << endl;
      else success = false;
    }
    if (result.state == sDoneStat) result.state = sNewStat;
  }
return success;
}
/// end of output


/// susEvent "clone" method, to allocate new susEvents
susEvent *susEvent::clone( void ) const {
  if (ResultConstruct::mDebug > 2) cout <<"Cloning susEvent -- "<<getName()<<endl;
  susEvent *p = new susEvent(*this, ResultConstruct::mDebug);
  if (ResultConstruct::mDebug > 3) cout <<"   (susEvent "<<getName()<<") "<<p<<endl;
  return p;
}


/// checks whether the susEvent object has been setup correctly
bool susEvent::isInitialized(void) const {
  string rcname("susEvent");
  if (!mInit) {
    bool allinit=true;
    if (getName().empty()) {
      if (ResultConstruct::mDebug) cout <<"susEvent doesn't have a name."<<endl;
      allinit=false; // this should never happen, but could...
    }
    allinit &= ResultConstruct::isInitialized();

    mInit = allinit;
  }
  return mInit;
}
