/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef KLEINEWELLEN_HH
#define KLEINEWELLEN_HH

#include "MultiStream.hh"
#include "FilterDesign.hh"
#include "Pipe.hh"
#include "TSeries.hh"
#include <fstream>
#include <string>
#include <iosfwd>
#include <cstdio>
#include <vector>
#include <list>

class LPEFilter;

struct Element {
    Element(double energy, double normEnergy, int index, int scale, 
	    double startTime, double endTime, double dt, int clusterID)
	: energy(energy), normEnergy(normEnergy), index(index), scale(scale), 
	  startTime(startTime), endTime(endTime), dt(dt), clusterID(clusterID) 
    {}
    int dist(const Element& b) const;
    bool operator<(const Element& b) const {return startTime < b.startTime;}
    double energy;
    double normEnergy;
    int index;
    int scale;
    double startTime;
    double endTime;
    double dt;
    int clusterID;
};

inline int 
Element::dist(const Element& b) const {
    int dscale = scale - b.scale;
    if (dscale <= 0) return -dscale + abs((index   >> -dscale) - b.index);
    else             return  dscale + abs((b.index >>  dscale) -   index);
}

typedef std::vector<Element>   element_vect;
typedef element_vect::iterator element_iter;

struct Event {
    Event(const std::string& name);
    bool operator<(const Event& b) const {return startTime < b.startTime;}
    Event& operator += (const Element& el);
    void average(const Time& toff, double fs);
    double startTime;
    double endTime;
    double centralTime;
    double frequency;
    double energy;
    double normEnergy;
    int npts;
    double significance;
    std::string channelName;
};
    
typedef std::vector<Event>   event_vect;
typedef event_vect::iterator event_iter;

// negative natural logarithm of significance
double 
logSignificance(double normalizedEnergy, double timeFrequencyArea,
		double precision);

class kwChan {
public:
    /**  Channel processor constructor.
      */
    kwChan(void);

    /**  Copy constructor
     */
    kwChan(const kwChan& x) {*this = x;}

    /**  Channel processor destructor.
      */
    ~kwChan(void);

    /**  Assignment operator
     */
    kwChan& operator=(const kwChan& x);

    /**  Print out the channel parameters.
     */
    std::ostream& dump_params(std::ostream& out) const;

    /**  Process a data stride.
     */
    void process(MultiDacc& dacc);

    /**  Set up the sample rate dependent filtering.
      */
    void setup(void);

public:
    std::string channel;       // channel name of data (noise)
    std::string filter;        // prefilter option
    std::string filename;      // injection file name
    double   significance;     // significance threshold for reporting clusters
    double   threshold;        // gaussian threshold for black pixels
    Interval stride;           // Processing chunk size.
    int      fs;               // sample rate (after any decimations)
    int      size;             // size of the wavelet decomposition array
    int      lpefLength;
    int      maxDist;          // max distance between two black pixels
    int      scale;            // log(size), base 2
    float    lowFrequency;     // low frequency boundary
    float    highFrequency;    // high frequency boundary
    int      lowScale;         //  obsolted but take precedence if defined
    int      highScale;
    int      decimateFactor;   // number of decimations to do
    Interval transientDuration; // amount of data to skip at boundaries
    Wavelet*     wavePtr;      // pointer to wavelet functions library
    Pipe*        prefilter;    // decimator and highpass prefilter
    LPEFilter*   lpefilter;    // linear predictive whitening filter
    Time         nextTime;     // bookeeping to check for data continuity
    TSeries  inputBuffer;      // Input buffer to decouple strides.

    //----------------------------------  Injection parameters.
    std::string injections;    // channel name of injections to add
    float       factor;        // injection scale factor.
    float*      waveInjection; // loaded injection signal
};


class kleineWelle : public MultiStream {
public:

    kleineWelle(int argc, const char *argv[]);
    ~kleineWelle(void);

    void ProcessData(void);

    bool pop_segment(void);

private:
    Time     segmentStart;     // top bound of a segment to be analyzed
    Time     segmentStop;     // bottom bound
    Interval stride;     // number of seconds to analyze at a time

    typedef unsigned long gps_type;
    typedef std::pair<gps_type, gps_type> segment_node;
    typedef std::vector<segment_node> segment_vect;
    segment_vect mSegVect;

    typedef std::list<kwChan>      channel_list;
    typedef channel_list::iterator channel_iter;
    channel_list mChanList;
};

std::ostream& 
operator<<(std::ostream& output, const Event& a);

#endif // KLEINEWELLE_HH 
