/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef KLEINEWELLEN_HH
#define KLEINEWELLEN_HH

#include "MultiStream.hh"
#include "FilterDesign.hh"
#include "Pipe.hh"
#include "TSeries.hh"
#include "TrigClient.hh"
#include "TrigRslt.hh"
#include "Segment.hh"
#include "autopipe.hh"
#include "OperStateCondList.hh"
#include "frame_name.hh"
#include <fstream>
#include <string>
#include <iosfwd>
#include <cstdio>
#include <vector>
#include <list>
#include <time.h>

class LPEFilter;

struct Element {
    Element(double energy, double normEnergy, int index, int scale,
        double startTime, double endTime, double dt, int clusterID)
    : energy(energy), normEnergy(normEnergy), index(index), scale(scale),
      startTime(startTime), endTime(endTime), dt(dt), clusterID(clusterID)
    {}
    int dist(const Element& b) const;
    bool operator<(const Element& b) const {return startTime < b.startTime;}
    double energy;
    double normEnergy;
    int index;
    int scale;
    double startTime;
    double endTime;
    double dt;
    int clusterID;
};

inline int
Element::dist(const Element& b) const {
    int dscale = scale - b.scale;
    if (dscale <= 0) return -dscale + abs((index   >> -dscale) - b.index);
    else             return  dscale + abs((b.index >>  dscale) -   index);
}

typedef std::vector<Element>   element_vect;
typedef element_vect::iterator element_iter;

struct Event {
    Event(const std::string& name);
    bool operator<(const Event& b) const {return startTime < b.startTime;}
    Event& operator += (const Element& el);
    void average(const Time& toff, double fs);
    double startTime;
    double endTime;
    double centralTime;
    double frequency;
    double energy;
    double normEnergy;
    int npts;
    double significance;
    std::string channelName;
};

typedef std::vector<Event>   event_vect;
typedef event_vect::iterator event_iter;

// negative natural logarithm of significance
double
logSignificance(double normalizedEnergy, double timeFrequencyArea,
        double precision);

class kwChan {
public:
    /**  Channel processor constructor.
      */
    kwChan(const std::string& name);

    /**  Copy constructor
     */
    kwChan(const kwChan& x) {*this = x;}

    /**  Channel processor destructor.
      */
    ~kwChan(void);

    /**  Assignment operator
     */
    kwChan& operator=(const kwChan& x);

    /**  Print out the channel parameters.
     */
    std::ostream& dump_params(std::ostream& out) const;

    /**  Process a data stride.
     */
    event_vect process(MultiDacc& dacc);

    /**  Set up the sample rate dependent filtering.
      */
    void setup(const TSeries* ts);

public:                        // public options that need to be set from outside
    std::string channel;       // channel name of data (noise)
    float    lowFrequency;     // low frequency boundary
    float    highFrequency;    // high frequency boundary
    std::string uniqueID;      // channel_lowFrequency_highFrequency
    double   significance;     // significance threshold for reporting clusters
    double   threshold;        // gaussian threshold for black pixels
    int      maxDist;          // max distance between two black pixels
    int      decimateFactor;   // number of decimations to do
    Interval transientDuration; // amount of data to skip at boundaries
    auto_pipe    prefilter;    // decimator and highpass prefilter
    LPEFilter*   lpefilter;    // linear predictive whitening filter
    bool     skipNext;         // flag to skip one stride after filter reset
    bool     simdata;          // flag indicating simulated data

private:
    int      fs;               // sample rate (after any decimations)
    int      lowScale;         // frequency bounds converted into wavelet scale
    int      highScale;
    int      size;             // size of the wavelet decomposition array
    int      lpefLength;
    Wavelet*     wavePtr;      // pointer to wavelet functions library
    TSeries      tsLpef;       // linear predictive filtered timeseries

};

class kwRateFlag {
public:
    kwRateFlag(const std::string& name);
    ~kwRateFlag(void);
	std::string channelName;
	double sigThreshold;
	double rateThreshold;
	int nStride;
	int* counts;
};

class kleineWelle : public MultiStream {
public:

    kleineWelle(int argc, const char *argv[]);
    ~kleineWelle(void);

    void ProcessData(void);

private:
    TrigClient mTrigClient; // trigger manager client for XML output
    TrigClient mSegClient; // trigger manager client for DQ XML output
    Interval stride;     // number of seconds to analyze at a time
    OperStateCondList* mOsc;

    std::ifstream optionsFile;
    typedef std::list<kwChan>                   channel_list;
    typedef channel_list::iterator              channel_iter;
    typedef std::map<std::string,double>        vetoseg_map;
    typedef vetoseg_map::iterator               vetoseg_iter;
    typedef std::list<kwRateFlag>				rateflag_list;
    typedef rateflag_list::iterator				rateflag_iter;
    vetoseg_map  mVetoList;
    channel_list mChanList;
    rateflag_list mRateFlagList;
    std::string basename;
    std::string segname;
    frame_name  trg_path;
    frame_name  xml_path;
    frame_name  trend_path;
    frame_name  seg_path;
    Time lasttime;       // start time of previous anlaysis segment
    time_t rawtime;
    struct tm * timeinfo;

    void sendSeg(vetoseg_iter this_veto, Time start_sec, Time end_sec, int active);
    void sendRateSeg(rateflag_iter this_rateflag, Time start_sec, Time end_sec, int active);
};

std::ostream&
operator<<(std::ostream& output, const Event& a);

#endif // KLEINEWELLE_HH 
