/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "DQ_Bit.hh"
#include "DVecType.hh"

using namespace std;

/**  The %DQ_BitMask tool tests whether selected bit patterns are set. Two 
  *  masks are used:
  *  - all: if all the bits in the mask are set in all of the input channel
  *         data words, the result is true.
  *  - any: if any of the bits in the mask are set in any of the input 
  *         channel data words, the condition is true.
  *  Both the any and all conditions must be true for the output flag to be 
  *  set. If either of the masks (any or all) are zero the corresponding 
  *  condition evaluates to true.
  *  \brief Test bit patterns in an input channel.
  */
class DQ_BitMask : public DQ_bit {
public:
   /**  initialize the instance with the "any" and "all" masks from the
     *  parameter dictionary.
     */
   void init(void) {
      if (numeric_param_exists("any")) {
	 _any = uint32_t(numeric_param("any"));
      } else {
	 _any = 0;
      }
      if (numeric_param_exists("any")) {
	 _all = uint32_t(numeric_param("any"));
      } else {
	 _all = 0;
      }
   }

   /**  This function evaluates the mask conditions. All the data word in 
     *  the input time series (data[0]) are converted to uint32_t data type
     *  and then ORed together in any_mask and ANDed in all_mask. The 
     *  return value is true if any of the bits selected by the "any" parameter
     *  are set and if all the bits selected by the "all" parameter are set. 
     *  \brief Evaluate the bit condition
     *  \param data Contains a single time series (TSeries)
     *  \returns True if at least one selected "any" bit is set and all
     *           selected "all" bits are set.
     */
   bool bit_value(const tser_vect& data) {
      DVectU dv(*data[0].refDVect());
      size_t N=dv.getLength();
      if (!N) return 0;
      uint32_t any_mask = dv[0];
      uint32_t all_mask = dv[0];
      for (size_t i=1; i<N; i++) {
	 uint32_t dvi = dv[i];
	 any_mask |= dvi;
	 all_mask &= dvi;
      }
      return (!_any || (any_mask & _any) != 0) && ((all_mask & _all) == _all); 
   }
   
private:
   uint32_t _any;  ///< any (OR) mask
   uint32_t _all;  ///< all (AND) mask
};

//======================================  Define the dynamic load entry point
DQ_PLUGIN(DQ_BitMask)
