/* -*- mode: c++; c-basic-offset: 3; -*- */
#ifndef DQ_BIT_HH
#define DQ_BIT_HH 1

#include "TSeries.hh"
#include <iosfwd>
#include <map>
#include <vector>
#include <string>
#include <cmath>

/** \name DQ_bit
 */
//@{
/**  %DQ_bit is a base class for dq-module plugins that return a boolean state 
  *  to be recorded as a bit in a data quality vector. The %DQ_bit class 
  *  defines the API used to calculate the DQ state and manages the input 
  *  channel list, parameter dictionaries and other information needed to
  *  evaluate the data quality flag and set the requested channel bit.
  */
class DQ_bit {
public:
   typedef std::vector<std::string> chanlist_type; ///< Channel list data type

   /**  %Time series vector used to pass a set of channel data to the 
     *  bit evaluation method.
     */	
   typedef std::vector<TSeries>     tser_vect;
   
public:
   /**  Default data quality plug-in constructor (uses automatic data 
     *  constructors only).
     */  
   DQ_bit(void);

   /**  Data quality plug-in destructor.
     */
   virtual ~DQ_bit(void);

   /**  Get the channel name in which the data quality bit is to be set.
     *  This name is defined by the configuration file.
     *  \brief Get the data quality channel name.
     * 	\return Channel name string.
     */ 
   const std::string& bit_channel(void) const {return _flag_channel;}

   /**  Set the name of the channel in which the data quality flag bit 
     *  is to be set.
     *  \brief Set the data quality channel name.
     *  \param chan Channel name string.
     */
   void bit_channel(const std::string& chan);

   /**  Get the bit number to be set in the specified data quality channel
     *  if this flag is true. The bit number is an integer from 0-31 where 
     *  0 is the least significant bit.
     *  \brief Get the flag bit number.
     *  \returns Bit number for this flag.
     */
   int bit_number(void) const {return _flag_bit;}

   /**  Set the bit number in the specified data quality channel that is to
     *  indicate the state of this flag. The bit number is an integer from 
     *  0-31 where 0 is the least significant bit.
     *  \brief Set the flag bit number.
     *  \param bit bit number.
     */
   void bit_number(int bit);

   /**  This function is called to evaluate the current state of the data 
     *  quality flag. The input channels listed in the configuration file
     *  are passed as a vector of TSeries in the order of appearance
     *  in the input channel list.
     *  \brief Flag state evaluation method. 
     *  \param data Vector of time series.
     *  \return true if the data quality flag is to be set.
     */
   virtual bool bit_value(const tser_vect& data)=0;

   /**  Get a constant reference to the list of channel names that are to 
     *  be passed to the evaluation function.
     *  \brief Get the list of input channel names.
     *  \returns Constant reference to the list of input channel names.
     */
   const chanlist_type& channel_list(void) const {return _chan_list;}

   /**  Set the list of names of channels to be passed to the data quality 
     *  evaluation method.
     *  \brief Set the input channel name list.
     *  \param chans A list of channel names from which the list for this
     *               flag is to be set.
     */ 
   void channel_list(const chanlist_type& chans);

   /**  Get the data quality tool (class) name.
     *  \brief Data Quality class name.
     *  \returns Reference to the data quality class name string.
     */
   const std::string& dq_class(void) const {return _dq_class;}

   /**  Set the data quality tool (class) name.
     *  \brief Set the DQ class name.
     *  \param dqclass Constant reference to the data quality class name string.
     */
   void dq_class(const std::string& dqclass);

   /**  Get the data quality flag name. If this is used to generate a segment,
     *  this would be used as the segment name.
     *  \brief Get the DQ flag name.
     *  \returns Constant reference to the data quality flag name.
     */
   const std::string& dq_name(void) const {return _name;}

   /**  Set the data quality flag name. If this is used to generate a segment,
     *  this would be used as the segment name.
     *  \brief Set the DQ flag name.
     *  \param dqname Constant reference to the data quality flag name.
     */
   void dq_name(const std::string& dqname);

   /**  Initialize the flag evaluation tool. Note that this is an optional 
     *  virtual method that may be used to perform initialization of the
     *  tool instance after it has been constructed. Typical uses would be
     *  to construct filters or to evaluate operation parameters or constants
     *  based on the parameters specified in the configuration file.
     *  \brief Tool initialization function.
     */
   virtual void init(void);

   /**  Tell the dq monitor to invert the sense of this dq flag, \e i.e. if 
     *  the function returns true, it indicates that the appropriate bit of
     *  the DQ channel bit mask is to be cleared.
     *  \brief Tool initialization function.
     *  \param inv true if the sense of the function output is to be inverted
     */
   void invert(bool inv);

   /**  Test whether the result state is to be inverted when setting the 
     *  data quality bit state. Specifically, the data quality bit will be
     *  cleared if the plugin return value and the invert flag are both true
     *  (or both false). 
     *  \brief Tool initialization function.
     *  \return true if the sense of the function output is to be inverted
     */
   bool invert(void) const {return _invert;}

   /**  Get a numeric parameter value that was defined in the configuration 
     *  file. Numeric parameters are converted to double precision floating
     *  point quantities. Boolean parameters are converted to 1.0 (true) or
     *  0.0 (false).
     *  \brief Get a numeric parameter value.
     *  \param name Name of the numeric or boolean parameter as specified in 
     *              the configuration file.
     *  \returns Value of the numeric parameter.
     */
   double numeric_param(const std::string& name) const;

   /**  Define a numeric parameter from its name and double precision floating
     *  point value.
     *  \brief Define a numeric parameter value.
     *  \param name Name of the numeric parameter.
     *  \param val  Numeric parameter value.
     */
   void numeric_param(const std::string& name, double val);

   /**  Test whether a numeric parameter with the specified name exists.
     *  \brief Test numeric parameter existence
     *  \param name Name of parameter to be tested.
     *  \return true if parameter exists.
     */
   bool numeric_param_exists(const std::string& name) const;
   
   /**  Print the configuration reconstructed from the contents of the %DQ_bit 
     *  instance to the specified output stream.
     *  \brief Print the configuration.
     *  \param out Output stream reference.
     */
   void put_config(std::ostream& out) const;

   /**  Get a string parameter value that was defined in the configuration 
     *  file.
     *  \brief Get a string parameter value.
     *  \param name Name of the string parameter as specified in the 
     *              configuration file.
     *  \returns Constant reference to the string value.
     */
   const std::string& string_param(const std::string& name) const;

   /**  Define a string parameter and set it to its nominal value.
     *  \brief Define a string parameter.
     *  \param name Name of the string parameter.
     *  \param val  Constant reference to the string value.
     */
   void string_param(const std::string& name, const std::string& val);

   /**  Test whether a string parameter with the specified name exists.
     *  \brief Test string parameter existence
     *  \param name Name of parameter to be tested.
     *  \return true if parameter exists.
     */
   bool string_param_exists(const std::string& name) const;

private:
   std::string   _name;          ///<  Data quality flag name
   std::string   _dq_class;      ///<  Plug-in class name
   chanlist_type _chan_list;     ///<  List of input channel names
   int           _flag_bit;      ///<  Bit number the flag is to be ANDed
   std::string   _flag_channel;  ///<  Output data quality channel name  
   bool          _invert;        ///<  Invert the return state.
protected:
   ///  Numeric parameter dictionary data types
   typedef std::map<std::string, double>      num_par_type;
   typedef num_par_type::iterator num_par_iter;
   typedef num_par_type::const_iterator const_num_par_iter;
   num_par_type _num_par;

   ///  String parameter dictionary data types
   typedef std::map<std::string, std::string> str_par_type;
   typedef str_par_type::iterator str_par_iter;
   typedef str_par_type::const_iterator const_str_par_iter;
   str_par_type _str_par;
};

//======================================  inline methods.
inline bool
DQ_bit::numeric_param_exists(const std::string& name) const {
   return _num_par.find(name) != _num_par.end();
}

inline bool
DQ_bit::string_param_exists(const std::string& name) const {
   return _str_par.find(name) != _str_par.end();
}

/**  This pre-processor macro defines the plug-in interface function. It 
  *  is called when the plug-in is loaded dynamically (by dlopen/dlsym) 
  *  and must be coded for every plug-in module to be used dynamically by 
  *  dq-module. The syntax is simply \c DQ_PLUGIN(\<class-name\>) and it is
  *  placed after the plug-in class definition.
  *  \brief Define a plugin definition macro.
  */
#define DQ_PLUGIN(name) extern "C" DQ_bit* name##_init(void) {return new name;}
//@}

#endif
