#include "TrigSpec.hh"
#include "FSeries.hh"
#include "Histogram1.hh"
#include "ParseLine.hh"
#include "Pipe.hh"
#include "TrigBase.hh"
#include "TSeries.hh"
#include <math.h>
#include <exception>
#include "TROOT.h"
#include "TApplication.h"
#include "TFile.h"
#include "TH2.h"
#include <iostream>

using namespace std;

// EXEROOT(TrigSpec)
EXECDAT(TrigSpec)

//======================================  Default constructor
HLimit::HLimit(void) 
  : mLower(0.0), mUpper(0.0), mName(""), mHisto(0)
{}

//======================================  Data constructor
HLimit::HLimit(limit_t low, limit_t hi, std::string name)
  : mLower(low), mUpper(hi), mName(name), mHisto(0)
{}

//======================================  Copy constructor
HLimit::HLimit(const HLimit& x) 
  : mLower(x.mLower), mUpper(x.mUpper), mName(x.mName)
{
    if (x.mHisto) mHisto = new Histogram1(*x.mHisto);
}

//======================================  Hist constructor
HLimit::~HLimit(void) {
    if (mHisto) delete mHisto;
}

//======================================  Compare to limits
bool
HLimit::inLimits(limit_t x) const {
    if (mHisto) mHisto->Fill(x);
    return mLower == mUpper || (x >= mLower && x < mUpper);
}

//======================================  Clear histogram
void 
HLimit::clear(void) {
    mHisto->Clear();
}

//======================================  (Re)Set limits
void 
HLimit::setLimits(limit_t low, limit_t hi) {
    mLower = low;
    mUpper = hi;
}

//======================================  Enable histogramming, define bins
void 
HLimit::setHisto(int nbin, limit_t binLo, limit_t binHi) {
    if (mHisto) delete mHisto;
    mHisto = new Histogram1(getName(), nbin, binLo, binHi);
}

//======================================  Trigger ID default constructor
TrigID::TrigID(const string& ID, const string& subID)
  : mTrigID(ID), mSubID(subID)
{}

//======================================  Trigger ID copy constructor
TrigID::TrigID(const TrigID& t) 
  : mTrigID(t.mTrigID), mSubID(t.mSubID)
{}

//======================================  Trigger ID destructor
TrigID::~TrigID(void) {
}

//======================================  Test if trigger matches ID
bool 
TrigID::isTrig(const trig::TrigBase& t) {
    if (mTrigID != t.getID() || mSubID != t.getSubID())  return false;
    return true;
}
 
//======================================  T/F plot default constructor.
TFPlot::TFPlot(const char* name, const char* ID, const char* subID)
  : TrigID(ID, subID), mName(name), mFilter(0), mSigCount(0),
    mSNValid(false), mBakDefault(true), mALimit(0, 0, name)
{
    if (subID[2] == ':') mChannel = subID;
    setTAxis( 64, -2.0,    2.0);
    setFAxis(128,  0.0, 2048.0);
    mALimit.setHisto(50, 0, 1000.0);
    string title("Trigger: ");
    title += ID;
    title += " SubID: ";
    title += subID;
    mSigHist.SetTitle(title.c_str());
}

//======================================  T/F plot copy constructor
TFPlot::TFPlot(const TFPlot& t) 
  : TrigID(t), mName(t.mName), mChannel(t.mChannel), mFilter(0),
    mBakOff1(t.mBakOff1), mNBSlice1(t.mNBSlice1), mBakOff2(t.mBakOff2),
    mNBSlice2(t.mNBSlice2), mTBins(t.mTBins), mTmin(t.mTmin), 
    mTstep(t.mTstep), mFBins(t.mFBins), mFmin(t.mFmin), mFstep(t.mFstep),
    mSigCount(t.mSigCount), mSigHist(t.mSigHist), mBakSpect(t.mBakSpect),
    mSNValid(t.mSNValid), mSNHist(t.mSNHist), mBakDefault(t.mBakDefault),
    mALimit(t.mALimit)
{
    setFilter(t.mFilter);
}

//======================================  T/F plot destructor
TFPlot::~TFPlot(void) { 
    setFilter(0);
}

//======================================  T/F plot destructor
void
TFPlot::stats(ostream& out) const { 
    out << getName() << "  Trigger: " << getID() << "  SubID: " << getSubID()
	<< "  Channel: " << getChannel() << endl;
    out << "        nTBin: " << mTBins << "  tMin: " << mTmin
	<< "  tMax: " << mTmin + double(mTBins)*mTstep << endl;
    out << "        nFBin: " << mFBins << "  fMin: " << mFmin
	<< "  fMax: " << mFmin + mFBins*mFstep << endl;
    out << "        Triggers: " << mSigCount << "  Background Slices: "
	<< mBakSpect.getCount() << endl;
}

//======================================  Filter data as appropriate
TSeries 
TFPlot::Filter(const TSeries& t) {
    if (mFilter) return (*mFilter)(t);
    return t;
}

//======================================  Calculate signal/noise Histogram
Interval 
TFPlot::getEpoch(void) const {
    Interval r = mBakOff1 + double(mNBSlice1) * mTstep;
    Interval t = mBakOff2 + double(mNBSlice2) * mTstep;
    if (t > r) r = t;
    t = mTmin + double(mTBins) * mTstep;
    if (t > r) r = t;
    return r - getOffset();
}

//======================================  Calculate signal/noise Histogram
Interval 
TFPlot::getOffset(void) const {
    Interval r = mBakOff1;
    if (mBakOff2 < r) r = mBakOff2;
    if (mTmin < r)    r = mTmin;
    return r;
}

//======================================  Calculate signal/noise Histogram
const Histogram2& 
TFPlot::getSN(void) {

    //----------------------------------  recalculate constens if necessary
    if (mSNValid) {
    } else if (!mBakSpect.getCount() || !mSigCount) {
        mSNHist.Clear(); // throw exception?
    } else {
        mSNHist = mSigHist;
	//	mSigHist.Dump(cout);
        double norm = mBakSpect.getCount()/mSigCount;
	for (int j=0 ; j<mFBins ; j++) {
	    double fj = mFmin + j*mFstep;
	    double BackPD = mBakSpect.getSum(fj, mFstep);
	    for(int i=0 ; i<mTBins ; i++){
		double sigAv = mSigHist.GetBinContent(i+1, j+1);
		mSNHist.SetBinContent(i+1, j+1, sigAv*norm/BackPD);
	    }
	}
	mSNValid = true;
    }
    // mSNHist.Dump(cout);
    return mSNHist;
}

//======================================  Set the channel name
bool 
TFPlot::isTrig(const trig::TrigBase& t) {
    if (!TrigID::isTrig(t)) return false;
    double ampl = t.getIntensity();
    return mALimit.inLimits(ampl);
}

//======================================  Set the amplitude limits
void 
TFPlot::setALimits(double aMin, double aMax) {
    mALimit.setLimits(aMin, aMax);
    mALimit.setHisto(50, 0.0, 1000.0);
}

//======================================  Set the amplitude histogram limits
void 
TFPlot::setAHisto(double hMin, double hMax) {
    mALimit.setHisto(50, hMin, hMax);
}

//======================================  Set the channel name
void 
TFPlot::setBSlice(int nb1, Interval t1, int nb2, Interval t2) {
    mNBSlice1   = nb1;
    mBakOff1    = t1;
    mNBSlice2   = nb2;
    mBakOff2    = t2;
    mBakDefault = false;
}

//======================================  Set the channel name
void 
TFPlot::setChannel(const string& d) {
    mChannel = d;
}

//======================================  Set time axis
void 
TFPlot::setFilter(const Pipe* p) {
    if (mFilter) {
        delete mFilter;
	mFilter = 0;
    }
    if (p) mFilter = p->clone();
}

//======================================  Set time axis
void 
TFPlot::setTAxis(int nbin, Interval tLo, Interval tHi) {
    mTBins = nbin;
    mTmin  = tLo;
    mTstep = (tHi - tLo)/double(nbin);

    //----------------------------------  Set default background positions
    if (mBakDefault) {
        mNBSlice1 = mNBSlice2 = nbin/2;
	mBakOff1  = tLo - mTstep*double(nbin);
	mBakOff2  = tLo + mTstep*double(nbin*3/2);
    }
}

//======================================  Set frequency axis
void 
TFPlot::setFAxis(int nbin, double fLo, double fHi) {
    mFBins = nbin;
    mFmin  = fLo;
    mFstep = (fHi - fLo)/nbin;
}

//======================================  Bump data in the signal plot
void
TFPlot::addSigSpec(const TSeries& ts) {

    //----------------------------------  Check arguments
    if (ts.getInterval() < double(mTBins) * mTstep) {
        cout << "Signal series not long enough, size=" << ts.getInterval() 
	     << " expected=" << double(mTBins) * mTstep << endl;
	return;
    }

    //----------------------------------  See if histo is set up
    mSNValid = false;
    if (mSigHist.GetBinType() == Histogram2::kUndefinedBin) {
        mSigHist.SetBinLowEdges(mTBins, mTmin, mTmin+double(mTBins)*mTstep, 
				mFBins, mFmin, mFmin+double(mFBins)*mFstep);
	cout << "Signal Histo reset!" << endl;
    }

    //----------------------------------  Fill data
    Time tStart = ts.getStartTime();
    for (int i=0 ; i<mTBins ; i++) {
        Interval  tOff = mTstep * double(i);
        FSpectrum fs(FSeries(ts.extract(tStart + tOff, mTstep)));
	double x = double(mTmin + tOff);
	for (int j=0 ; j<mFBins ; j++) {
	    double f0 = mFmin + j*mFstep;
	    double w = fs.getSum(f0, mFstep);
	    mSigHist.Fill(x, f0, w);
	}
    }
    mSigCount++;
}

//======================================  Bump data in the background spectrum
void
TFPlot::addBakSpec(const TSeries& ts) {
    Time tStart = ts.getStartTime();
    int nBins = int(ts.getInterval() / mTstep);
    for (int i=0 ; i<nBins ; i++) {
        Interval  tOff = mTstep * double(i);
        mBakSpect += FSpectrum(FSeries(ts.extract(tStart + tOff, mTstep)));
    }
    mSNValid = false;
}

//======================================  Add data for one trigger
void 
TFPlot::addSlice(const Time& t0, const TSeries& ts) {

    const TSeries* tsFilt;
    if (mFilter) tsFilt = new TSeries(mFilter->apply(ts));
    else         tsFilt = &ts;

    addBakSpec(tsFilt->extract(t0+mBakOff1, double(mNBSlice1)*mTstep));
    addSigSpec(tsFilt->extract(t0+mTmin,       double(mTBins)*mTstep));
    addBakSpec(tsFilt->extract(t0+mBakOff2, double(mNBSlice2)*mTstep));

    if (mFilter) delete const_cast<TSeries*>(tsFilt);
}

//======================================  Monitor contructor
TrigSpec::TrigSpec(int argc, const char* argv[]) 
  : TrigEnv(argc, argv), mHistFile("TrigSpec.file")
{

    //----------------------------------  Parse command line
    bool syntax(false);
    const char* pConf="TrigSpec.conf";
    for (int i=1 ; i<argc ; i++) {
        string argi = argv[i];
	if (argi == "-conf") {
	    pConf = argv[++i];
	} else if (argi == "-hfile") {
	    mHistFile = argv[++i];
	} else if (isTrigEnvArg(argv[i])) {
	    i++;
	} else {
	    syntax=true;
	    cerr << "Unidentified argument " << argi << endl;
	    break;
	}
    }

    //----------------------------------  Tell user if syntax is bad.
    if (syntax) {
        cerr << "Error found in command line. The command syntax is:" << endl;
	cerr << "TrigSpec [-conf <file>] [-hfile <root-file>]" << endl;
	finish();
	return;
    }

    //----------------------------------  Read configuration file.
    readConfig(pConf);
}

//======================================  Make a root histogram.
static TH2D*
makeTH2D(const char* name, const Histogram2& h) {
    int    xN  = h.GetNBins(0);
    int    yN  = h.GetNBins(1);
    double xLo = h.GetBinLowEdge(0, 0);
    double xHi = xLo + xN * h.GetBinSpacing(0);
    double yLo = h.GetBinLowEdge(0, 1);
    double yHi = yLo + yN * h.GetBinSpacing(1);
    TH2D* th = new TH2D(name, h.GetTitle(), xN, xLo, xHi, yN, yLo, yHi);
    if (xN && yN) {
        for (int i=0 ; i<xN+2 ; i++) {
	    for (int j=0 ; j < yN+2 ; j++) {
	        th->SetBinContent(i, j, h.GetBinContent(i,j));
	    }
	}
	th->SetEntries(h.GetNEntries());
    }
    return th;
}

static TH1D*
makeTH1D(const char* name, const Histogram1& h) {
    int    xN  = h.GetNBins();
    double xLo = h.GetBinLowEdge(0);
    double xHi = xLo + xN * h.GetBinSpacing();
    TH1D* th = new TH1D(name, h.GetTitle(), xN, xLo, xHi);
    if (xN) {
        for (int i=0 ; i<xN+2 ; i++) {
	    th->SetBinContent(i, h.GetBinContent(i));
	}
	th->SetEntries(h.GetNEntries());
    }
    return th;
}

//======================================  TrigSpec monitor destructor
TrigSpec::~TrigSpec(void) {

    //----------------------------------  Open a root file.
    TFile f(mHistFile.c_str(), "RECREATE", "Trigger based TF Plots");

    //----------------------------------  Loop over tools, save the histograms
    for (trig_iter i=mList.begin() ; i != mList.end() ; i++) {
        string name(i->getName());
        cout << "Writing " << name << "...";

	//------------------------------  Write signal / noise
	TH2D* th2 = makeTH2D(name.c_str(), i->getSN());
	if (th2) th2->Write(name.c_str());

	//------------------------------  Write amplitude histo
	name += "_Ampl";
	TH1D* th1 = makeTH1D(name.c_str(), i->getAHist());
	if (th1) th1->Write(name.c_str());
	cout << endl;
    }
    f.Close();

    //----------------------------------  Print stats
    for (trig_iter i=mList.begin() ; i != mList.end() ; i++) i->stats(cout);

    //----------------------------------  Delete the list
    mList.clear();
}

//======================================  Process data from one trigger
void 
TrigSpec::ProcessData(void) {

    //----------------------------------  Get the channel.
    const TSeries* tsRaw = getDacc().refData(mCurTrig->getChannel());
    if (!tsRaw) {
        cout << "Raw data channel: " << mCurTrig->getChannel() 
	     << " is not in the input file" << endl;
	return;
    }
    if (tsRaw->getInterval() < mCurTrig->getEpoch()) {
        cout << "Epoch not available for trigger" << endl;
        return;
    }
    mCurTrig->addSlice(getTrigger().getTime(), *tsRaw);
}

//========================================  Get trigger information
bool
TrigSpec::ProcessTrigger(const trig::TrigBase& t) {
    for (trig_iter i=mList.begin() ; i != mList.end() ; i++) {
        if (i->isTrig(t)) {
	    Interval tEpoch = i->getEpoch();
	    Interval tOff = i->getOffset();

	    //----------------------------  Check veto times.
	    Time tStart = t.getTime() + tOff;
	    Time tStop  = tStart + tEpoch;
	    int nOmit = mOmit.size();
	    for (int j=0 ; j<nOmit ; j++) {
	        if (mOmit[j] >= tStart && mOmit[j] <= tStop) return false;
	    }

	    //----------------------------  Set up time epoch and channel
	    setEpochLength(tEpoch);
	    setStride(tEpoch);
	    setEpochOffset(tOff);
	    if (mSvChannel.empty()) {
	        mSvChannel = i->getChannel();
		getDacc().addChannel(mSvChannel.c_str());
	    } else if (mSvChannel != i->getChannel()) {
	        getDacc().rmChannel(mSvChannel.c_str());
	        mSvChannel = i->getChannel();
		getDacc().addChannel(mSvChannel.c_str());
	    }
	    if (Debug()) cout << "Process data, Channel: " << mSvChannel
			      << " Offset: " << tOff << " Epoch: " << tEpoch
			      << endl;
	    mCurTrig = i;
	    return true;
	}
    }
    return false;
}

//========================================  Read configuration file
void
TrigSpec::readConfig(const char* f) {

    //------------------------------------  Open the configuration file.
    if (Debug()) cout << "Reading configuration file: " << f << endl;
    ParseLine in(f);
    if (!in.isOpen()) {
        cout << "Unable to open configuration file: " << f << endl;
	finish();
	return;
    }
    in.setLog(cout);

    //----------------------------------  Loop over configuration commands.
    while (in.getLine() >= 0) {
        int nArg = in.getCount();
	if (!nArg) continue;
	string cmd = in[0];

	//------------------------------  Specify Lock detection.
	if (cmd == "Lock") {

	//------------------------------  Omit a time
	} else if (cmd == "Omit") {
	    long osec(0), nsec(0);
	    if (nArg < 2) {
	        cout << "Insufficient arguments on line" << endl;
		continue;
	    }
	    osec = in.getInt(1);
	    if (nArg >= 3) nsec = in.getInt(2);
	    mOmit.push_back(Time(osec, nsec));

	//--------------------------  Specify a trigger plot
	} else if (cmd == "TrigPlot") {
	    if (nArg < 4) {
	        cout << "Insufficient arguments on line" << endl;
		continue;
	    }

	    if (Debug()) cout << "TFPlot: " << in[1] << " Trigger ID: " 
			      << in[2] << " Sub-ID: " << in[3] << endl;

	    TFPlot tf(in[1], in[2], in[3]);

	    double temp[4];
	    for (int i=4 ; i<nArg ; i++) {
	        string argi(in[i]);
		if (argi == "-ahist") {
		    const char* pAhs = in[++i];
		    for (int j=0 ; j<2 ; j++) {
		        temp[j] = strtod(pAhs, const_cast<char**>(&pAhs));
		    }
		    tf.setAHisto(temp[0], temp[1]);
		} else if (argi == "-alims") {
		    const char* pAlm = in[++i];
		    for (int j=0 ; j<2 ; j++) {
		        temp[j] = strtod(pAlm, const_cast<char**>(&pAlm));
		    }
		    tf.setALimits(temp[0], temp[1]);
		} else if (argi == "-channel") {
		    tf.setChannel(in[++i]);
		} else if (argi == "-taxis") {
		    const char* pTax = in[++i];
		    for (int j=0 ; j<3 ; j++) {
		        temp[j] = strtod(pTax, const_cast<char**>(&pTax));
		    }
		    tf.setTAxis(int(temp[0]), temp[1], temp[2]);
		} else if (argi == "-faxis") {
		    const char* pFax = in[++i];
		    for (int j=0 ; j<3 ; j++) {
		        temp[j] = strtod(pFax, const_cast<char**>(&pFax));
		    }
		    tf.setFAxis(int(temp[0]), temp[1], temp[2]);
		} else if (argi == "-bkgnd") {
		    int nBarg(0);
		    for (const char* p=in[++i] ; *p && nBarg<4 ; ) {
		        temp[nBarg++] = strtod(p, const_cast<char**>(&p));
		    }
		    if (nBarg == 2) {
		        mList.back().setBSlice(int(temp[0]), Interval(temp[1]));
		    } else {
		        mList.back().setBSlice(int(temp[0]), Interval(temp[1]),
					       int(temp[2]), Interval(temp[3]));
		    }
		} else {
		    cout << "Indecipherable argument: " << argi << endl;
		    break;
		}
	    }
	    mList.push_back(tf);
	} else {
	    cout << "Unidentified command: " << cmd << endl;
	    finish();
	    break;
	}
    }
}
