//
//    Skeleton Frame processing routine Template
//
#include "SpectrumFold.hh"
#include "FilterDesign.hh"
#include "FSpectrum.hh"
#include "Hanning.hh"

#ifndef __CINT__

//-->  The next three lines are needed if you are going to generate triggers.
//     The text in PIDTITLE should describe the monitor function.
#define PIDCVSHDR "$Header: https://redoubt.ligo-wa.caltech.edu/svn/gds/trunk/Monitors/SpectrumFold/SpectrumFold.cc 7987 2018-01-11 08:18:07Z john.zweizig@LIGO.ORG $"
#define PIDTITLE  "SpectrumFold"
#include "ProcIdent.hh"

#include "Dacc.hh"
#include <iostream>
#include <string>
#include <stdlib.h>
#include <math.h>

typedef int (*compfn)(const void *, const void *);

using namespace std;

//======================================  Generate the main routine.
EXECDAT(SpectrumFold)
#endif               // !def(__CINT__)

BIN_DATA::BIN_DATA(void)
      	: name(""), mHistory(43200)
{
}

CHAN_DATA::CHAN_DATA(void)
  : name(""), mFilter(""), mPipe(0), SFT_level(), veto2(), count()
{
}

CHAN_DATA&
CHAN_DATA::operator=(const CHAN_DATA& x) {
  name        = x.name;
  mFilter     = x.mFilter;
  if (x.mPipe) mPipe = x.mPipe->clone();
  f_start     = x.f_start;
  f_stop      = x.f_stop;
  fold_length = x.fold_length;
  bins        = x.bins;
  SFT_level   = x.SFT_level;
  veto2       = x.veto2;
  count       = x.count;
   return *this;
};

CHAN_DATA::CHAN_DATA(const CHAN_DATA& x) {
  *this = x;
}

void
SpectrumFold::add_channel(const string& chan_name, float f_start, float f_stop,
			  float fold_length, const string& mFilter) {
  std::string mChannel = chan_name;
  if (mChannel.empty()) return;
  std::string IFO = mChannel.substr(0, 2);
  std::string chName = mChannel.substr(3, mChannel.size()-3);
  for(string::size_type i=0;i<chName.size();i++) {
    if (chName[i]=='-') chName[i]='_';
  }
  CHAN_DATA chan_temp;
  /*chan_temp.mStep=mStep;*/
  chan_temp.mPipe=NULL;
  chan_temp.f_start=f_start;
  chan_temp.f_stop=f_stop;
  chan_temp.fold_length=fold_length;
  chan_temp.name=mChannel;
  chan_temp.mFilter=mFilter;
  chan_temp.SFT_level.name=IFO+":DMT-SPCF_" + chName + "_SFT_LEVEL";
  chan_temp.SFT_level.mHistory=FixedLenTS(43200);
  chan_temp.veto2.name=IFO+":DMT-SPCF_" + chName + "_VETO2";
  chan_temp.veto2.mHistory=FixedLenTS(43200);
  chan_temp.count.name=IFO+":DMT-SPCF_" + chName + "_COUNT";
  chan_temp.count.mHistory=FixedLenTS(43200);
  size_t binsz = ceil(fold_length*mStep.GetSecs());
  chan_temp.bins.resize(binsz);
  for(size_t k=0; k<binsz; k++){
    char s[20];
    sprintf(s, "%d", int(k));
    chan_temp.bins[k].name=IFO+":DMT-SPCF_" + chName + "_BIN" + s;
    chan_temp.bins[k].mHistory=FixedLenTS(43200);
  }
  channels.push_back(chan_temp);
}

//======================================  Skeleton object constructor.
SpectrumFold::SpectrumFold(int argc, const char *argv[]) 
  : DatEnv(argc, argv), mStep(60.0)
{
    //----------------------------------  Look for arguments
    bool dvIndex = false;
    float     fold_length = 0.25, f_start = 100.0, f_stop = 1200.0;
    std::string mFilter;

    cerr << "Initializing SpectrumFold" << endl;
    
    debug=0;
    
    bool syntax  = false;
    for (int i=1 ; i<argc ; i++) {
        string argi = argv[i];
        if (argi == "-channel") {
	  add_channel(argv[++i], f_start, f_stop, fold_length, mFilter);
	} else if (argi == "-dvindex") {
	    dvIndex = true;
	} else if (argi == "-sdebug") {
	    debug = 1;
	} else if (argi == "-filter") {
	    mFilter = argv[++i];
	} else if (argi == "-stride") {
	    mStep = strtod(argv[++i], 0);
	} else if (argi == "-f-start") {
	    f_start = strtod(argv[++i], 0);
	} else if (argi == "-f-stop") {
	    f_stop = strtod(argv[++i], 0);
	} else if (argi == "-fold-length") {
	    fold_length = strtod(argv[++i], 0);
 	} else if (isDatEnvArg(argv[i])) {
	    i++;
	} else {
	    syntax = true;
	    cerr << "Unrecognized argument: " << argi << endl;
	}
    }

    //----------------------------------  Bail out if command line not correct
    if (syntax) {
        cerr << "Command syntax:" << endl;
	cerr << argv[0] 
	     << " [-frames <nMax>] [-stride <time>] \\" 
	     << endl
	     << "      [-filter <filter>] [-dvIndex] \\"
	     << endl
	     << "      [-f-start <freq>] [-f-stop <freq>] [-fold-length <freq>] [-channel <name>] \\"
	     << endl
	     << "      [<DatEnv-args>]" 
	     << endl;
	finish();
	return;
    }

    //----------------------------------  Make sure there are some channels
    if(channels.empty()) {
      add_channel("H0:PEM-PSL1_MIC", f_start, f_stop, fold_length, mFilter);
      add_channel("H1:LSC-AS_Q", f_start, f_stop, fold_length, mFilter);
    }

    std::string str;
    
    str="SPECTRUM_FOLD";
    setServerName(str.c_str());

    //----------------------------------  Set up the trender
    mTrend.setName("SPECTRUM_FOLD");    //  Set trend frame name
    mTrend.setFrameCount(1);          //  Set frames per file
    mTrend.setType(Trend::kMinute);   //  Set trend type (minute trends)
    mTrend.setAutoUpdate(false);      //  Avoid automatic screwup.
    
    //----------------------------------  set the data accessor
    getDacc().setStride(mStep);       // in before calling ProcessData
    for(size_t i=0; i<channels.size(); i++) {
        cerr << "Hooking up channel " << channels[i].name << endl;
	getDacc().addChannel(channels[i].name.c_str());
	    
	serveData(channels[i].SFT_level.name.c_str(),
		  &channels[i].SFT_level.mHistory, 0);
	mTrend.addChannel(channels[i].SFT_level.name.c_str());
	    
	serveData(channels[i].veto2.name.c_str(), &channels[i].veto2.mHistory, 0);
	mTrend.addChannel(channels[i].veto2.name.c_str());
	    
	serveData(channels[i].count.name.c_str(), &channels[i].count.mHistory, 0);
	mTrend.addChannel(channels[i].count.name.c_str());

	for(size_t k=0; k<channels[i].bins.size(); k++) {
	    serveData(channels[i].bins[k].name.c_str(),
		      &channels[i].bins[k].mHistory, 0);
	    mTrend.addChannel(channels[i].bins[k].name.c_str());
	}
    }

    //----------------------------------  Specify DMT Viewer channel names
//    char username[32];
//    cuserid(username);
        

    //----------------------------------  Optional dataviewer index
    if (dvIndex) {
        string filename;
	const char* dmtout = getenv("DMTOUTPUT");
	if (dmtout) {
	    filename = dmtout;
	    filename += "/";
	}
	filename += "channel.cfg";
	mTrend.writeIndex(filename.c_str());
    }
    
    cerr << "Waiting for data" << endl;
}

//======================================  Skeleton object destructor.
SpectrumFold::~SpectrumFold() 
{
    cout << "SpectrumFold is finished" << endl;
}

static int float_cmp(float *a, float *b)
{
if(*a<*b)return -1;
if(*a>*b)return 1;
return 0;
}

#if 0
static int 
double_cmp(double *a, double *b){
  if(*a<*b)return -1;
  if(*a>*b)return 1;
  return 0;
}
#endif

//======================================  Frame processing function.
void
SpectrumFold::ProcessData(void) 
{
  for(int i=0; i<channels.size(); i++) ProcessChannelData(i);
}

void
SpectrumFold::ProcessChannelData(int ch) {
    //----------------------------------  Get pointers to the current data.
    const TSeries* ts = getDacc().refData(channels[ch].name.c_str());
    if(debug)cerr << "At start of ProcessChannelData, channel=" << channels[ch].name << endl;
    if (!ts || ts->isEmpty()) {
        cerr << "Channel: " << channels[ch].name << " was not found." << endl;
        return;
    }

    //----------------------------------  Construct the filter if it isn't 
    //                                    already there
    if (!channels[ch].mPipe && !channels[ch].mFilter.empty()) {
        double samplerate(1./double(ts->getTStep()));
	FilterDesign fd(channels[ch].mFilter.c_str(), samplerate);
	channels[ch].mPipe = fd.release();
    }

    //------ Filter the data and apply Hanning window if necessary.
    TSeries tf;
    {
	TSeries tf2;
	
    	if (channels[ch].mPipe) tf2 = channels[ch].mPipe->apply(*ts);
    	else       tf2 = *ts;

	Hanning hw(ts->getNSample());
	tf=hw.apply(tf2);	
	}

    FSpectrum fs(tf);
    float Excess, Reference;
    float *data=fs.refData();
    long binStop, binStart;

    long average_size=channels[ch].bins.size();
    double *average=new double[average_size];
    double total_weight=0.0;
    long veto1=0, veto2=0, count=0;
    
    for(long i=0;i<average_size;i++){
    	average[i]=0.0;
	}
	
    if(channels[ch].f_stop > fs.getHighFreq()){
    	channels[ch].f_stop = fs.getHighFreq()-1.0;
	fprintf(stderr, "Warning: f-stop is higher than Nyquist frequency (%f), setting to (%f)\n", fs.getHighFreq(), channels[ch].f_stop);
    	}
        
    if(channels[ch].f_start > channels[ch].f_stop){
    	fprintf(stderr, "f_start=%f is greater than f_stop=%f, aborting\n", channels[ch].f_start, channels[ch].f_stop);
	delete[] average;
	return;
    	}

    double rough_average_level=0.0, rough_weight=0.0;
	
    for(float f=channels[ch].f_start;f<channels[ch].f_stop;f+=channels[ch].fold_length){
    	/* ignore neighbourhoods of 60 Hz */
    	if(fmod(f+5.0, 60.0)<10.0)continue;
	binStart=int(fabs(f/fs.getFStep()));
	binStop=int(fabs((f+channels[ch].fold_length)/fs.getFStep()));
	
	for(int i=1;i<binStop-binStart;i++){
		rough_average_level+=data[i+binStart];
		rough_weight+=1.0;
	}	
    }	
    rough_average_level/=rough_weight;
		
    for(float f=channels[ch].f_start;f<channels[ch].f_stop;f+=channels[ch].fold_length){
    	/* ignore neighbourhoods of 60 Hz */
    	if(fmod(f+5.0, 60.0)<10.0){
		veto1++;
		continue;
		}
    
	binStart=int(fabs(f/fs.getFStep()));
	binStop=int(fabs((f+channels[ch].fold_length)/fs.getFStep()));
    
    	//cout << "data[binStart]=" << data[binStart] << endl;
	
	float *chunk=new float[binStop-binStart];
	
	//cout << "f=" << f << " binStop=" << binStop << " binStart="
	//     << binStart << endl;
	//cout << "binStop-binStart=" << binStop-binStart << " fold_length="
	//     << fold_length << " fStep=" << fs.getFStep() << endl;
	
	/* ignore regions with very high bins ... so they do not corrupt the results */
	for(int i=0;i<binStop-binStart;i++)
		if(data[i+binStart]>rough_average_level*20){
			veto2++;
			delete[] chunk; 
			chunk=NULL;
			break;
			}

	if (chunk==NULL) continue;	

	memcpy(chunk, &(data[binStart]), (binStop-binStart)*sizeof(*chunk));
	qsort(chunk, binStop-binStart, sizeof(*chunk), (compfn)float_cmp);
	
	if(chunk[(binStop-binStart)/2]==0.0){
		delete[] chunk;
		chunk=NULL;
		if (debug) cerr << "1.0/0.0 weight for f=" << f <<endl;
		continue;
		}
	
	float weight=1.0/chunk[(binStop-binStart)/2];
	
	delete[] chunk;
	chunk=NULL;
	
	for(int i=0;(i<binStop-binStart) && (i<average_size); i++){
		average[i]+=data[binStart+i]*weight;
		}
	total_weight+=weight;
	count++;
    	}
    
    if(total_weight==0.0){
    	delete[] average;
	return;
	}
	
    double mean=0.0;
    for(int i=0;i<average_size;i++){
    	average[i]/=total_weight;
	if((i>1) && (i<(average_size-1)))mean+=average[i];
	}
    mean/=average_size-3;
    
    int reference_bin=1+(average_size-3)/2;
    Excess=average[0]-mean;
    Reference=average[reference_bin]-mean;
    
    
    if(debug)cerr << "total_weight=" << total_weight << " veto1=" << veto1 << " veto2=" << veto2 << " count=" << count << endl;
    if(debug)cerr << "average: 0=" << average[0] << " " << reference_bin << "=" << average[reference_bin] << endl;
	
    //----------------------------------  Trend the data points
    Time t0 = tf.getStartTime();
    
    Reference=mean;
    mTrend.trendData(channels[ch].SFT_level.name.c_str(), t0, Reference);
    channels[ch].SFT_level.mHistory.fixedAppend(t0, tf.getInterval(), &Reference);

    Reference=veto2;
    mTrend.trendData(channels[ch].veto2.name.c_str(), t0, Reference);
    channels[ch].veto2.mHistory.fixedAppend(t0, tf.getInterval(), &Reference);
        
    Reference=count;
    mTrend.trendData(channels[ch].count.name.c_str(), t0, Reference);
    channels[ch].count.mHistory.fixedAppend(t0, tf.getInterval(), &Reference);
        
    for(int k=0;k<channels[ch].bins.size();k++){
    	    Excess=average[k]-mean;
	    mTrend.trendData(channels[ch].bins[k].name.c_str(), t0, Excess);
	    channels[ch].bins[k].mHistory.fixedAppend(t0, tf.getInterval(), &Excess);
    	    }
    mTrend.Update();

    //----------------------------------  set up data for display.
        
    delete[] average;
    if(debug)cerr << "At end of ProcessData" << endl;
}


//======================================  Handle Messages
void
SpectrumFold::Attention(void) {
    MonServer::Attention();
}
