//******************************************************************************
//
//  SpectrumArchiver    May 10, 2004
//
//    contact info:
//        tim_olson@skc.edu
//
//  SpectrumArchiver creates a power spectra archive once each hour. The list 
//  of channels to be archived is read from a configuration file named
//  LHOSpectrumArchiverConfig or LLOSpectrumArchiverConfig for LHO and LLO,
//  respectively. The archived spectra are averaged over 10
//  Hanning windowed time series segments with a 50% overlap between each segment.
//  Each channel with a sample rate slower than 1000 Hz is archived with a 0.1 Hz
//  bin spacing from DC to the Nyquist frequency. Two spectra are archived in two
//  separate files for each channel with a sample rate faster than 1000 Hz, one
//  spectrum with 0.1 Hz bin spacing from DC to 1000 Hz, and another spectrum with
//  1 Hz bins running from DC to the Nyquist frequency. The spectra are stored in
//  xml format as amplitude spectra (square root of the magnitude of the power
//  spectral density).
//
//  At LLO the spectra are archived from delaronde to
//
//                 /export/DMTtrends/SpectrumArchiver/
//
//  At LHO the spectra are archived from sand to
//
//                 /export/DMTtrends/spectrumArchiver/
//
//  The files are placed in directories corrsponding to the calendar year, month
//  day, UTC hour, interferometer code (L0, L1, H0, H1, or H2), channel type 
//  and channel name. Channels with sample rates faster than 1000 Hz include the
//  bin spacing used as part of the filename. For example,
//
//  channels with sample rates under 1000 Hz:
//    /export/DMTtrends/SpectrumArchiver/2003/May/28/14/L0/PEM/L0:PEM-EX_SEISX.xml
//
//  channels with sample rates over 1000 Hz:
//    /export/DMTtrends/SpectrumArchiver/2003/May/28/14/L1/LSC/L1:LSC-AS_Q:1Hz.xml
//    /export/DMTtrends/SpectrumArchiver/2003/May/28/14/L1/LSC/L1:LSC-AS_Q:0.1Hz.xml
//
//*********************************************************************************


#include "SpectrumArchiver.hh"
#include <cstring>
#include <cstdlib>

#ifndef __CINT__
  EXECDAT(SpectrumArchiver)
#endif

SpectrumArchiver::SpectrumArchiver(int argc, const char *argv[])
  : DatEnv(argc, argv)
{
  Configuration();

  setStrideAlignment(3600,0.);
//  setStrideAlignment(1,0.);
  getDacc().setStride(55.);
  getDacc().setIgnoreMissingChannel(true);
  for (int i=0; i < numberChannels; i++)
    getDacc().addChannel(channelNames[i]);
}

SpectrumArchiver::~SpectrumArchiver()
{
}

void SpectrumArchiver::ProcessData(void)
{
  currentTime = Now();
  TimeStr(currentTime,currentUTCTime,"%M:%d:%Y:%H:%N:%S");
  TimeStr(currentTime,currentGPSTime,"%s");
  TimeStr(currentTime,year,"%Y");
  TimeStr(currentTime,month,"%M");
  TimeStr(currentTime,day,"%d");
  TimeStr(currentTime,hour,"%H");

  strcpy(topLevelPath,archiveLocation);
  strcat(topLevelPath,"/");
  strcat(topLevelPath,year);
  mkdir(topLevelPath,0755);
  strcat(topLevelPath,"/");
  strcat(topLevelPath,month);
  mkdir(topLevelPath,0755);
  strcat(topLevelPath,"/");
  strcat(topLevelPath,day);
  mkdir(topLevelPath,0755);
  strcat(topLevelPath,"/");
  strcat(topLevelPath,hour);
  mkdir(topLevelPath,0755);
  strcat(topLevelPath,"/");

  for (int i=0; i < numberChannels; i++)
  {
    strcpy(currentPath,topLevelPath);
    strncpy(tempString,channelNames[i],2);
    tempString[2] = '\0';
    strcat(currentPath,tempString);
    mkdir(currentPath,0755);
    strcpy(currentPath,topLevelPath);
    strncpy(tempString,channelNames[i],6);
    tempString[2] = '/';
    tempString[6] = '\0';
    strcat(currentPath,tempString);
    mkdir(currentPath,0755);
    strcat(currentPath,"/");
    WriteXML(currentPath,channelNames,i);
  }
//  exit(0);
}

void SpectrumArchiver::Configuration(void)
{
  FILE* in;
  in = fopen("LLOSpectrumArchiverConfig","r");
  if (in == NULL)
  {
    in = fopen("LHOSpectrumArchiverConfig","r");
    if (in == NULL)
    {
      std::cout << "Missing configuration file, SpectrumArchiver is quitting ... \n";
      exit(0);
    }
  }
  fscanf(in,"%s",archiveLocation);
  fscanf(in,"%d",&numberChannels);
  channelNames = new char*[numberChannels];
  for (int i=0; i < numberChannels; i++) channelNames[i] = new char[40];
  for (int i=0; i < numberChannels; i++) fscanf(in,"%s",channelNames[i]);
  fclose(in);
}

void SpectrumArchiver::WriteXML(char *path, char **names, int channelIndex)
{
//  std::cout << "writing channel spectrum " << channelIndex << " = " << names[channelIndex] << "\n";
//  std::cout << "to path = " << path << "\n";

  fineSegmentLength = 10.;
  coarseSegmentLength = 1.;

  currentTimeSeries = getDacc().refData(names[channelIndex]);
  startTime = currentTimeSeries->getStartTime();
  if (!startTime)
  {
//    std::cout << names[channelIndex] << " was missing" << "\n";
    return;
  } 
  samplingRate = 1./(currentTimeSeries->getTStep());

// estimate the power spectrum for 0.1 Hz bin spacing up to no more than 1000 Hz
  fineWindowedTimeSeriesSegment = HanningWindow.apply(currentTimeSeries->extract(startTime, fineSegmentLength)); 
  finePowerSpectrum = FSpectrum(FSeries(fineWindowedTimeSeriesSegment));

  for (int k=0; k < 9; k++)
  {
    startTime += 5.;
    fineWindowedTimeSeriesSegment = HanningWindow.apply(currentTimeSeries->extract(startTime, fineSegmentLength)); 
    finePowerSpectrum += FSpectrum(FSeries(fineWindowedTimeSeriesSegment));
  }
  finePowerSpectrum *= 1./10;
  finePowerSpectrum = finePowerSpectrum.extract(0., 1000.);

  numberFrequencyPoints = finePowerSpectrum.getNStep() + 1;
  finePowerSpectrum.getData(numberFrequencyPoints,fineAmplitudeData);
  for (int k=0; k < numberFrequencyPoints;  k++) fineAmplitudeData[k] = pow(fineAmplitudeData[k], 0.5);
  finePowerSpectrum.setData(numberFrequencyPoints,fineAmplitudeData);

// estimate the power spectrum for fast channels for 1 Hz bin spacing up to the Nyquist frequency
  startTime = currentTimeSeries->getStartTime();
  if (samplingRate > 1000.)
  {
    coarseWindowedTimeSeriesSegment = HanningWindow.apply(currentTimeSeries->extract(startTime, coarseSegmentLength)); 
    coarsePowerSpectrum = FSpectrum(FSeries(coarseWindowedTimeSeriesSegment));

    for (int k=0; k < 9; k++)
    {
      startTime += 0.5;
      coarseWindowedTimeSeriesSegment = HanningWindow.apply(currentTimeSeries->extract(startTime, coarseSegmentLength)); 
      coarsePowerSpectrum += FSpectrum(FSeries(coarseWindowedTimeSeriesSegment));
    }
    coarsePowerSpectrum *= 1./10.;

    numberFrequencyPoints = coarsePowerSpectrum.getNStep() + 1;
    coarsePowerSpectrum.getData(numberFrequencyPoints,coarseAmplitudeData);
    for (int k=0; k < numberFrequencyPoints;  k++) coarseAmplitudeData[k] = pow(coarseAmplitudeData[k], 0.5);
    coarsePowerSpectrum.setData(numberFrequencyPoints,coarseAmplitudeData);
  }

// archive the 0.1 Hz spectrum
  strcpy(currentArchivePath,path);
  strcat(currentArchivePath,names[channelIndex]);
  strcpy(thisChannelName,names[channelIndex]);
  if (samplingRate > 1000.) strcat(currentArchivePath,":0.1Hz.xml");
  else strcat(currentArchivePath,".xml");
  std::ofstream outputFileStream(currentArchivePath);
  if (outputFileStream.good())
  {
    xsil::Xwriter writer(outputFileStream);
    xsil::ligolw document(strcat(thisChannelName,":0.1Hz"), "container");
    document.addObject(xsil::XSpectrum(thisChannelName, finePowerSpectrum));
    document.Spew(writer);
//     cout << names[channelIndex] << " 0.1 Hz resolution spectrum was archived successfully for GPS time " << startTime << endl;
  }
  else
  {
//    cout << "ERROR: file " << currentArchivePath << " failed to open" << endl;
  }

// archive the 1 Hz spectrum
  if (samplingRate > 1000.)
  {
    strcpy(currentArchivePath,path);
    strcat(currentArchivePath,names[channelIndex]);
    strcpy(thisChannelName,names[channelIndex]);
    strcat(currentArchivePath,":1Hz.xml");
    std::ofstream outputFileStream(currentArchivePath);
    if (outputFileStream.good())
    {
      xsil::Xwriter writer(outputFileStream);
      xsil::ligolw document(strcat(thisChannelName,":1Hz"), "container");
      document.addObject(xsil::XSpectrum(thisChannelName, coarsePowerSpectrum));
      document.Spew(writer);
//      cout << names[channelIndex] << " 0.1 Hz resolution spectrum was archived successfully for GPS time " << startTime << endl;
    }
    else
    {
//      cout << "ERROR: file " << currentArchivePath << " failed to open" << endl;
    }
  }
}
