/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef PSLGLITCH_HH
#define PSLGLITCH_HH

#include "PSLTool.hh"
#include "PSLOChan.hh"
#include "PSLfilter.hh"
#include "FixedLenTS.hh"
#include "Histogram1.hh"
#include "SegAccountant.hh"
#include <vector>

class ParseLine;
class Pipe;
class MonServer;

/*  PSLGlitch is used by the PSL monitor to count glitches in a specified
 *  channel. A glitch is defined to be an excursion from the mean value by 
 *  greater than a specified number of standard deviations or than a fixed  
 *  absolute value.
 *  @memo PSL glitch detector.
 *  @author  J. Zweizig.
 *  @version 1.3; modified May 16, 2006.
 */
class PSLGlitch : public PSLTool {
public:
    /**  Construct a glitch finder with a specified name and channel.
      *  @memo construc a glitch finder.
      *  @param name Glitch finder name.
      *  @param chan Pointer to the channel accessor.
      */
    PSLGlitch(const std::string& name, const PSLChan* chan);

    /** Destroy a glich finder.
      */
    ~PSLGlitch(void);

    /** Clone a glitch tool.
      */
    PSLGlitch* clone(void) const;

    /** Configure the segment.
      */
    void confSegment(trig::SegAccountant& sa);

    /**  Dump out the glitch tool configuration to the specified stream.
      */
    std::ostream& dumpConfig(std::ostream& out) const;

    /**  Get the alarm glitch rate. Any short term increase in the rate of 
      *  glitches above the specified alarm limit will cause an alarm to be
      *  generated.
      *  @memo Get alarm rate.
      *  @return Maximum glitch rate.
      */
    double getAlarmRate(void) const;

    /**  Get the maximum deviation (amplitude) of a specified glitch.
      *  @memo Get glitch amplitude.
      *  @param i Glitch number.
      *  @return Maximum amplitude of the specified glitch.
      */
    double getAmpl(unsigned int i) const;

    /**  Get the average amplitude of found glitches.
      *  @memo Get the average amplitude.
      *  @return Average amplitude.
      */
    double getAvgAmpl(void) const;

    /**  Get the average glitch rate.
      */
    double getAvgRate(void) const;

    /**  Get the average glitch rate.
      */
    double getAvgTime(void) const;

    /**  Get the duration of the specified glitch.
      */
    Interval getDuration(int inx) const;

    /**  Get the glitch count for the last frame.
      */
    int getCount(void) const;

    /**  Get the filter name
      */
    const char* getFilterName(void) const;

    /**  Get the fraction of triggers that follow the previous trigger 
      *  within 1 second. Note that this number should be ~1/rate.
      */
    double getFracLt1(void) const;

    /**  Get the total number of glitches found.
      */
    long getNTotal(void) const;

    /**  Get the significance of the ith glitch.
      */
    double getSigma(unsigned int i) const;

    /**  The glitch tool keeps a recent average rate for use in alarm testing.
      *  GetRecentRate returns the most recently calaulted rate value. This
      *  remains fixed for the length of the time Alarm interval.
      *  @memo Get the recent rate of glitches.
      */
    double getRecentRate(void) const;

    /**  Get the sigma threshold.
      *  Get the threshold setting in sigma.
      */
    double getSigmaT(void) const;

    /**  Get the size threshold.
      */
    double getSize(void) const;

    /**  Get the start time of the specified glitch from the last stride.
      *  @param inx Number of glitch.
      *  @return Start time of the specified glitch.
      */
    Time getStartTime(int inx) const;

    /**  Get the time of the largest sample in the specified glitch.
      */
    Time getTMax(unsigned int i) const;

    /**  Get the glitch trend channel name.
      */
    int getTrigger(void) const;

    /**  Get the glitch time series.
      */
    const TSeries& refGlitch(void) const;

    /**  Get the rate output channel.
      */
    const OutChan& refHistoryChan(void) const;
    OutChan&       refHistoryChan(void);

    /**  Get the rate history time series.
      */
    const TSeries& refHistory(void) const;
    const Histogram1& refSigHist(void) const;
    const Histogram1& refDurHist(void) const;
    const Histogram1& refDtHist(void) const;

    /**  Process one time interval of data. The bool return value is 
      *  use to indicate that one or more glitches were found.
      *  @memo Find glitches
      *  @return true If one or more glitches were found.
      */
    bool crunch(void);

    /** Perfoem post-configure processing.
      */
    void postConfig(MonServer& x);

    /** Print statistics to the specified output stream.
      */
    void printStats(std::ostream& o) const;

    /** Print a statistics header to the specified output stream.
      */
    void printStatHdr(std::ostream& o) const;

    /**  Reset the glitch count.
      */
    void reset(void);

    /**  Send trigger(s) to the trigger manager.
      */
    void sendTrigger(TrigClient& tc);

    /**  Send segment(s) to the segment accountant.
      */
    void sendSegment(trig::SegAccountant& sa);

    /**  Set the glitch rate alarm threshold and integration time. A glitch
      *  rate alarm is set if the rate of glitches in a specific interval 
      *  is greater than a specified interval. Setting the threshold to
      *  zero or negative, disables the alarm generation. The default 
      *  averaging interval is 60 seconds. If the interval is set to zero,
      *  the current interval is not modified.
      *  @memo Set glitch rate alarm parameters.
      */
    void setAlarmParams(double thresh, Interval dt=0.0);

    /**  Set the pre-processing filter. The filter passed to the glitch code
      *   is copied using the clone mechanism.
      */
    void setFilter(const PSLfilter& f);

    /**  Set the segment identification.
      */
    void setSegment(const std::string& seg);

    /**  Set the threshold glitch deviation in sigma.
      */
    void setSigmaT(double Sigma);

    /**  Set the minimum glitch deviation in raw data units.
      */
    void setSize(double Size);

    /**  Set estimated length of a glitch (snippet time). The snippet time is
      *  used as the minimum glitch duration and the time from the last sample 
      *  found to excede the sigma and the amplitude thresholds to the end of 
      *  the glitch. 
      *  @memo Set the snippet time.
      *  @param time snippet interval.
      */
    void setTime(Interval time);

    /**  Set the maximum number of triggers to generate per stride. If the
      *  number of triggers is set to zero, generation of triggers for this 
      *  channel will be disabled.
      *  @memo Set maximum number of triggers per stride.
      *  @param n Maximum number of triggers to generate per stride.
      */
    void setTrigger(int n);

    void statUpdate(const Time& t0, Interval dT);

    bool testAlarm(void) const;

public:
    struct SaveGlitch {
	SaveGlitch(const Time& t, Interval dt, double s, double a)
	    : _time(t), _dt(dt), _peak(0.5*dt), _sigma(s), _ampl(a) {}
	Time     _time;
	Interval _dt;
	Interval _peak;
	double   _sigma;
	double   _ampl;
    };

    const SaveGlitch& operator[](int i) const;

private:
    typedef std::vector<SaveGlitch> GlitchVector;
    void saveIt(const Time& start, const Time& stop, const Time& peak, 
		double sig, double A, const TSeries* ts);

private:
    //------------------------------------  Parameters
    /// Threshold value
    double   mSigma;
    double   mSize;

    /// Snippet size
    Interval mTime;

    /// Preprocessing filter.
    PSLfilter    mFilter;

    /// Generate mTrigger triggers per stride.
    int          mTrigger;

    /// Segment ID
    trig::SegAccountant::seg_id mSegID;

    //------------------------------------  Status
    GlitchVector mList;
    TSeries  mSvLast;
    Time     mStartTime;
    Time     mLastTime;
    Time     mLastTrig;

    //------------------------------------  Rate calculation and alarm
    OutChan  mRateHstry;
    int      mRecentGlitches;
    Interval mRecentTime;
    double   mRecentRate;
    bool     mAlarmSet;
    double   mAlarmThresh;
    Interval mAlarmTime;

    //----------------------------  Statistics measures
    long     mNTotal;   /// Total number of glitches detected
    Interval mObserved; /// Total time searched for glitches
    double   mSumAmpl;  /// Sum of amplitudes (for average ampl)
    double   mSumTime;  /// Sum of trigger durations (for average duration)
    long     mDtLt1;    /// Number of triggers where delta-t < 1 second.

    //----------------------------  Histograms
    Histogram1 mSigHist;
    Histogram1 mDurHist;
    Histogram1 mDtHist;
};

#ifndef __CINT__
//======================================  Glitch inline functions.

inline double 
PSLGlitch::getAlarmRate(void) const {
    return mAlarmThresh;
}

inline int
PSLGlitch::getCount(void) const {
    return mList.size();
}

inline double 
PSLGlitch::getAmpl(unsigned int inx) const {
    if (inx >= mList.size()) return 0.0;
    return mList[inx]._ampl;
}

inline double 
PSLGlitch::getAvgRate(void) const {
    if (double(mObserved) == 0.0) return 0.0;
    return double(mNTotal)/double(mObserved);
}

inline double 
PSLGlitch::getAvgAmpl(void) const {
    if (mNTotal == 0) return 0.0;
    return double(mSumAmpl)/double(mNTotal);
}

inline double 
PSLGlitch::getAvgTime(void) const {
    if (mNTotal == 0) return 0.0;
    return double(mSumTime)/double(mNTotal);
}

inline Interval 
PSLGlitch::getDuration(int inx) const {
    if (inx < 0 || inx >= getCount()) return Interval(0.0);
    return mList[inx]._dt;
}

inline const char*
PSLGlitch::getFilterName(void) const {
    return mFilter.getName();
}

inline double
PSLGlitch::getFracLt1(void) const {
    if (!mNTotal) return 0.0;
    else          return double(mDtLt1)/mNTotal;
}

inline long 
PSLGlitch::getNTotal(void) const {
    return mNTotal;
}

inline double
PSLGlitch::getRecentRate(void) const {
    return mRecentRate;
}

inline double 
PSLGlitch::getSigmaT(void) const {
    return mSigma;
}

inline double 
PSLGlitch::getSize(void) const {
    return mSize;
}

inline Time
PSLGlitch::getTMax(unsigned int inx) const {
    if (inx >= mList.size()) return Time(0);
    return mList[inx]._time + mList[inx]._peak;
}

inline int
PSLGlitch::getTrigger(void) const {
    return mTrigger;
}

inline void
PSLGlitch::setTrigger(int trig) {
    mTrigger = trig;
}

inline const TSeries&
PSLGlitch::refGlitch(void) const {
    return mSvLast;
}

inline const Histogram1& 
PSLGlitch::refSigHist(void) const {
    return mSigHist;
}

inline const Histogram1& 
PSLGlitch::refDurHist(void) const {
    return mDurHist;
}

inline const Histogram1& 
PSLGlitch::refDtHist(void) const {
    return mDtHist;
}


inline const PSLGlitch::SaveGlitch& 
PSLGlitch::operator[](int i) const {
    return mList[i];
}

inline const TSeries&
PSLGlitch::refHistory(void) const {
    return mRateHstry.refTSeries();
}

inline const OutChan&
PSLGlitch::refHistoryChan(void) const {
    return mRateHstry;
}

inline OutChan&
PSLGlitch::refHistoryChan(void) {
    return mRateHstry;
}

inline bool
PSLGlitch::testAlarm(void) const {
    return mAlarmSet;
}
#endif  // !defined(__CINT__)

#endif  // !defined(PSLGLITCH_HH)
