/* -*- mode: c++; c-basic-offset: 3; -*- */
#ifndef WPIPE_WSTRCHAN_HH
#define WPIPE_WSTRCHAN_HH

#include "wtypes.hh"
#include "weventlist.hh"
#include "wparameters.hh"
#include "wtile.hh"
#include "wresample.hh"
#include "thread_base.hh"
#include <vector>

class MultiDacc;
class OperStateCondList;

namespace wpipe {
   class woutput;

   /**  The %wstream_chan class defines a processes a group of channels that
     *  are to be processed in an identical manner and with a common tiling. 
     *  Any channels that are to be analysed coherently (not implemented yet)
     *  must be in the same group.
     */
   class wstream_chan : public thread::thread_base
   {
   public:
      /**  Construct a %wstream_chan instance. The channel names, analysis 
        *  parameters, etc. are specified in the wparameters instance passed
	*  to the constructor.
	*  \param par Channel names, sources and analysis parameters for 
	*             this channel group.
	*  \param debugLevel Debugging printout level.
	*/
      wstream_chan(const wparameters& par, int dbugLvl);

      /**  Destroy the wstream channel group descriptor.
 	*/ 
      ~wstream_chan(void);
      void addChannels(MultiDacc& md);
      void addOutputFiles(woutput& wo, const std::string& fileTime, 
			  const std::string& outdir) const;

      /**  Book a channel in the appropriate %MultiDacc stream. THe frame type 
        *  argument is used to select an input stream. I there is only only one
	*  stream the channel will be book to that stream regardless of the 
	*  stream name or frame type. If there is more than one stream, the 
	*  site name and frame type (assuming a name of the form 
	*  \c \<dir>/\<site>-\<type>-\* ) of the first file in the stream 
	*  input index is compared to \a frameType. If they are the same, 
	*  the channel is booked in that steam. If no streams math the 
	*  \a frameType argument, the first stream is used. 
        *  \brief Book a %MultiDacc input channel.
        *  \param md        MultiDacc input device.
	*  \param channel   chanel name.
	*  \param frameType frame type string.
	*/
      void bookInput(MultiDacc& md, const std::string& channel,
		     const std::string& frameType) const;

      void inject(MultiDacc& md);
      double minimumOverlap(void) const;
      size_t numberOfChannels(void) const;
      void process(const woutput& wo, const Time& blockStartTime, 
		   Interval blockDuration, Interval blockOverlap);

      /**  Read in state data for times from start - start+dT and select 
        *  channels that should have good data. Read data from the good 
	*  channels for the specified times. On return, validChannels 
	*  indicates channels that pass the state criteria and input data 
	*  for the valid channels are available in input_data[].
	*/
      void read_data(MultiDacc& md, const Time& start, double dT);

      /**  Perform threaded processing.
	*/ 
      void* thread_entry(void);

      const wparameters& refParameters(void) const;
      const wtile& refTiling(void) const;
      void reset(void);
      void start_chunk(void);
      weventstack& trigClusters(void);
      weventstack& trigDownselect(void);
      void validChans(str_vect& processedChannels) const;

   private:
      wparameters pars;
      wtile       tiling;
      int         debugLevel;
      double      blockOverlap;
      double      minimumTimeShift;
      double      maximumTimeShift;
      resampler   resample;

      tser_vect   input_data;
      tser_vect   state_data;
      
      /**  The state_index vector indicates which states need to be evaluated 
        *  independently. If state_index < 0, the state is always true, \e i.e.
	*  the state channel name was empty or "NONE" or the mask was zero.
	*  If \c state_index>=0 it indicates the first channel with an identical
	*  state. The state need be evaluated only for states where 
	*  state_index[i] == i
	*/ 
      std::vector<int> state_index;


      /**  OSC condition test list.
       */
      OperStateCondList* oscList;
      std::vector<int> osc_index;
      tser_vect   osc_data;
      
      bool_vect   validChannels;
      weventstack triggersDownselected;
      weventstack clusters;
#ifndef NO_COINCIDENCE
      weventstack event;
#endif
   };

   typedef std::vector<wstream_chan> strchan_vect;

   //===================================  Inline methods
   inline size_t
   wstream_chan::numberOfChannels(void) const {
      return pars.numberOfChannels();
   }

   inline const wparameters& 
   wstream_chan::refParameters(void) const {
      return pars;
   }

   inline const wtile& 
   wstream_chan::refTiling(void) const {
      return tiling;
   }

   inline weventstack& 
   wstream_chan::trigClusters(void) {
      return clusters;
   }

   inline weventstack& 
   wstream_chan::trigDownselect(void) {
      return triggersDownselected;
   }
}
#endif // !defined(WPIPE_WSTRCHAN_HH)
