/* -*- mode: c++; c-basic-offset: 3; -*- */
// WSTREAM Top level function for the Omega_C stream search
//
// WSTREAM applies the discrete Q transform to search for statistically
// significant transient events in data from interferometric
// gravitational-wave detectors. The rstreaming search mode reads fixed 
// length strides from an infinite data stream.
//
// usage: wstream(parameterFile, frameCacheFile, outputDirectory, debugLevel)
//
//   parameterFile       parameter file
//   frameCacheFile      readframedata formatted frame cache file
//   outputDirectory     directory to write results
//   debugLevel          verboseness level of debug output
//
// If no output directory is specified, WSTREAM places the resulting trigger
// and event files in a subdirectory of the current directory, named after 
// the specified segment:
//
//   segments/
//     <startTime>-<stopTime>/
//       livetime.txt
//       <channelName1>.txt
//       <channelName2>.txt
//       
//
// All output is written to the specified output directory (outputDirectory 
// parameter).
//
// If no parameter file or frame cache file is specified, WSTREAM looks
// for the files parameters.txt or framecache.txt in the current directory.
//
// The specified debugLevel controls the amount of detail in the output log.
// A debugLevel of unity is assumed by default.
//
// See also wchunk, wevent

// Authors:
// Shourov K. Chatterji <shourov@ligo.caltech.edu>
// Leo C. Stein <lstein@ligo.mit.edu>
// Adapted for DMT streaming by:
// John G. Zweizig
//
#include "wtypes.hh"
#include "wchunk.hh"
#include "woutput.hh"
#include "matlab_fcs.hh"
#include "wstream_chan.hh"

#include "EggTimer.hh"
#include "MultiDacc.hh"

#include <iostream>
#include <iomanip>
#include <fstream>
#include <sstream>
#include <cstdlib>

//-->  The next three lines are needed if you are going to generate triggers.
//     The descriptive title in PIDTITLE should the monitor function.
#define PIDCVSHDR "$Id: wstream.cc 6710 2012-08-27 23:58:12Z john.zweizig@LIGO.ORG $"
#define PIDTITLE  "Streaming Omega search"
#include "ProcIdent.hh"

using namespace wpipe;
using namespace std;

//==================================  Search function prototype
void
wstream(const std::string& params, const std::string& cache,
	const std::string& outdir, int debugLevel);


//==================================  Main function
void
syntax(void) {
   cerr << "Command line syntax: " << endl;
   cerr << "  wstream <par-file> <frame-cache> <temp-dir> <debug-level>"
	<< endl;
}


//==================================  Main function
//
//  Command line:
//    wstream <par-file> <frame-cache> <temp-dir> <debug-level>
//
int
main(int argc, const char* argv[]) {
   if (argc < 3) {
      cerr << "insufficient parameters:" << endl;
      syntax();
      return 1;
   }
   string parFile = argv[1];
   string frameCache = argv[2];
   string outDir;
   if (argc > 3) outDir = argv[3];
   int debugLevel = 1;
   if (argc > 4) debugLevel = strtol(argv[4], 0, 0);

   try {
      wstream(parFile, frameCache, outDir, debugLevel);
   } catch (std::exception& e) {
      cerr << "Caught exception: " << e.what() << endl;
   }
}

//==================================  Event search function.
void
wstream(const std::string& params, const std::string& cache,
	const std::string& outdir, int debugLevel) {

   /////////////////////////////////////////////////////////////////////////////
   //                           start analysis timer                          //
   /////////////////////////////////////////////////////////////////////////////

   // start time of analysis
   EggTimer analysisTimer;

   /////////////////////////////////////////////////////////////////////////////
   //                         parse command line arguments                    //
   /////////////////////////////////////////////////////////////////////////////
   // apply default arguments
   string parameterFile = params;
   if (params.empty()) {
      parameterFile = "parameters.txt";
   }

   string frameCacheFile = cache;
   if (frameCacheFile.empty()) {
      frameCacheFile = "framecache.txt";
   }

   string outputDirectory = outdir;

   // create output path if not specified
   ostringstream oss;
   if (outputDirectory.empty()) {
      oss << "./segments/" << Now().getS() << "-online";
      outputDirectory = oss.str();
      oss.str("");
      // path to segment output directory
   }

   // check if specified path is absolute
   else if (outputDirectory[0] != '/') {
      // make explicitly relative to current directory if not absolute
      outputDirectory = string("./") + outputDirectory;
   }

   ////////////////////////////////////////////////////////////////////////////
   //                              write header                              //
   ////////////////////////////////////////////////////////////////////////////

   if (debugLevel >= 1) {
      cout << ">> DMT Omega streaming analysis" << endl;
      cout << "Run by " << getenv("USER") << " on " << datestr(0, 29)
	   << " at " << datestr(0, 13) << endl;
      cout << "  parameter file:          " << parameterFile << endl;
      cout << "  framecache file:         " << frameCacheFile << endl;
      cout << "  output directory:        " << outputDirectory << endl;
      cout << "  debug level:             " << debugLevel << endl;
      cout << endl;
   }

   /////////////////////////////////////////////////////////////////////////////
   //              Loop over channel blocks... read parameters                //
   /////////////////////////////////////////////////////////////////////////////

   //-----------------------------------  Look for defaults in .wstreamrc files
   double blockDuration = 0;
   double blockOverlap  = 0;
   wparameters defpars;
   defpars.set_defaults();
   string rcfile = subst_env("$HOME/.wstreamrc");
   if (exist(rcfile, "file")) {
      ifstream inrc(rcfile.c_str());
      if (inrc.is_open()) {
	 wlog(debugLevel, 1, string("Read default parameters from: ") + rcfile);
	 defpars.read_params(inrc, debugLevel);
	 if (debugLevel > 3) defpars.display(cout);
      }
   }
   
   rcfile = subst_env(".wstreamrc");
   if (exist(rcfile, "file")) {
      ifstream inrc(rcfile.c_str());
      if (inrc.is_open()) {
	 wlog(debugLevel, 1, string("Read default parameters from: ") + rcfile);
	 defpars.read_params(inrc, debugLevel);
	 if (debugLevel > 3) defpars.display(cout);
      }
   }

   //-----------------------------------  Read in parameter file
   ifstream parFile(parameterFile.c_str());
   if (!parFile.is_open()) {
      error("error opening parameter file");
   }
   wlog(debugLevel, 1, string("reading parameter file: ") + parameterFile);

   //----------------------------------- Loop over channel groups
   strchan_vect chanvect;
   size_t totalChans = 0;
   int chan_loop = 0;
   while (!chan_loop) {

      //--------------------------------  Read parameters for one channel group
      wparameters pars;
      chan_loop = pars.read_params(parFile, debugLevel);
      if (chan_loop != 0) continue;

      //--------------------------------  Save default structure.
      if (pars.groupName == "default") {
	 wparameters temp;
	 temp.copy(defpars);
	 defpars.copy(pars);
	 defpars.merge_defaults(temp);
	 cout << "================  Default parameters  ==============" << endl;
	 defpars.display(cout);
	 cout << "====================================================" << endl;
	 cout << endl;
	 continue;
      }

      //--------------------------------  Merge defaults and validate
      pars.merge_defaults(defpars);
      cout << "==========================  Channel group: " 
	   << pars.groupName << endl;
      pars.display(cout);
      cout << endl;
      pars.validate();

      //--------------------------------  Check the block duration is constant.
      if (chanvect.empty()) {
	 blockDuration = pars.blockDuration;
      } 
      else if (pars.blockDuration != blockDuration) {
	 error("Block duration must be equal for all groups");
      }

      //--------------------------------  Build the group descriptor
      wstream_chan chan(pars, debugLevel);

      //--------------------------------  Remember block overlap
      double ovlp_i = chan.minimumOverlap();
      if (chanvect.empty() || blockOverlap < ovlp_i) {
	 blockOverlap = ovlp_i;
	 if (fmod(blockOverlap, 2.0) > 0) {
	    blockOverlap += 2.0 - fmod(blockOverlap, 2.0);
	 }
      }

      // report segment information
      if (debugLevel >= 1) {
	 cout << "  block duration:      " << blockDuration << " seconds"
	      << endl;
	 cout << "  group overlap:       " << ovlp_i << " seconds"<< endl;
      }

      //--------------------------------  Group statistics
      cout << "END OF GROUP: " << pars.groupName << " total channels: "
	   << pars.numberOfChannels() << endl;
      cout << endl;
      totalChans += pars.numberOfChannels();

      //--------------------------------  Add to list
      chanvect.push_back(chan);
   }
   if (chanvect.empty()) {
      error("No channel groups specified in configuration file.");
   }

   // report segment information
   if (debugLevel >= 1) {
      cout << "  block duration:      " << blockDuration << " seconds" << endl;
      cout << "  block overlap:       " << blockOverlap  << " seconds" << endl;
      cout << "  Number of groups:    " << chanvect.size() << endl;
      cout << "  Total channels:      " << totalChans << endl;
   }

   /////////////////////////////////////////////////////////////////////////////
   //                           create results directory                      //
   /////////////////////////////////////////////////////////////////////////////

   // name of livetime file
   string livetimeFile = outputDirectory + "/livetime.txt";

   // check if (output directory exists
   if (!exist(outputDirectory,"dir")) {
      wlog(debugLevel, 1, "creating output directory");

      // create output directory
      string shellcmd = "mkdir -p ";
      shellcmd += outputDirectory;
      int sysrc = system(shellcmd.c_str());
      if (sysrc == -1) {
	 cerr << "failed to fork command shell!" << endl;
      } else if (sysrc != 0) {
	 cerr << "failed to execute shell command: " << shellcmd << endl;
      }
   }

   ////////////////////////////////////////////////////////////////////////////
   //                              output file paths                         //
   ////////////////////////////////////////////////////////////////////////////

   // time string
   Time tStart(Now());
   ostringstream fileTimeString;
   fileTimeString <<  tStart.getS() << "-" << blockDuration;

   // generate output file path
   str_vect dummy_types;
   woutput outputFiles;

   ////////////////////////////////////////////////////////////////////////////
   //                     Set up input data                                  //
   ////////////////////////////////////////////////////////////////////////////
   wlog(debugLevel, 1, "Set up input stream");

   MultiDacc frameSource(frameCacheFile.c_str());
   if (!frameSource.getNDacc()) {
      throw runtime_error("No frame sources specified");
   }
   // frameSource.setDebug(debugLevel);
   size_t nGroup = chanvect.size();
   for (size_t i=0; i<nGroup; i++) {
      chanvect[i].addChannels(frameSource);
      chanvect[i].addOutputFiles(outputFiles, fileTimeString.str(), 
				 outputDirectory);
   }
   dummy_types.push_back("*");
   outputFiles.addtype(dummy_types, tStart, blockDuration, outputDirectory, 
		       "txt", "livetime", "");

   outputFiles.display(cout);

   ////////////////////////////////////////////////////////////////////////////
   //                              begin block loop                          //
   ////////////////////////////////////////////////////////////////////////////

   //----------------------------------  Initialize block counts.
   size_t blockNumber = 0;
   size_t errorCount  = 0;

   //  Get a boundary on an even number of nanoseconds (for time specification)
   //double boundary = gcd(fSample, 1000000000);

   // initialize coordinate variable
   dble_vect coordinate;
   // begin loop over blocks
   wchunk block;
   while (!block.term()) {
 
      // start time of block analysis
      EggTimer blockTimer;

      if (debugLevel >= 1) {
	 cout << "analyzing block " << blockNumber+1 << " duration: "
	      << blockDuration << " overlap: " << blockOverlap << endl; 
      }

      /////////////////////////////////////////////////////////////////////////
      //                            block analysis                           //
      /////////////////////////////////////////////////////////////////////////
      str_vect channelNames;
      try {
	 int rc = block.process(blockDuration, blockOverlap, chanvect,
				frameSource, outputFiles, debugLevel, 
				channelNames);
	 if (rc == -1) {
	    cerr << "Break in data at gps=" 
		 << frameSource.getCurrentTime().getS() << endl;
	    continue;
	 }
	 else if (rc) {
	    cerr << "Processing terminated with rc = " << rc << endl;
	    break;
	 }
      }

      //--------------------------------  Count errors
      catch (std::exception& e) {

	 if (debugLevel >= 1) {
	    cerr << "Exception: " << e.what() << endl;
	 }

	 errorCount += 1;
	 continue;
      }

      //--------------------------------  Report block processing time
      double blockClockTime = blockTimer.elapsed();
      if (debugLevel >= 1) {
	 cout << "  block complete" << endl;
	 cout << "    elapsed time:          " << blockClockTime << " seconds"
	      << endl;

      }

      blockNumber +=1;
   } // end of loop over blocks

   //-----------------------------------  Write footer
   if (debugLevel >= 1) {
      cout << "segment complete" << endl;
      cout << "  elapsed time:            " << analysisTimer.elapsed() 
	   << " seconds" << endl;
   }

   //-----------------------------------  All done
   if (errorCount) error("analysis had errors.");
}
