/* -*- mode: c++; c-basic-offset: 3; -*- */
#ifndef WPIPE_WRESAMPLER_HH
#define WPIPE_WRESAMPLER_HH

// WRESAMPLE Resample time series data at a common sample frequency
//
// WRESAMPLE resamples the given time series data at the specified
// common sample frequency.  The input time series data must have the
// same length in time and the ratio of input sample frequencies must
// be expressable as a ratio of integers.
//
// usage:
//
//   resampledData = wresample(data, sampleFrequencies, sampleFrequency);
//
//   data                 cell array of input time series data
//   sampleFrequencies    vector of sample frequencies of input data [Hz]
//   sampleFrequency      desired sample frequency of resampled data [Hz]
//
//   resampledData        cell array of resampled time series data
//
// Note that any decrease in the sample frequency of a channel requires
// the application of a low pass filter to prevent aliasing at the new
// Nyquist frequency.  This filtering is performed using a zero phase
// filter to avoid introducing unwanted time delays into the resampled
// data stream.  However, filter transients are still expected at the
// beginning and end of the resample time series.  In addition, the
// resampled data stream has limited high frequency content since the
// anti-alias filter has a cutoff frequency of approximately 85 percent
// of the new Nyquist frequency.  It is up to the user to account for
// these effects.
//
// WRESAMPLE uses the Matlab RESAMPLE function in the Signal Processing
// Toolbox.
//
// See also RESAMPLE, WCONDITION, and WSEARCH.

// Shourov K. Chatterji <shourov@ligo.mit.edu>

// $Id: wresample.m 2753 2010-02-26 21:33:24Z jrollins $
#include "wtypes.hh"
#include "autopipe.hh"
#include <map>

class Pipe;

namespace wpipe {

   /**  Structure to resample a single series.
    */
   struct single_resample {
      single_resample(void);
      single_resample(long from, long to);
      ~single_resample(void);
 
      /**  Design a filter for the specified initial and final frequencies.
       *  The filter is a zero-phase FIR filter designed using the firls
       *  method. The filter is implemented using the FIRdft class to 
       *  optimize the speed.
       *  \brief Design an antialiasing filter.
       *  \param from Initial sample rate.
       *  \param to   Output sample rate.
       *  \return A pointer to a FIRdft instance.
       */
      Pipe* filter(long from, long to);

      /**  Initialize a single resampler.
       */
      void init(long from, long to);

      /**  Resample a single time series to the specified sample rate. THe 
        *  data are added to the input history series and the results are 
	*  place in _output.
        *  \brief Resample a a time series.
	*  \param ts %Time series to be resampled.
	*  \param fSample New sample rate.
	*/
      void resample(const TSeries& ts, double fSample);

      void reset(void);

      auto_pipe _filter;
      TSeries   _outData;

      /**  The rate pair structure identifies the input and output rates 
        *  for a filter in the map.
	*/
      struct rate_pair {
	 /**  Rate pair constructor.
	   *  \brief Constructor.
	   *  \param from Initial sample frequency.
	   *  \param to   Output sample frequency.
	   */
	 rate_pair(long from, long to);

	 /**  Compare rate pairs for map ordering.
	   *  \brief Rate comparison.
	   *  \param x Rates to be compared
	   *  \return True if this instance is less.
	   */
	 bool operator<(const rate_pair& x) const;

	 long _from; ///<  Initial sample rate
	 long _to;   ///< output sample rate.
      };

      /// Filter map type definition
      typedef std::map<rate_pair, Pipe*> filt_map;

      /// Filter map value type definition.
      typedef filt_map::value_type filt_node;

      ///  Filter map iterator type definition.
      typedef filt_map::iterator  filt_iter;

      ///  Use count for the static filter map
      static size_t use_count;

      ///  Static filter map.
      static filt_map rate_map;
   };

   /**  Data resampling class. A vector of input series is resampled to a
     *  (single) specified frequency and the result is returned as a new 
     *  vector of resampled time series.
     *
     *  The antialiasing is implemented with a zero-phase FIR filter designed
     *  by the firls method. All designed filters are stored in a static filter 
     *  map indexed by integer input and output sample rates.
     *
     *  \note This class could be mutithreaded by reorganizing the channel 
     *  list as a single vector of thread-based single-channel class instances.
     *
     *  \brief Resampling class.
     *  \author John G. Zweizig based on code by Shourov Chatterji.
     */
   class resampler {
   public:
      /**  Construct a resampler class and increment the use count for 
        *  the common filter map.
        *  \brief Default constructor
        */
      resampler(void);

      /**  Destroy the resampler class and decrement the use count for 
        *  the common filter map. Empty the map if the use count is zero.
        *  \brief Default constructor
        */
      ~resampler(void);
 
      /**  Reset the filter pointer and output data vectors.
        *  \brief reset the channel list.
        */
      void reset(void);

      /**  Run a fake resample to fill the history with zeros and to set 
        *  the correct time. This function builds a time series vector, each 
	*  element of which contains 1 second of data that ends at the start
	*  time of the argument time series. I then resamples the fake data
	*  with the resample function.
        */
      void prime(const tser_vect& data, double fSample);

      /**  Run a fake resample to fill the history with zeros and to set 
        *  the correct time. This function builds a time series vector, each 
	*  element of which contains 1 second of data that ends at the start
	*  time of the argument time series. I then resamples the fake data
	*  with the resample function.
        */
      void prime_from_data(tser_vect& data, Interval ptime, double fSample);

      /**  Resample all time series in the input data vector to the specified
        *  sample rate.
        *  \brief Resample a list of time series.
        *  \param data    Input vector of time series.
        *  \param fSample Output sample frequency.
        *  \return Vector of resampled time series.
        */
      tser_vect wresample(const tser_vect& data, double fSample);

      /**  Resample selected time series in the input data vector to the 
        *  specified sample rate.
        *  \brief Resample a list of time series.
        *  \param data    Input vector of time series.
        *  \param fSample Output sample frequency.
        *  \param select  Bool vector selecting series to resample
        *  \return Vector of resampled time series.
        */
      tser_vect wresample(const tser_vect& data, double fSample, 
			  const bool_vect& select);
   private:
      std::vector<single_resample> pipe_vect;
   };

   //====================================  Inline methods.
   inline 
   single_resample::rate_pair::rate_pair(long from, long to) 
      : _from(from), _to(to) 
   {}

   inline bool 
   single_resample::rate_pair::operator<(const rate_pair& x) const {
      return _from < x._from || (_from == x._from && _to < x._to);
   }

} // namespace wpipe
#endif // !defined(WPIPE_WRESAMPLER_HH)
