#ifndef WPIPE_MATLAB_FCS
#define WPIPE_MATLAB_FCS

#include <string>
#include <cmath>
#include <iosfwd>
#include "wtypes.hh"

namespace wpipe {

  /**  Convert boolean to "true" or "false"
    *  \brief Convert bool to string (true/false).
    *  \param test Boolean value.
    *  \return Converted string.
    */
  const char* boolstr_tf(bool test);

  /**  Convert boolean to "yes" or "no"
    *  \brief Convert bool to string (yes/no).
    *  \param test Boolean value.
    *  \return Converted string.
    */
  const char* boolstr_yn(bool test);

  /**  Return a local date/time string.
    *  \brief Date/time string.
    *  \param t   Unix time code. 
    *  \param len Output string length
    *  \return Date/time string.
    */
  std::string datestr(unsigned long t, int len);

  /**  Remove leading and trailing blanks from a string.
    *  \brief Remove leading/trailing blanks.
    *  \param str input string.
    *  \return String with blanks removed.
    */
  std::string deblank(const std::string& str);

  /**  Display a vector of doubles.
    *  \brief Display a double vector
    *  \param dv vector of doubles
    *  \param out output stream.
    *  \return Output stream reference.
    */
  std::ostream& display(const dble_vect& dv, std::ostream& out);

  /**  Display a vector of strings.
    *  \brief Display a string vector
    *  \param dv vector of strings
    *  \param out output stream.
    *  \return Output stream reference.
    */
  std::ostream& display(const str_vect& dv, std::ostream& out);
  
  /**  Throw a runtime error exception.
    *  \brief Runtime error.
    *  \param err Error message text.
    */
  void error(const std::string& err);

  /**  Parse a string containing one or more numeric values. The values are 
    *  returned as a vector of double floats. Infinite values specified as
    *  "Inf" or "inf" are stored as the run-time result of (1/0).
    *  \brief Parse a numeric string.
    *  \param str Input string
    *  \return Vector of double floats.
    */
  dble_vect eval(const std::string& str);

  /**  Parse matlab string array text into a vector of strings.
    *  \brief Parse array of strings.
    *  \param str Array source string
    *  \return Vector of strings.
    */
  str_vect eval_str(const std::string& str);

  /**  Test whether a file of the specified type exists with the specified path.
    *  \brief Test file existance
    *  \param path File path to be tested.
    *  \param type File type to be tested.
    *  \return True if file of requested type exists.
    */
  bool exist(const std::string& path, const std::string& type);

  /**  Find the greatest common denominator of two integers.
    *  \brief Greatest common denominator
    *  \param a First integer
    *  \param b Second integer.
    *  \return Greatest common denominator
    */
  long gcd(long a, long b);

  /**  Find the power of two that is greater or equal to the specified number.
    *  \brief Least power of two greater than x
    *  \param x Number to be tested.
    *  \return Exponent of two such that \c 2^N\>=x
    */
  long nextpow2(double x);
  
  /**  Evaluate a polynomial of order N.
    *  \brief Evaluate a polynomial.
    *  \param coef List of polynomial coefficients
    *  \param    N Number of coefficients (order + 1)
    *  \param    x Point at which the polynomial is to be evaluated. 
    *  \return Polynomial value at x.
    */
  double polyval(const double* coef, int N, double x);

  /**  Round a double to the nearest integer.
    *  \brief Round a double precieion float.
    *  \param x Double precision value.
    *  \return Rounded value.
    */
  double round(double x);
  
  /**  Replace all occurrances of sub-string \a from in \a str by sub-string
    *  \a to and return the resulting string. Replaced values are not tested 
    *  for replacement.
    *  \brief replace all occurrances of a sub-string.
    *  \param str  String to be modified.
    *  \param from Substring to be replaced
    *  \param to   Replacement string.
    *  \return Modified string.
    */
  std::string strrep(const std::string& str,
		     const std::string& from,
		     const std::string& to);

  /**  Substitute environment variables indicated by \c \$\<name> in a string
    *  by the variable contents.
    *  \brief Substitute enviraonment variables
    *  \param str String containing zero or more environment variable
    *             references and other text.
    *  \return string with environment variable refereces substituted.
    */
  std::string subst_env(const std::string& str);
 
  /**  Split the input string at each delimiter character in the input list.
    *  \brief Split a string at delimeter positions. The delimiter character
    *  is not included in the output string.
    *  \param str Initial string.
    *  \param delim String of delimiter characters.
    *  \return Vector of sub-strings
    */
  str_vect strsplit(const std::string& str, const std::string& delim=" ");

  /**  Convert a string to lower case.
    *  \brief Convert to lower case.
    *  \param str Initial string.
    *  \return Lower case string.
    */
  std::string tolower(const std::string& in);

  /**  Convert a string to upper case.
    *  \brief Convert to upper case.
    *  \param str Initial string.
    *  \return Upper case string.
    */
  std::string toupper(const std::string& in);

  /**  Remove leading and trailing quotes ("s) or apostrophes ('s). No 
    *  matching betweemn existance or type of leading and trailing quotes 
    *  is performed.
    *  \brief unquote a string
    *  \param in Initial string
    *  \return Unquoted string.
    */
  std::string unquote(const std::string& in);

  /**  Write a warning (status) message to the log file if the specified 
    *  level is greater than the specified threshold level.
    *  \brief Conditionally log a status message.
    *  \param thresh Threshold levvel.
    *  \param level  Message level.
    *  \param txt    Message text.
    */
  void wlog(int thresh, int level, const std::string& txt);

} // namespace wpipe

namespace matlab {
  std::string system(const std::string& par);
}


//======================================  Inline functions
inline const char*
wpipe::boolstr_tf(bool test) {
  return test ? "true" : "false";
}

inline const char*
wpipe::boolstr_yn(bool test) {
  return test ? "yes" : "no";
}

inline double
wpipe::round(double x) {
  return std::floor(x + 0.5);
}

#endif // !defined(WPIPE_MATLAB_FCS)
