#include "matlab_fcs.hh"
#include "scandir.hh"
#include "Time.hh"
#include <stdexcept>
#include <cstdlib>
#include <iostream>
#include <unistd.h>

using namespace std;

static const string env_char_list =
  "ABCDEFGHIJKLMNOPQRSTUVWXYZ_abcdefghijklmnopqrstuvwxyz0123456789";

namespace wpipe {

  //====================================  Skip pointer past spaces and tabs.
  inline const char*
  skip_spaces(const char* p) {
    while (*p == ' ' || *p == '\t') p++;
    return p;
  }

  //====================================  Convert date to string
  std::string 
  datestr(unsigned long t, int typ) {
    Time t0 = !t ? Now() : fromUTC(t);
    const char* fmt=0;
    switch (typ) {
    case 0: //            'dd-mmm-yyyy HH:MM:SS'   01-Mar-2000 15:45:17 
      fmt = "%02d-%3M-%Y %02H:%02N:%02S";
      break;
    case 1: //            'dd-mmm-yyyy'            01-Mar-2000  
      fmt = "%02d-%3M-%Y %02H:%02N:%02S";
      break;
    case 2: //            'mm/dd/yy'               03/01/00     
      fmt = "%02m/%02d/%02y";
      break;
    case 3: //            'mmm'                    Mar          
      fmt = "%3M";
      break;
    case 4: //            'm'                      M            
      fmt = "%1M";
      break;
    case 5: //            'mm'                     03
      fmt = "%02m";
      break;
    case 6: //            'mm/dd'                  03/01        
      fmt = "%02m/%02d";
      break;
    case 7: //            'dd'                     01            
      fmt = "%02d";
      break;
    case 8: //            'ddd'                    Wed          
      fmt = "%3W";
      break;
    case 9: //            'd'                      W            
      fmt = "%1W";
      break;
    case 10: //            'yyyy'                   2000         
      fmt = "%Y";
      break;
    case 11: //            'yy'                     00           
      fmt = "%02y";
      break;
    case 12: //            'mmmyy'                  Mar00        
      fmt = "%3M%02d";
      break;
    case 13: //            'HH:MM:SS'               15:45:17     
    case 14: //            'HH:MM:SS PM'             3:45:17 PM  
      fmt = "%02H:%02N:%02S";
      break;
    case 15: //            'HH:MM'                  15:45        
    case 16: //            'HH:MM PM'                3:45 PM     
      fmt = "%02H:%02N";
      break;
    //----------------------------------  replace Qx with month
    case 17: //            'QQ-YY'                  Q1-96        
    case 18: //            'QQ'                     Q1
      fmt = "%3M%02d";
      break;
    case 19: //            'dd/mm'                  01/03        
      fmt = "%02d/%02m";
      break;
    case 20: //            'dd/mm/yy'               01/03/00     
      fmt = "%02d/%02m/%02y";
      break;
    case 21: //            'mmm.dd,yyyy HH:MM:SS'   Mar.01,2000 15:45:17 
      fmt = "%3M.%02d,%Y %02H:%02N:%02S";
      break;
    case 22: //            'mmm.dd,yyyy'            Mar.01,2000  
      fmt = "%3M.%02d,%Y";
      break;
    case 23: //            'mm/dd/yyyy'             03/01/2000 
      fmt = "%02m/%02d/%Y";
      break;
    case 24: //            'dd/mm/yyyy'             01/03/2000 
      fmt = "%02d/%02m/%Y";
      break;
    case 25: //            'yy/mm/dd'               00/03/01 
      fmt = "%02y/%02m/%02d";
      break;
    case 26: //            'yyyy/mm/dd'             2000/03/01 
      fmt = "%Y/%02m/%02d";
      break;
    case 27: //            'QQ-YYYY'                Q1-1996        
      fmt = "%02m-%Y";
      break;
    case 28: //            'mmmyyyy'                Mar2000        
      fmt = "%3M%Y";
      break;
    case 29: //(ISO 8601)  'yyyy-mm-dd'             2000-03-01
      fmt = "%Y-%02m-%02d";
      break;
    case 30: //(ISO 8601)  'yyyymmddTHHMMSS'        20000301T154517 
      fmt = "%Y%02m%02dT%02H%02N%02S";
      break;
    case 31: //            'yyyy-mm-dd HH:MM:SS'    2000-03-01 15:45:17 
      fmt = "%Y-%02m-%02d %02H:%02N:%02S";
      break;
    default:
      fmt = "%02d-%3M-%Y %02H:%02N:%02S";
      break;
    }
    return TimeString(t0, fmt);
  }

  //====================================  Remove blanks from start/end of string
  std::string
  deblank(const std::string& str) {
    std::string::size_type ifrst = str.find_first_not_of(" \t\n");
    if (ifrst == std::string::npos) return "";
    std::string::size_type ilast = str.find_last_not_of(" \t\n");
    if (ilast != std::string::npos) ilast++;
    return str.substr(ifrst, ilast-ifrst);
  }

  //====================================  Diaplay a numeric vector parameter.
  std::ostream& 
  display(const dble_vect& dv, std::ostream& out) {
    size_t N = dv.size();
    out << "[";
    for (size_t i=0; i<N; i++) {
      if (i) out << ", ";
      out << dv[i];
    }
    return out << "]";
  }

  //====================================  Diaplay a numeric vector parameter.
  std::ostream& 
  display(const str_vect& sv, std::ostream& out) {
    size_t N = sv.size();
    out << "{";
    for (size_t i=0; i<N; i++) {
      if (i) out << ", ";
      out << "\"" << sv[i] << "\"";
    }
    return out << "}";
  }

  //====================================  Throw a run-time error exception.
  void
  error(const std::string& err) {
    throw std::runtime_error(err);
  }

  //====================================  Inline function to test for "Inf"
  inline int
  test_inf(const char* p) {
    int rc = 1;
    if (*p == '-') {
      rc = 2;
      p++;
    }
    if (*p != 'i' && *p != 'I') return 0;
    if (*(++p) != 'n') return 0;
    if (*(++p) != 'f') rc = 0;
    return rc;
  }

  //====================================  Parse a list of numbers (allow inf).
  dble_vect
  eval(const std::string& str) {
    dble_vect r;
    const char* ptr = str.c_str();
    if (*ptr != '[') {
      char* pLast = 0;
      r.push_back(strtod(ptr, &pLast));
      if (*pLast) error(string("Unparseable numeric vector: ")+str);
    } else {
      ptr++;
      while (*ptr && *ptr != ']') {
	ptr = skip_spaces(ptr);
	const char* svptr = ptr;
	char* eptr = const_cast<char*>(ptr);
	switch (test_inf(ptr)) {
	case 0:
	  r.push_back(strtod(ptr, &eptr));
	  break;
	case 1:
	  r.push_back(1.0/0.0);
	  eptr += 3;
	  break;
	case 2:
	  r.push_back(-1.0/0.0);
	  eptr += 4;
	  break;
	}
	ptr = skip_spaces(eptr);
	if (*ptr == ',' || *ptr == ';') ptr = skip_spaces(++ptr);
	if (ptr == svptr) error(string("Unparseable numeric vector: ")+str);
      }
    }
    return r;
  }

  //====================================  Parse a list of strings.
  str_vect
  eval_str(const std::string& str) {
    str_vect r;
    string s;
    const char* ptr = str.c_str();
    bool instr = false, plaintext = true;
    int inbrace=0;
    int strdelim = 0;
    while (*ptr) {
      switch (*ptr) {
      //--------------------------------  String delimiters
      case '\'':
      case '"':
	//------------------------------  Start of string
	if (!instr) {
	  instr = true;
	  strdelim = *ptr;
	  plaintext = false;
	}
	//------------------------------  end of string
	else if (*ptr == strdelim) {
	  r.push_back(s);
	  s.clear();
	  strdelim = 0;
	  instr = false;
	}
	//------------------------------  wrong type delimiter in string
	else {
	  s += *ptr;	  
	}
	break;
	
      //--------------------------------  inter-string delimiters
      case '{':
      case '}':
      case ',':
      case ' ':
	if (!instr) {
	  if (*ptr == '{') inbrace++;
	  else if (*ptr == '}') inbrace--;
	  plaintext = false;
	  break;
	}
      default:
	if (instr || plaintext) {
	  s += *ptr;
	} else {
	  error("Unexpected character");
	}
      }
      ptr++;
    }
    if (plaintext && !s.empty()) {
      r.push_back(s);
    } else if (instr) {
      error("Unterminated string");
    } else if (inbrace) {
      error("Unterminated braces");
    }
    //cout << "eval_str output = [";
    //for (size_t i=0; i<r.size(); i++) cout << " " << r[i];
    //cout << " ]" << endl;
    return r;
  }

  //====================================  Test the exiistence of a file.
  bool
  exist(const std::string& path, const std::string& type) {
    bool rc = false;
    if (type == "file") {
      rc = (access(path.c_str(), F_OK) == 0);
    }
    else if (type == "dir") {
      rc = scandir::test_directory(path);
    }
    else {
      rc = false;
    }
    return rc;
  }

  //====================================  Greatest common divisor
  //                                      Use Euler's algorithm
  long
  gcd(long a, long b) {
    if (!a) return b;
    if (b > a) {
      long t = b;
      b = a;
      a = t;
    }
    while (b) {
      long t = b;
      b = a % b;
      a = t;
    }
    return a;
  }

  //====================================  Exponent of 2 for 2^N >= x
  long
  nextpow2(double x) {
    double powval(1.0);
    long ipow(0);
    while (powval < x) {
      powval *= 2.0;
      ipow++;
    }
    return ipow;
  }

  //====================================  Polynomial value.
  double 
  polyval(const double* coef, int N, double x) {
    double sum = coef[0];
    for (int i=1; i<N; i++) {
      sum *= x;
      sum += coef[i];
    }
    return sum;
  }

  //====================================  Replace a string.
  std::string 
  strrep(const std::string& str, const std::string& from, const std::string& to) {
    std::string temp(str);
    string::size_type frlen = from.size();
    if (!frlen) return temp;
    while (str.find(from) != string::npos) {
      temp.replace(str.find(from), frlen, to);
    }
    return temp;
  }

  //====================================  Split a string at specified delimiters
  str_vect 
  strsplit(const std::string& str, const std::string& delim) {
    str_vect ret;
    string temp = str;
    while (!temp.empty()) {
      string::size_type inx = temp.find_first_of(delim);
      ret.push_back(temp.substr(0,inx));
      if (inx == string::npos) temp.clear();
      else                     temp.erase(0,inx+1);
    }
    return ret;
  }

  //====================================  Substitute environment variables
  std::string
  subst_env(const std::string& str) {
    string path = str;
    size_t ienv = path.find("$");
    while (ienv != string::npos) {
      size_t lenv = path.find_first_not_of(env_char_list, ienv+1);
      if (lenv == string::npos) lenv = path.size() - ienv;
      else                      lenv -= ienv;
      const char* cenv = getenv(path.substr(ienv+1, lenv-1).c_str());
      if (cenv) path.replace(ienv, lenv, cenv);
      else      path.erase(ienv, lenv);
      ienv = path.find("$");
    }
    return path;
  }

  //====================================  Convert string to lower case
  std::string 
  tolower(const std::string& str) {
    std::string temp(str);
    string::size_type strlen = temp.size();
    if (!strlen) return temp;

    for (string::size_type i=0; i<strlen; i++) {
      temp[i] = ::tolower(temp[i]);
    }
    return temp;
  }

  //====================================  Convert string to upper case
  std::string 
  toupper(const std::string& str) {
    std::string temp(str);
    string::size_type strlen = temp.size();
    if (!strlen) return temp;

    for (string::size_type i=0; i<strlen; i++) {
      temp[i] = ::toupper(temp[i]);
    }
    return temp;
  }

  //====================================  Remove leading and/or trailing quotes
  std::string 
  unquote(const std::string& str) {
    if (str.empty()) return str;
    string::size_type front = 0;
    string::size_type last  = str.size();
    if (str[front]  == '\'' || str[front]  == '"') front++;
    if (str[last-1] == '\'' || str[last-1] == '"') last--;
    if (last <= front) return "";
    return str.substr(front, last-front);
  }

  //====================================  Conditionally log a message.
  void 
  wlog(int thresh, int level, const std::string& txt) {
    if (thresh >= level) cout << txt << endl;
  }
}

#include <sys/wait.h>

namespace matlab {
  std::string
  system(const std::string& cmd) {
    int fd[2];
    if (pipe(fd) < 0) {
      perror("Error creating pipe in system");
      return "";
    }
    pid_t pid = fork();
    if (pid < 0) {
      perror("error forking process in system");
      close(fd[0]);
      close(fd[1]);
    } else if (!pid) {
      if (dup2(fd[1], 1) < 0) perror("dup2 error in system");
      int rc = execl("/bin/sh", "/bin/sh", "-c", cmd.c_str());
      close(fd[1]);
    } else {
      char buf[257];
      int status;
      waitpid(pid, &status, 0);
      string result;
      int rc = read(fd[0], buf, 256);
      result += string(buf, rc);
      while (rc == 256) {
	rc = read(fd[0], buf, 256);
	result += string(buf, rc);
      }
      if (rc < 0) perror("error reading pipe in system");
      close(fd[0]);
      return result;
    }
    return ""; // error return
  }
}
