////////////////////////////////////////////////////////////////////////////////
//                                                                            //
//                                OmegaMon.cc                                 //
//                                                                            //
//              DMT interface to Omega Online trigger properties              //
//                                                                            //
//                      Zhang Fan <zf19830622@gmail.com>                      //
//              Shourov K. Chatterji <shourov@ligo.caltech.edu>               //
//                                                                            //
//                                2009-Jan-19                                 //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////

// $Id: OmegaMon.cc 6444 2011-06-07 21:33:13Z keith.thorne@LIGO.ORG $

// include header files
#include "OmegaMon.hh"
#include "Dacc.hh"
#include "ParseLine.hh"
#include "Segment.hh"
#include <iostream>
#include <iomanip>
#include <fstream>
#include <string>
#include <stdlib.h>
#include <cstdio>
#include <cmath>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <pwd.h>

// processor identifier for triggers
#define PIDCVSHDR "$Header: https://redoubt.ligo-wa.caltech.edu/svn/gds/trunk/Monitors/OmegaMon/OmegaMon.cc 6444 2011-06-07 21:33:13Z keith.thorne@LIGO.ORG $"
#define PIDTITLE  "DMT interface to Omega Online trigger properties"
#include "ProcIdent.hh"

// generate main routine
EXECDAT(OmegaMon)

//======================================  Skeleton object constructor.
  OmegaMon::OmegaMon(int argc, const char *argv[]) 
    : DatEnv(argc, argv), maxFrame(999999), mStep(2.0), mHistory(43200)
{
  //----------------------------------  Look for arguments
  bool dvIndex = false;

  bool syntax  = false;
  for (int i=1 ; i<argc ; i++) {
    std::string argi = argv[i];
    if (argi == "-channel") {
      mChannel = argv[++i];
    } else if (argi == "-dvindex") {
      dvIndex = true;
    } else if (argi == "-filter") {
      mFilter = argv[++i];
    } else if (argi == "-frame") {
      maxFrame = strtol(argv[++i], 0, 0);
    } else if (argi == "-stride") {
      mStep = strtod(argv[++i], 0);
    } else if (argi == "-settle") {
      mSettle = strtod(argv[++i], 0);
    } else if (isDatEnvArg(argv[i])) {
      i++;
    } else {
      syntax = true;
      std::cerr << "Unrecognized argument: " << argi << std::endl;
    }
  }

  //----------------------------------  Bail out if command line not correct
  if (syntax) {
    std::cerr << "Command syntax:" << std::endl;
    std::cerr << argv[0] 
	 << " [-channel <nMax>] [-frames <nMax>] [-stride <time>] \\" 
	 << std::endl
	 << "      [-filter <filter>] [-settle <s-time>] [-dvIndex] \\"
	 << std::endl
	 << "      [<DatEnv-args>]" 
	 << std::endl;
    finish();
    return;
  }

  //----------------------------------  Make sure there's a valid channel
  if (mChannel.empty()) mChannel = "H0:PEM-LVEA_SEISX";

  //----------------------------------  set the data accessor
  getDacc().setStride(mStep);       // in before calling ProcessData
  getDacc().addChannel(mChannel.c_str());

  //----------------------------------  Specify DMT Viewer channel names
  //--- replaced cuserid() with getpwuid - K. Thorne June, 2011
  struct passwd *pws;
  pws = getpwuid(geteuid());
  setServerName(pws->pw_name);

  mSigmaName = mChannel + "_sigma";
  serveData(mSigmaName.c_str(), &mHistory, 0);

  //----------------------------------  Set up the trender
  mTrend.setName("OmegaMon");    //  Set trend frame name
  mTrend.setFrameCount(1);          //  Set frames per file
  mTrend.setType(Trend::kMinute);   //  Set trend type (minute trends)
  mTrend.setAutoUpdate(false);      //  Avoid automatic screwup.
  mTrend.addChannel(mSigmaName.c_str());

  //----------------------------------  Optional dataviewer index
  if (dvIndex) {
    std::string filename;
    const char* dmtout = getenv("DMTOUTPUT");
    if (dmtout) {
      filename = dmtout;
      filename += "/";
    }
    filename += "channel.cfg";
    mTrend.writeIndex(filename.c_str());
  }
}

//======================================  Skeleton object destructor.
OmegaMon::~OmegaMon() 
{
  std::cout << "OmegaMon is finished" << std::endl;
}

//======================================  Frame processing function.
void
OmegaMon::ProcessData(void) {
  //----------------------------------  Get pointers to the current data.
  const TSeries* ts = getDacc().refData(mChannel.c_str());
  if (!ts || ts->isEmpty()) {
    std::cout << "Channel: " << mChannel << " was not found." << std::endl;
    return;
  }

  //----------------------------------  Calculate channel sigma.
  float Sig = sqrt( *ts * *ts/ts->getNSample() - pow(ts->getAverage(), 2) );

  //----------------------------------  Trend the data points
  Time t0 = ts->getStartTime();
  mTrend.trendData(mSigmaName.c_str(), t0, Sig);
  mTrend.Update();

  //----------------------------------  set up data for display.
  mHistory.fixedAppend(t0, ts->getInterval(), &Sig);
}


//======================================  Handle Messages
void
OmegaMon::Attention(void) {
  MonServer::Attention();
}
					
